package com.bizunited.platform.core.controller;

import com.alibaba.fastjson.JSON;
import com.bizunited.platform.core.controller.model.ResponseCode;
import com.bizunited.platform.core.controller.model.ResponseModel;
import com.bizunited.platform.core.entity.DictCategoryEntity;
import com.bizunited.platform.core.entity.DictEntity;
import com.bizunited.platform.core.entity.DictItemEntity;
import com.bizunited.platform.core.service.DictCategoryService;
import io.swagger.annotations.ApiOperation;
import io.swagger.annotations.ApiParam;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.util.CollectionUtils;
import org.springframework.web.bind.annotation.*;

import java.util.ArrayList;
import java.util.List;
import java.util.Set;

/**
 * DictCategoryController
 *
 * @description:
 * @author: yanwe
 * @date: 12/Jun/2019 09:56
 */
@RestController
@RequestMapping("/v1/nebula/dictCategories")
public class DictCategoryController extends BaseController {

  private static final Logger LOGGER = LoggerFactory.getLogger(DictCategoryController.class);

  @Autowired private DictCategoryService dictCategoryService;

  @ApiOperation(value = "创建字典分类")
  @RequestMapping(value = "", method = RequestMethod.POST)
  public ResponseModel create(
      @ApiParam(name = "字典分类信息") @RequestBody DictCategoryEntity dictCategoryEntity,
      @RequestParam(name = "parentId") @ApiParam(name = "parentId", value = "父级字典ID")
          String parentId) {
    LOGGER.debug("创建字典分类：{}", JSON.toJSONString(dictCategoryEntity));
    try {
      DictCategoryEntity entity = dictCategoryService.create(dictCategoryEntity,parentId);
      return this.buildHttpReslutW(entity, "");
    } catch (Exception e) {
      LOGGER.error(e.getMessage(), e);
      return buildHttpReslutForException(e);
    }
  }

  @ApiOperation(value = "修改字典分类")
  @RequestMapping(value = "", method = RequestMethod.PATCH)
  public ResponseModel update(
      @ApiParam(name = "字典分类信息") @RequestBody DictCategoryEntity dictCategoryEntity) {
    LOGGER.debug("修改字典分类：{}", JSON.toJSONString(dictCategoryEntity));
    try {
      DictCategoryEntity entity = dictCategoryService.update(dictCategoryEntity);
      return this.buildHttpReslutW(entity, "");
    } catch (Exception e) {
      LOGGER.error(e.getMessage(), e);
      return buildHttpReslutForException(e);
    }
  }

  @ApiOperation(value = "设置字典分类层级")
  @RequestMapping(value = "/bind", method = RequestMethod.POST)
  public ResponseModel bind(
      @RequestParam(name = "childId") @ApiParam(name = "childId", value = "子级字典ID") String childId,
      @RequestParam(name = "parentId") @ApiParam(name = "parentId", value = "父级字典ID")
          String parentId) {
    try {
      dictCategoryService.bind(childId, parentId);
      return this.buildHttpReslut();
    } catch (Exception e) {
      LOGGER.error(e.getMessage(), e);
      return buildHttpReslutForException(e);
    }
  }
  
  @ApiOperation(value = "根据数据字典业务编号code进行查询，不包括任何关联信息")
  @RequestMapping(value = "/findByCode", method = RequestMethod.GET)
  public ResponseModel findByCode(@RequestParam("code") @ApiParam(name = "code" , value = "业务编号code在全系统唯一") String code) {
    try {
      DictCategoryEntity current = dictCategoryService.findByCode(code);
      return this.buildHttpReslutW(current , new String[] {});
    } catch (Exception e) {
      LOGGER.error(e.getMessage(), e);
      return buildHttpReslutForException(e);
    }
  }
  
  @ApiOperation(value = "查询所有字典分类")
  @RequestMapping(value = "/findAll", method = RequestMethod.GET)
  public ResponseModel findAll() {
    try {
      Set<DictCategoryEntity> entities = dictCategoryService.findAll();
      // 组装过滤返回树级结构
      List<DictCategoryEntity> entitiesList = new ArrayList<>();
      entitiesList.addAll(entities);
      this.filterDictCate(entitiesList);
      ResponseModel responseModel = new ResponseModel();
      responseModel.setTimestamp(System.currentTimeMillis());
      responseModel.setData(entitiesList);
      responseModel.setSuccess(true);
      responseModel.setResponseCode(ResponseCode.E0);
      return responseModel;
    } catch (Exception e) {
      LOGGER.error(e.getMessage(), e);
      return buildHttpReslutForException(e);
    }
  }

  /**
   * 自组装返还的多层级字典分类结构,过滤掉entity中关联的非子级数据
   * @param entities
   */
  private void filterDictCate(List<DictCategoryEntity> entities) {
    if (CollectionUtils.isEmpty(entities)) {
      return;
    }
    for (DictCategoryEntity entity : entities) {
      entity.setParentCategory(null);
      entity.setCreateUser(null);
      entity.setModifyUser(null);
      Set<DictEntity> dicts = entity.getDicts();
      if (!CollectionUtils.isEmpty(dicts)) {
        for (DictEntity dict : dicts) {
          dict.setCategory(null);
          if(!CollectionUtils.isEmpty(dict.getDictItemEntities())){
            for(DictItemEntity item : dict.getDictItemEntities()){
              item.setDictEntity(null);
            }
          }
        }
      }
      Set<DictCategoryEntity> child = entity.getChildCategory();
      List<DictCategoryEntity> childList = new ArrayList<>();
      childList.addAll(child);
      if (!CollectionUtils.isEmpty(child)) {
        this.filterDictCate(childList);
      }
    }
  }
}
