package com.bizunited.platform.core.controller;

import com.bizunited.platform.core.controller.model.ResponseModel;
import com.bizunited.platform.core.entity.MigrateConfigAnalysisEntity;
import com.bizunited.platform.core.entity.MigrateImportEntity;
import com.bizunited.platform.core.service.migrate.MigrateImportService;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import io.swagger.annotations.ApiParam;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.domain.Pageable;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestMethod;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.RestController;
import org.springframework.web.multipart.MultipartFile;

import java.security.Principal;
import java.util.HashMap;
import java.util.Map;
import java.util.Set;

/**
 * MigrateImportController
 *
 * @description: 迁入管理
 * @author: yanwe
 * @date: 03/Sep/2019 14:22
 */
@Api(value = "MigrateImportController")
@RestController
@RequestMapping("/v1/nebula/migrationsImport")
public class MigrateImportController extends BaseController {

  @Autowired private MigrateImportService migrateImportService;

  /** 日志 */
  private static final Logger LOGGER = LoggerFactory.getLogger(MigrateImportController.class);

  @ApiOperation(value = "分页查询上传记录")
  @RequestMapping(value = "/findByCondition", method = RequestMethod.GET)
  public ResponseModel findByCondition(
      @ApiParam(name = "编号") @RequestParam(name = "code") String code,
      @ApiParam(name = "本次任务是否最终执行") @RequestParam(name = "executed") Boolean executed,
      Pageable pageable) {
    try {
      Map<String, Object> conditions = new HashMap<>();
      if (StringUtils.isNotBlank(code)) {
        conditions.put("code", code);
      }

      if (null != executed) {
        conditions.put("executed", executed);
      }
      if (null != pageable) {
        pageable = PageRequest.of(0, 50);
      }
      Page<MigrateImportEntity> result = migrateImportService.findByCondition(conditions, pageable);
      return this.buildHttpReslutW(result, new String[] {});
    } catch (Exception e) {
      LOGGER.error(e.getMessage(), e);
      return this.buildHttpReslutForException(e);
    }
  }

  @ApiOperation(value = "根据迁入的基础配置，查询历史分析记录")
  @RequestMapping(value = "/findConfigAnalysisByImport", method = RequestMethod.GET)
  public ResponseModel findConfigAnalysisByImport(
      @RequestParam(name = "migrateImportId") @ApiParam(name = "migrateImportId", value = "迁入文件ID")
          String migrateImportId) {
    try {
      Set<MigrateConfigAnalysisEntity> entities =
          migrateImportService.findConfigAnalysisByImport(migrateImportId);
      return this.buildHttpReslutW(entities, new String[] {});
    } catch (Exception e) {
      LOGGER.error(e.getMessage(), e);
      return this.buildHttpReslutForException(e);
    }
  }


  @ApiOperation(value = "上传迁入文件")
  @RequestMapping(value = "/upload", method = RequestMethod.POST)
  public ResponseModel upload(
      @RequestParam(name = "code") @ApiParam(name = "code", value = "迁入文件编码") String code,
      @RequestParam(name = "migrateDesc") @ApiParam(name = "migrateDesc", value = "迁入文件描述")
          String migrateDesc,
      @RequestParam(name = "file", required = true)
          @ApiParam(required = true, name = "file", value = "迁入文件对象")
          MultipartFile file,
      Principal currentUser) {
    try {
      MigrateImportEntity entity =
          migrateImportService.upload(code, migrateDesc, currentUser, file);
      return this.buildHttpReslutW(entity, new String[] {"creator", "executor"});
    } catch (Exception e) {
      LOGGER.error(e.getMessage(), e);
      return this.buildHttpReslutForException(e);
    }
  }

  @ApiOperation(value = "分析上传的基础信息，返回分析结果")
  @RequestMapping(value = "/analysisConfig", method = RequestMethod.POST)
  public ResponseModel analysisConfig(
      @RequestParam(name = "migrateImportId") @ApiParam(name = "migrateImportId", value = "迁入文件编码")
          String migrateImportId,
      Principal currentUser) {
    try {
      MigrateConfigAnalysisEntity entity =
          migrateImportService.analysisConfig(migrateImportId, currentUser);
      return this.buildHttpReslutW(entity, new String[] {});
    } catch (Exception e) {
      LOGGER.error(e.getMessage(), e);
      return this.buildHttpReslutForException(e);
    }
  }


  @ApiOperation(value = "执行迁移上传的迁移文件")
  @RequestMapping(value = "/execute", method = RequestMethod.POST)
  public ResponseModel execute(
      @RequestParam(name = "migrateImportId") @ApiParam(name = "migrateImportId", value = "迁入文件编码")
          String migrateImportId,
      Principal currentUser) {
    try {
      MigrateImportEntity entity = migrateImportService.execute(migrateImportId, currentUser);
      return this.buildHttpReslutW(entity, new String[] {});
    } catch (Exception e) {
      LOGGER.error(e.getMessage(), e);
      return this.buildHttpReslutForException(e);
    }
  }
}
