package com.bizunited.platform.core.controller;

import com.alibaba.fastjson.JSONArray;
import com.bizunited.platform.core.controller.model.ResponseModel;
import com.bizunited.platform.rbac.server.service.RoleService;
import com.bizunited.platform.rbac.server.vo.RoleVo;
import com.google.common.collect.Sets;
import com.google.common.collect.Sets.SetView;
import io.swagger.annotations.ApiOperation;
import io.swagger.annotations.ApiParam;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestMethod;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.RestController;

import java.util.List;
import java.util.Set;

@RestController
@RequestMapping("/v1/nebula/roles")
public class RoleController extends BaseController {

  @Autowired
  private RoleService roleService;
  /**
   * 配置的那些可以忽略方法级检查的，具有超级管理员性质的角色名
   */
  @Value("${rbac.ignoreMethodCheckRoles:ADMIN}")
  private String[] ignoreMethodCheckRoles;
  /**
   * 日志
   */
  private static final Logger LOGGER = LoggerFactory.getLogger(RoleController.class);

  /**
   * 添加一个角色信息
   * 
   * @param role
   * @return 创建后的角色基本信息将会被返回
   */
  @ApiOperation(value = "添加一个角色信息")
  @RequestMapping(value = "", method = RequestMethod.POST)
  public ResponseModel addRole(@RequestBody @ApiParam(value = "role" , name="新建的角色对象") RoleVo role) {
    try {
      if (role == null) {
        throw new IllegalArgumentException("role info not be null!");
      }
      RoleVo currentRole = roleService.create(role);
      return this.buildHttpReslutW(currentRole, new String[] {});
    } catch (Exception e) {
      LOGGER.error(e.getMessage(), e);
      return this.buildHttpReslutForException(e);
    }
  }

  /**
   * 修改一个指定的角色信息，注意配置在roles.deleteDeny属性的信息不能进行修改操作<br>
   * 且指定的一个角色只能修改角色的comment信息
   * 
   * @param role 指定的修改信息
   */
  @ApiOperation(value = "修改一个指定的角色信息，注意配置在roles.deleteDeny属性的信息不能进行修改操作。且指定的一个角色只能修改角色的comment信息")
  @RequestMapping(value = "", method = RequestMethod.PATCH)
  public ResponseModel updateRole(@RequestBody @ApiParam(value = "role" , name="修改的角色对象") RoleVo role) {
    try {
      RoleVo currentRole = this.roleService.update(role);
      return this.buildHttpReslutW(currentRole,new String[] {});
    } catch (Exception e) {
      LOGGER.error(e.getMessage(), e);
      return this.buildHttpReslutForException(e);
    }
  }

  /**
   * 重新启用某一个指定的角色信息
   * 
   * @param roleId
   * @return
   */
  @ApiOperation(value = "重新启用某一个指定的角色信息")
  @RequestMapping(value = "/enable/{roleId}", method = RequestMethod.PATCH)
  public ResponseModel enableRole(@PathVariable("roleId") String roleId) {
    try {
      this.roleService.enable(roleId);
      return this.buildHttpReslut();
    } catch (Exception e) {
      LOGGER.error(e.getMessage(), e);
      return this.buildHttpReslutForException(e);
    }
  }

  /**
   * 禁用某一个指定的角色信息（相当于删除）<br>
   * 只是系统中不能真正的删除某一个角色，只能是将这个角色作废掉或者恢复正常状态
   * 
   * @param roleId
   * @return
   */
  @ApiOperation(value = "禁用某一个指定的角色信息（相当于删除）<br>" + "只是系统中不能真正的删除某一个角色，只能是将这个角色作废掉或者恢复正常状态")
  @RequestMapping(value = "/disable/{roleId}", method = RequestMethod.PATCH)
  public ResponseModel disableRole(@PathVariable("roleId") String roleId) {
    try {
      this.roleService.disable(roleId);
      return this.buildHttpReslut();
    } catch (Exception e) {
      LOGGER.error(e.getMessage(), e);
      return this.buildHttpReslutForException(e);
    }
  }

  @ApiOperation(value = "按照指定的用户编号，为用户绑定角色信息")
  @RequestMapping(value = "/bindUsers", method = RequestMethod.POST)
  public ResponseModel bindUsers (
      @ApiParam(name = "userId", value = "指定的用户编号", required = true)  String userId,
      @ApiParam(name = "roleNames", value = "指定的角色名集合(角色名)，多个角色名以“,”分割", required = true) @RequestParam(name="roleNames") String roleNames) {
    try {
      roleNames = roleNames == null? "":roleNames;
      String[] currentRoleNames = StringUtils.split(roleNames , ",");
      this.roleService.bindRoles(userId, currentRoleNames);
      return this.buildHttpReslut();
    } catch (Exception e) {
      LOGGER.error(e.getMessage(), e);
      return this.buildHttpReslutForException(e);
    }
  }
  
  @ApiOperation(value = "给指定id的用户重新进行角色绑定，也就是说以前的绑定信息就失效了，按照最新给定的角色名进行角色绑定")
  @RequestMapping(value = "/reBindUsers", method = RequestMethod.POST)
  public ResponseModel reBindUsers(
      @ApiParam(name = "userId", value = "指定的用户编号", required = true) @RequestParam(name="userId") String userId,
      @ApiParam(name = "roleNames", value = "指定的信息角色名集合(角色名)，多个角色名以“,”分割", required = true) @RequestParam(name="roleNames") String roleNames) {
    try {
      roleNames = roleNames == null? "":roleNames;
      String[] currentRoleNames = StringUtils.split(roleNames , ",");
      this.roleService.reBindRoles(userId, currentRoleNames);
      return this.buildHttpReslut();
    } catch (Exception e) {
      LOGGER.error(e.getMessage(), e);
      return this.buildHttpReslutForException(e);
    }
  }

  /**
   * 该接口方法用于查询符合指定状态的角色信息，只返回角色的基本信息，没有任何的关联信息但是包括了可能存在的修改者信息
   * 
   * @param status
   * @return
   */
  @ApiOperation(value = "该接口方法用于查询符合指定状态的角色信息，只返回角色的基本信息，没有任何的关联信息但是包括了可能存在的修改者信息。")
  @RequestMapping(value = "/findByStatus", method = RequestMethod.GET)
  public ResponseModel findByStatus(Integer status) {
    try {
      List<RoleVo> roles = this.roleService.findByStatus(status);
      return this.buildHttpReslutW(roles, new String[] {});
    } catch (Exception e) {
      LOGGER.error(e.getMessage(), e);
      return this.buildHttpReslutForException(e);
    }
  }

  /**
   * 查询一个指定的角色信息，只查询这个角色的基本信息
   */
  @ApiOperation(value = "查询一个指定的角色信息，只查询这个角色的基本信息")
  @RequestMapping(value = "/find/{roleId}", method = RequestMethod.GET)
  public ResponseModel findById(@PathVariable("roleId") String roleId) {
    try {
      RoleVo currentRole = this.roleService.findById(roleId);
      return this.buildHttpReslutW(currentRole, new String[] {});
    } catch (Exception e) {
      LOGGER.error(e.getMessage(), e);
      return this.buildHttpReslutForException(e);
    }
  }

  /**
   * 获取指定用户已绑定的角色信息
   * @param userId 用户id
   * @return List<RoleEntity>
   */
  @ApiOperation(value = "获取指定用户已拥有的角色信息，包括自己绑定的，所属组织机构的，所属用户组的，甚至第三方系统的，等等",
      notes = "获取指定用户已拥有的角色信息，包括自己绑定的，所属组织机构的，所属用户组的，甚至第三方系统的，等等")
  @RequestMapping(value = "/findByUserId", method = {RequestMethod.GET})
  public ResponseModel findByUserId(@ApiParam(value = "指定的人员数据编号") @RequestParam String userId) {
    try {
      List<RoleVo> nowList = roleService.findAllByUserId(userId);
      return this.buildHttpReslutW(nowList, new String[] {});
    } catch (Exception e) {
      LOGGER.error(e.getMessage(), e);
      return this.buildHttpReslutForException(e);
    }
  }
  
  @ApiOperation(value = "该方法给定一批角色名，系统会返回这批角色名中，是否有一个或者多个角色拥有管理员性质的角色。而判定具有超级管理员性质角色的用户会被返回",
      notes = "获取指定用户已拥有的角色信息，包括自己绑定的，所属组织机构的，所属用户组的，甚至第三方系统的，等等")
  @RequestMapping(value = "/findByIgnoreRoles", method = {RequestMethod.GET})
  public ResponseModel findByIgnoreRoles(@ApiParam(value = "当前需要验证的是否具有本地超级管理员") @RequestParam(name="roleNames") String[] roleNames) {
    try{
      Validate.isTrue(roleNames != null && roleNames.length > 0 , "要进行判定的角色信息必须传入");
      SetView<String> intersections = Sets.intersection(Sets.newHashSet(ignoreMethodCheckRoles), Sets.newHashSet(roleNames));
      
      if(intersections == null || intersections.isEmpty()) {
        return this.buildHttpReslut();
      }
      // 组装成json返回
      JSONArray results = new JSONArray();
      for (String item : intersections) {
        results.add(item);
      }
      return this.buildHttpReslut(results);
    } catch(Exception e) {
      return this.buildHttpReslutForException(e);
    }
  }
  
  /**
   * 该接口方法用于查询所有角色信息，无论它们的状态如何。<br>
   * 只返回角色的基本信息，没有任何的关联信息但是包括了可能的修改者信息"
   * 
   */
  @ApiOperation(value = "该接口方法用于查询所有角色信息，无论它们的状态如何(只推荐的管理端使用该接口，在业务端不推荐)。<br>" + "只返回角色的基本信息，没有任何的关联信息但是包括了可能的修改者信息")
  @RequestMapping(value = "/findAll", method = RequestMethod.GET)
  public ResponseModel findAll() {
    try {
      List<RoleVo> roles = this.roleService.findAll();
      return this.buildHttpReslutW(roles, new String[] {});
    } catch (Exception e) {
      LOGGER.error(e.getMessage(), e);
      return this.buildHttpReslutForException(e);
    }
  }


  @ApiOperation(value = "查询多个角色信息")
  @RequestMapping(value = "/findByIds", method = RequestMethod.POST)
  public ResponseModel findByIds(@RequestBody List<String> ids) {
    try {
      Set<RoleVo> roles = this.roleService.findByIds(ids);
      return this.buildHttpReslutW(roles,new String[]{});
    } catch (Exception e) {
      LOGGER.error(e.getMessage(), e);
      return this.buildHttpReslutForException(e);
    }
  }
}
