package com.bizunited.platform.core.controller;

import com.bizunited.platform.core.controller.model.ResponseModel;
import com.bizunited.platform.rbac.server.service.UserService;
import com.bizunited.platform.rbac.server.vo.UserVo;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import io.swagger.annotations.ApiParam;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.Pageable;
import org.springframework.data.web.PageableDefault;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestMethod;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.RestController;

import java.security.Principal;
import java.util.List;
import java.util.Set;

@Api(value = "UserController")
@RestController
@RequestMapping("/v1/nebula/users")
public class UserController extends BaseController {

  @Autowired
  private UserService userService;

  /**
   * 日志
   */
  private static final Logger LOGGER = LoggerFactory.getLogger(UserController.class);
  
  @ApiOperation(value = "添加一个新的用户信息（后台管理人员，业务功能操作者）", notes = "")
  @RequestMapping(value = "", method = RequestMethod.POST)
  public ResponseModel create(
      @ApiParam(name = "user", value = "用户对象信息", required = true) @RequestBody UserVo user) {
    try {
      UserVo currentPoliceMan = this.userService.create(user);
      return this.buildHttpReslutW(currentPoliceMan, new String[] {});
    } catch (Exception e) {
      LOGGER.error(e.getMessage(), e);
      return this.buildHttpReslutForException(e);
    }
  }

  @ApiOperation(value = "修改一个已经存在的用户信息", notes = "可以修改的包括属性有：<br>" + "用户姓名、性别.0保密，1男  2女 信息")
  @RequestMapping(value = "", method = RequestMethod.PATCH)
  public ResponseModel update(
      @ApiParam(name = "user", value = "以对象方式传入的需要修改的属性，不能修改的属性即使传入该属性也不会发生变化", required = true) @RequestBody UserVo user) {
    try {
      UserVo currentUser = this.userService.update(user);
      return this.buildHttpReslutW(currentUser, new String[] {});
    } catch (Exception e) {
      LOGGER.error(e.getMessage(), e);
      return this.buildHttpReslutForException(e);
    }
  }

  @ApiOperation(value = "重置指定用户的密码信息")
  @RequestMapping(value = "/updatePassword", method = RequestMethod.PATCH)
  public ResponseModel updatePassword(
      @ApiParam(name = "userId", value = "指定的用户", required = true) String userId,
      @ApiParam(name = "oldPassword", value = "原始密码（没有加密的）", required = true) String oldPassword,
      @ApiParam(name = "newPassword", value = "新的密码（没有加密的）", required = true) String newPassword) {
    try {
      UserVo currentUser = this.userService.updatePassword(userId , newPassword , oldPassword);
      return this.buildHttpReslutW(currentUser, new String[] {});
    } catch (Exception e) {
      LOGGER.error(e.getMessage(), e);
      return this.buildHttpReslutForException(e);
    }
  }

  @ApiOperation(value = "重置指定用户的密码信息（忽略原始密码，该方法一般由管理员使用）")
  @RequestMapping(value = "/updatePasswordIgnoreOld", method = RequestMethod.PATCH)
  public ResponseModel updatePasswordIgnoreOld(
      @ApiParam(name = "userId", value = "指定的用户", required = true) String userId,
      @ApiParam(name = "newPassword", value = "新的密码（没有加密的）", required = true) String newPassword) {
    try {
      UserVo currentUser = this.userService.updatePassword(userId, newPassword);
      return this.buildHttpReslutW(currentUser, new String[] {});
    } catch (Exception e) {
      LOGGER.error(e.getMessage(), e);
      return this.buildHttpReslutForException(e);
    }
  }

  @ApiOperation(value = "修改指定的用户信息的状态，包括可用状态和不可用状态")
  @RequestMapping(value = "/updateStatus", method = RequestMethod.PATCH)
  public ResponseModel updateStatus(
      @ApiParam(name = "account", value = "指定的新的用户账号", required = true) String account,
      @ApiParam(name = "flag", value = "指定的新的状态", required = true) boolean flag) {
    try {
      if(flag) {
        this.userService.enabled(account);
      } else {
        this.userService.disable(account);
      }      
      return this.buildHttpReslut();
    } catch (Exception e) {
      LOGGER.error(e.getMessage(), e);
      return this.buildHttpReslutForException(e);
    }
  }

  @ApiOperation(value = "查询当前用户信息，且包括当前用户已经绑定（包括间接绑定）的角色信息、用户组、组织机构和岗位")
  @RequestMapping(value = "/{userId}", method = RequestMethod.GET)
  public ResponseModel findDetailsById(
      @ApiParam(name = "userId", value = "指定的用户编号", required = true) @PathVariable("userId") String userId) {
    try {
      UserVo currentUser = this.userService.findDetailsById(userId);
      return this.buildHttpReslutW(currentUser,new String[] {"roles","positions","groups","orgs"});
    } catch (Exception e) {
      LOGGER.error(e.getMessage(), e);
      return this.buildHttpReslutForException(e);
    }
  }
  
  @ApiOperation(value = "查询当前用户信息(按照用户账号查询)，不包括当前用户已经绑定（包括间接绑定）的角色信息、用户组、组织机构和岗位")
  @RequestMapping(value = "/findByAccount", method = RequestMethod.GET)
  public ResponseModel findByAccount(@ApiParam(name = "account", value = "指定的用户账户（肯定是唯一的）") @RequestParam("account") String account) {
    try {
      UserVo currentUser = this.userService.findByAccount(account);
      return this.buildHttpReslutW(currentUser,new String[] {});
    } catch (Exception e) {
      LOGGER.error(e.getMessage(), e);
      return this.buildHttpReslutForException(e);
    }
  }
  
  @ApiOperation(value = "查询当前已登录的操作者信息(按照用户账号查询)，不包括当前操作者已经绑定（包括间接绑定）的角色信息、用户组、组织机构和岗位")
  @RequestMapping(value = "/findByPrincipal", method = RequestMethod.GET)
  public ResponseModel findByPrincipal() {
    try {
      Principal op = this.getPrincipal();
      UserVo currentUser = this.userService.findByAccount(op.getName());
      return this.buildHttpReslutW(currentUser,new String[] {});
    } catch (Exception e) {
      LOGGER.error(e.getMessage(), e);
      return this.buildHttpReslutForException(e);
    }
  } 
  
  @ApiOperation(value = "预制控件查询，暂时查询当前操作用户基础信息")
  @RequestMapping(value = "/findByUserId",method = RequestMethod.GET)
  public ResponseModel findByUserId(@ApiParam(name = "userId", value = "指定的用户id信息") @RequestParam("userId")String userId){
    try {
      UserVo creator = this.userService.findByUserId(userId);
      return this.buildHttpReslutW(creator,new String[] {});
    } catch (Exception e) {
      LOGGER.error(e.getMessage(), e);
      return this.buildHttpReslutForException(e);
    }
  }

  @ApiOperation(value = "按照条件搜索用户，不支持模糊查询")
  @RequestMapping(value = "/findByConditions", method = RequestMethod.GET)
  public ResponseModel findByConditions(
      @ApiParam(name = "userName", value = "用户姓名") String userName,
      @ApiParam(name = "account", value = "用户账户") String account,
      @ApiParam(name = "useStatus", value = "用户状态") Integer useStatus,
      @PageableDefault(value = 50) Pageable pageable) {
    try {
      Page<UserVo> page = this.userService.findByConditions(userName, account, useStatus, pageable);
      return this.buildHttpReslutW(page, new String[] {"roles"});
    } catch (Exception e) {
      LOGGER.error(e.getMessage(), e);
      return this.buildHttpReslutForException(e);
    }
  }

  @ApiOperation(value = "查询多个用户信息")
  @RequestMapping(value = "/findByIds",method = RequestMethod.POST)
  public ResponseModel findByIds(@RequestBody List<String> ids){
    try {
      Set<UserVo> users = this.userService.findByIds(ids);
      return this.buildHttpReslutW(users,new String[] {});
    } catch (Exception e) {
      LOGGER.error(e.getMessage(), e);
      return this.buildHttpReslutForException(e);
    }
  }
}
