package com.bizunited.platform.core.controller.dataview;

import com.bizunited.platform.core.controller.BaseController;
import com.bizunited.platform.core.controller.model.ResponseModel;
import com.bizunited.platform.core.entity.DataSourceEntity;
import com.bizunited.platform.core.repository.dynamic.DynamicDataSourceManager;
import com.bizunited.platform.core.service.dataview.DataSourceService;
import io.swagger.annotations.ApiOperation;
import io.swagger.annotations.ApiParam;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.PutMapping;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestMethod;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.RestController;
import java.util.List;

/**
 * 第三方数据源业务模型的MVC Controller层实现，基于HTTP Restful风格
 * @author saturn
 */
@RestController
@RequestMapping("/v1/nebula/dataSources")
public class DataSourceController extends BaseController { 
  @Autowired
  private DataSourceService dataSourceService;
  @Autowired
  private DynamicDataSourceManager dynamicDataSourceManager;

  /**
   * 相关的创建过程，http接口。请注意该创建过程除了可以创建dataSourceEntity中的基本信息以外，还可以对dataSourceEntity中属于OneToMany关联的明细信息一同进行创建注意：创建操作传入的dataSourceEntityJSON对象，其主键信息不能有值，服务端将会自动为其赋予相关值。另外，创建操作成功后，系统将返回该对象的基本信息（不包括任何关联信息）
   * */
  @ApiOperation(value = "相关的创建过程，http接口。请注意该创建过程除了可以创建dataSourceEntity中的基本信息以外，还可以对dataSourceEntity中属于OneToMany关联的明细信息一同进行创建注意：创建操作传入的dataSourceEntityJSON对象，其主键信息不能有值，服务端将会自动为其赋予相关值。另外，创建操作成功后，系统将返回该对象的基本信息（不包括任何关联信息）")
  @PostMapping(value="")
  public ResponseModel create(@RequestBody @ApiParam(name="dataSource" , value="相关的创建过程，http接口。请注意该创建过程除了可以创建dataSourceEntity中的基本信息以外，还可以对dataSourceEntity中属于OneToMany关联的明细信息一同进行创建注意：创建操作传入的dataSourceEntityJSON对象，其主键信息不能有值，服务端将会自动为其赋予相关值。另外，创建操作成功后，系统将返回该对象的基本信息（不包括任何关联信息）") DataSourceEntity dataSource) {
    try {
      DataSourceEntity currentDataSource = this.dynamicDataSourceManager.create(dataSource);
      return this.buildHttpReslutW(currentDataSource, new String[]{});
    } catch(Exception e) {
      return this.buildHttpReslutForException(e);
    }
  }

  /**
   * 相关的更新过程，http接口。请注意该更新过程只会更新在模型层被标记为了updateable的属性，包括一般属性、ManyToOne和OneToOne性质的关联属性，而ManyToMany、OneToMany的关联属性，虽然也会传入，但需要开发人员自行在Service层完善其更新过程注意：修改操作传入的dataSourceEntityJSON对象，其主键信息必须有值，服务端将验证这个主键值是否已经存在。另外，创建操作成功后，系统将返回该对象的基本信息（不包括任何关联信息）
   * */
  @ApiOperation(value = "相关的更新过程，http接口。请注意该更新过程只会更新在模型层被标记为了updateable的属性，包括一般属性、ManyToOne和OneToOne性质的关联属性，而ManyToMany、OneToMany的关联属性，虽然也会传入，但需要开发人员自行在Service层完善其更新过程注意：修改操作传入的dataSourceEntityJSON对象，其主键信息必须有值，服务端将验证这个主键值是否已经存在。另外，创建操作成功后，系统将返回该对象的基本信息（不包括任何关联信息）")
  @PutMapping(value="")
  public ResponseModel update(@RequestBody @ApiParam(name="dataSource" , value="相关的更新过程，http接口。请注意该更新过程只会更新在模型层被标记为了updateable的属性，包括一般属性、ManyToOne和OneToOne性质的关联属性，而ManyToMany、OneToMany的关联属性，虽然也会传入，但需要开发人员自行在Service层完善其更新过程注意：修改操作传入的dataSourceEntityJSON对象，其主键信息必须有值，服务端将验证这个主键值是否已经存在。另外，创建操作成功后，系统将返回该对象的基本信息（不包括任何关联信息）") DataSourceEntity dataSource) {
    try {
      DataSourceEntity currentDataSource = this.dynamicDataSourceManager.update(dataSource);
      return this.buildHttpReslutW(currentDataSource, new String[]{});
    } catch(Exception e) {
      return this.buildHttpReslutForException(e);
    }
  }
  
  @ApiOperation(value = "修改指定的数据源状态---(启用、禁用)")
  @RequestMapping(value = "/updateStatus/{dataSourceCode}", method = RequestMethod.PATCH)
  public ResponseModel updateStatus(@PathVariable(value="dataSourceCode") String dataSourceCode , @ApiParam(name="tstatus" , value="1：启；0和其它值为禁用") @RequestParam("tstatus") Integer tstatus) {
    try {
      if(tstatus == 1) {
        this.dynamicDataSourceManager.enable(dataSourceCode);
      } 
      // 其它情况均视为禁用
      else {
        this.dynamicDataSourceManager.disable(dataSourceCode);
      }
      return this.buildHttpReslut();
    } catch(Exception e) {
      return this.buildHttpReslutForException(e);
    }
  }
  
  @ApiOperation(value = "删除一个数据源，无论这个数据源的状态如何、是否正在工作")
  @RequestMapping(value = "/{dataSourceCode}", method = RequestMethod.DELETE)
  public ResponseModel delete(@PathVariable(value="dataSourceCode") String dataSourceCode) {
    try {
      this.dynamicDataSourceManager.delete(dataSourceCode);
      return this.buildHttpReslut();
    } catch(Exception e) {
      return this.buildHttpReslutForException(e);
    }
  }
  
  @ApiOperation(value="检查指定的数据源是否可用。检查方式就是用dataSource对象描述的信息基于最原始的java jdbc连接进行操作。 如果当前连接成功，则测试成功。"
      , notes="如果连接过程出现问题，则抛出运行时异常")
  @PostMapping(value="/validate")
  public ResponseModel validate(@RequestBody @ApiParam(name="dataSource" , value="数据源对象描述") DataSourceEntity dataSource) {
    try {
      this.dynamicDataSourceManager.check(dataSource);
      return this.buildHttpReslut();
    } catch(Exception e) {
      return this.buildHttpReslutForException(e);
    }
  }

  /**
   * 按照数据源编号进行查询明细查询，查询的明细包括当前业务表单所有的关联属性。
   * @param code 数据源编号
   */
  @ApiOperation(value = "按照数据源编号进行查询明细查询，查询的明细包括当前业务表单所有的关联属性。")
  @RequestMapping(value="/findDetailsByCode" , method={RequestMethod.GET})
  public ResponseModel findDetailsByCode(@RequestParam("code") @ApiParam("数据源编号") String code) {
    try {
      DataSourceEntity dataSource = this.dataSourceService.findDetailsByCode(code);
      return this.buildHttpReslutW(dataSource, new String[]{});
    } catch(Exception e) {
      return this.buildHttpReslutForException(e);
    }
  }
  
  /**
   * 查找当前记录的所有第三方数据源信息
   */
  @ApiOperation(value = "查找当前记录的所有第三方数据源信息")
  @RequestMapping(value="" , method={RequestMethod.GET})
  public ResponseModel findAll() {
    try {
      List<DataSourceEntity> dataSources = this.dataSourceService.findAll();
      return this.buildHttpReslutW(dataSources, new String[]{});
    } catch(Exception e) {
      return this.buildHttpReslutForException(e);
    }
  }
} 
