package com.bizunited.platform.core.controller.dataview;

import com.alibaba.fastjson.JSONArray;
import com.alibaba.fastjson.JSONObject;
import com.bizunited.platform.core.controller.BaseController;
import com.bizunited.platform.core.controller.model.ResponseModel;
import com.bizunited.platform.core.entity.DataViewEntity;
import com.bizunited.platform.core.service.dataview.DataViewService;
import com.bizunited.platform.core.service.invoke.model.InvokeOperations;
import com.bizunited.platform.core.service.invoke.model.InvokeParams;
import com.bizunited.platform.core.vo.dataview.DataviewExportExcelFieldVo;
import io.swagger.annotations.ApiImplicitParam;
import io.swagger.annotations.ApiImplicitParams;
import io.swagger.annotations.ApiOperation;
import io.swagger.annotations.ApiParam;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.domain.Pageable;
import org.springframework.web.bind.annotation.PatchMapping;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestMethod;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.RestController;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import java.security.Principal;
import java.util.List;
import java.util.concurrent.Future;
import java.util.concurrent.TimeUnit;

/**
 * DataViewEntity业务模型的MVC Controller层实现，基于HTTP Restful风格
 * @author ywt
 */
@RestController
@RequestMapping("/v1/nebula/dataViews")
public class DataViewController extends BaseController {
  /**
   * 日志
   */
  private static final Logger LOGGER = LoggerFactory.getLogger(DataViewController.class);
  /**
   * 超时时间（秒）
   */
  private static final Integer TIME_OUT = 10;

  private static final String PAGE_PARAM_NAME = "page";
  private static final String VIEW_PARAM_NAME = "dataView";
  private static final String INPUT_PARAM_NAME = "inputParams";
  private static final String KUIPER_PARAM_NAME = "kuiperInvokeParams";
  private static final String DATASOURCE_PARAM_NAME = "dataSource";
  private static final String GROUP_PARAM_NAME = "dataViewGroup";
  private static final String FIELDS_PARAM_NAME = "fields";
  private static final String FILTERS_PARAM_NAME = "filters";
  private static final String SYSTEM_PARAM_NAME = "systemFilters";
  private static final String[] FIELD_FILTERS = {DATASOURCE_PARAM_NAME,GROUP_PARAM_NAME,FIELDS_PARAM_NAME,FILTERS_PARAM_NAME,SYSTEM_PARAM_NAME};
  private static final String[] FIELD_FILTERS_F = new String[]{DATASOURCE_PARAM_NAME,GROUP_PARAM_NAME,FIELDS_PARAM_NAME,FILTERS_PARAM_NAME,"filters.field",SYSTEM_PARAM_NAME};
  
  @Autowired
  private DataViewService dataViewEntityService;

  /**
   * 相关的创建过程，http接口。请注意该创建过程除了可以创建dataViewEntity中的基本信息以外，还可以对dataViewEntity中属于OneToMany关联的明细信息一同进行创建注意：创建操作传入的dataViewEntityJSON对象，其主键信息不能有值，服务端将会自动为其赋予相关值。另外，创建操作成功后，系统将返回该对象的基本信息（不包括任何关联信息）
   * */
  @ApiOperation(value = "相关的创建过程，http接口。请注意该创建过程除了可以创建dataViewEntity中的基本信息以外，还可以对dataViewEntity中属于OneToMany关联的明细信息一同进行创建注意：创建操作传入的dataViewEntityJSON对象，其主键信息不能有值，服务端将会自动为其赋予相关值。另外，创建操作成功后，系统将返回该对象的基本信息（不包括任何关联信息）")
  @PostMapping(value="")
  public ResponseModel create(@RequestBody @ApiParam(name="dataViewEntity" , value="相关的创建过程，http接口。请注意该创建过程除了可以创建dataViewEntity中的基本信息以外，还可以对dataViewEntity中属于OneToMany关联的明细信息一同进行创建注意：创建操作传入的dataViewEntityJSON对象，其主键信息不能有值，服务端将会自动为其赋予相关值。另外，创建操作成功后，系统将返回该对象的基本信息（不包括任何关联信息）") DataViewEntity dataViewEntity) {
    try {
      DataViewEntity current = this.dataViewEntityService.create(dataViewEntity , false);
      return this.buildHttpReslutW(current, new String[]{});
    } catch(Exception e) {
      LOGGER.error(e.getMessage() , e);
      return this.buildHttpReslutForException(e);
    }
  }

  /**
   * 相关的更新过程，http接口。请注意该更新过程只会更新在模型层被标记为了updateable的属性，包括一般属性、ManyToOne和OneToOne性质的关联属性，而ManyToMany、OneToMany的关联属性，虽然也会传入，但需要开发人员自行在Service层完善其更新过程注意：修改操作传入的dataViewEntityJSON对象，其主键信息必须有值，服务端将验证这个主键值是否已经存在。另外，创建操作成功后，系统将返回该对象的基本信息（不包括任何关联信息）
   * */
  @ApiOperation(value = "相关的更新过程，http接口。请注意该更新过程只会更新在模型层被标记为了updateable的属性，包括一般属性、ManyToOne和OneToOne性质的关联属性，而ManyToMany、OneToMany的关联属性，虽然也会传入，但需要开发人员自行在Service层完善其更新过程注意：修改操作传入的dataViewEntityJSON对象，其主键信息必须有值，服务端将验证这个主键值是否已经存在。另外，创建操作成功后，系统将返回该对象的基本信息（不包括任何关联信息）")
  @PatchMapping(value="")
  public ResponseModel update(@RequestBody @ApiParam(name="dataViewEntity" , value="相关的更新过程，http接口。请注意该更新过程只会更新在模型层被标记为了updateable的属性，包括一般属性、ManyToOne和OneToOne性质的关联属性，而ManyToMany、OneToMany的关联属性，虽然也会传入，但需要开发人员自行在Service层完善其更新过程注意：修改操作传入的dataViewEntityJSON对象，其主键信息必须有值，服务端将验证这个主键值是否已经存在。另外，创建操作成功后，系统将返回该对象的基本信息（不包括任何关联信息）") DataViewEntity dataViewEntity) {
    try {
      DataViewEntity current = this.dataViewEntityService.update(dataViewEntity);
      return this.buildHttpReslutW(current, new String[]{});
    } catch(Exception e) {
      LOGGER.error(e.getMessage() , e);
      return this.buildHttpReslutForException(e);
    }
  }

  /**
   * 按照数据源code查询 视图、分组、字段等统计信息
   */
  @ApiOperation(value = "按照数据源code查询DataViewEntity")
  @ApiImplicitParam(name = "dataSourceCode", value = "按照数据源ID查询DataViewEntity", required = true)
  @RequestMapping(value="/findByDataSourceCode" , method={RequestMethod.GET})
  public ResponseModel findByDataSourceCode(@RequestParam(required=false) @ApiParam("数据源code") String dataSourceCode) {
    try {
      JSONArray jsonArr = this.dataViewEntityService.findByDataSourceCode(dataSourceCode);
      return this.buildHttpReslutW(jsonArr, new String[]{});
    } catch(Exception e) {
      LOGGER.error(e.getMessage() , e);
      return this.buildHttpReslutForException(e);
    } 
  } 
  
  /**
   * 按照数据源ID查询DataViewEntity
   */
  @ApiOperation(value = "按照数据源ID查询DataViewEntity")
  @ApiImplicitParam(name = "dataSourceId", value = "按照数据源ID查询DataViewEntity", required = true)
  @RequestMapping(value="/findByDataSource" , method={RequestMethod.GET})
  public ResponseModel findByDataSource(@RequestParam("dataSourceId") @ApiParam("数据源ID") String dataSourceId) {
    try {
      List<DataViewEntity> result = this.dataViewEntityService.findByDataSource(dataSourceId);
      return this.buildHttpReslutW(result, new String[]{DATASOURCE_PARAM_NAME});
    } catch(Exception e) {
      LOGGER.error(e.getMessage() , e);
      return this.buildHttpReslutForException(e);
    } 
  } 
  
  /**
   * 按照数据库分组查询DataViewEntity
   */
  @ApiOperation(value = "按照数据库分组查询DataViewEntity")
  @ApiImplicitParam(name = "dataViewGroupId", value = "按照数据库分组查询DataViewEntity", required = true)
  @RequestMapping(value="/findByDataViewGroup" , method={RequestMethod.GET})
  public ResponseModel findByDataViewGroup(@RequestParam("dataViewGroupId") @ApiParam("数据库分组") String dataViewGroupId) {
    try {
      List<DataViewEntity> result = this.dataViewEntityService.findByDataViewGroup(dataViewGroupId);
      return this.buildHttpReslutW(result, new String[]{GROUP_PARAM_NAME});
    } catch(Exception e) {
      LOGGER.error(e.getMessage() , e);
      return this.buildHttpReslutForException(e);
    } 
  } 
  /**
   * 按照结构字段查询DataViewEntity
   */
  @ApiOperation(value = "按照结构字段查询DataViewEntity")
  @ApiImplicitParam(name = "filtersId", value = "按照结构字段查询DataViewEntity", required = true)
  @RequestMapping(value="/findByFields" , method={RequestMethod.GET})
  public ResponseModel findByFields(@RequestParam("fieldsId") @ApiParam("结构字段") String fieldsId) {
    try {
      DataViewEntity result = this.dataViewEntityService.findByFields(fieldsId);
      return this.buildHttpReslutW(result, new String[]{FIELDS_PARAM_NAME});
    } catch(Exception e) {
      LOGGER.error(e.getMessage() , e);
      return this.buildHttpReslutForException(e);
    } 
  } 
  /**
   * 按照过滤筛选字段查询DataViewEntity
   */
  @ApiOperation(value = "按照过滤筛选字段查询DataViewEntity")
  @ApiImplicitParam(name = "filtersId", value = "按照过滤筛选字段查询DataViewEntity", required = true)
  @RequestMapping(value="/findByFilters" , method={RequestMethod.GET})
  public ResponseModel findByFilters(@RequestParam("filtersId") @ApiParam("过滤筛选字段") String filtersId) {
    try {
      DataViewEntity result = this.dataViewEntityService.findByFilters(filtersId);
      return this.buildHttpReslutW(result, new String[]{FILTERS_PARAM_NAME});
    } catch(Exception e) {
      LOGGER.error(e.getMessage() , e);
      return this.buildHttpReslutForException(e);
    } 
  } 
  /**
   * 按照系统查询字段查询DataViewEntity
   */
  @ApiOperation(value = "按照系统查询字段查询DataViewEntity")
  @ApiImplicitParam(name = "systemFiltersId", value = "按照系统查询字段查询DataViewEntity", required = true)
  @RequestMapping(value="/findBySystemFilters" , method={RequestMethod.GET})
  public ResponseModel findBySystemFilters(@RequestParam("systemFiltersId") @ApiParam("系统查询字段") String systemFiltersId) {
    try {
      DataViewEntity result = this.dataViewEntityService.findBySystemFilters(systemFiltersId);
      return this.buildHttpReslutW(result, new String[]{SYSTEM_PARAM_NAME});
    } catch(Exception e) {
      LOGGER.error(e.getMessage() , e);
      return this.buildHttpReslutForException(e);
    } 
  } 
  
  @ApiOperation(value = "按照数据视图的唯一业务编号信息进行详情查询（包括关联信息）")
  @RequestMapping(value="/findDetailsByCode" , method={RequestMethod.GET})
  public ResponseModel findDetailsByCode(@RequestParam("code") @ApiParam("数据视图的唯一业务编号信息") String code) {
    try { 
      DataViewEntity result = this.dataViewEntityService.findDetailsByCode(code); 
      return this.buildHttpReslutW(result, FIELD_FILTERS_F);
    } catch(Exception e) {
      LOGGER.error(e.getMessage() , e);
      return this.buildHttpReslutForException(e);
    } 
  }  
  
  @ApiOperation(value = "使用指定的第三方数据源，进行正式的查询SQL执行，但是最多只返回一条记录")
  @ApiImplicitParams({@ApiImplicitParam(name = "dataSourceCode", value = "第三方数据源编码", required = true),
    @ApiImplicitParam(name = "json", value = "第三方数据视图dataView和执行参数inputParams", required = true)
  })
  @RequestMapping(value="/executeTop1" , method={RequestMethod.POST})
  public ResponseModel executeTop1(@RequestParam("dataSourceCode")String dataSourceCode , @RequestBody JSONObject json) {
    try { 
      Principal principal = this.getPrincipal();
      DataViewEntity dataView = this.resolveDataView(json);
      InvokeParams inputParams = this.resolveKuiperInvokeParams(json);
      Future<?> futrue = this.dataViewEntityService.executeTop1(dataSourceCode,dataView,inputParams,principal);
      Object result = futrue.get(TIME_OUT,TimeUnit.SECONDS);
      if(result instanceof Exception) {
        throw new IllegalArgumentException((Exception)result);
      }
      JSONArray jsonArr = (JSONArray) result;
      return this.buildHttpReslutW(jsonArr, new String[]{});
    } catch(Exception e) {
      LOGGER.error(e.getMessage() , e);
      return this.buildHttpReslutForException(e);
    } 
  }
  
  
  @ApiOperation(value = "使用主数据源，进行正式的查询SQL执行，但是最多只返回一条记录")
  @ApiImplicitParams({
    @ApiImplicitParam(name = "json", value = "主数据源数据视图dataView和执行参数inputParams", required = true)
  })
  @RequestMapping(value="/mainExecuteTop1" , method={RequestMethod.POST})
  public ResponseModel executeTop1(@RequestBody JSONObject json) {
    try { 
      Principal principal = this.getPrincipal();
      DataViewEntity dataView = this.resolveDataView(json);
      InvokeParams inputParams = this.resolveKuiperInvokeParams(json);
      JSONArray jsonArr = this.dataViewEntityService.executeTop1(dataView,inputParams,principal); 
      return this.buildHttpReslutW(jsonArr, new String[]{});
    } catch(Exception e) {
      LOGGER.error(e.getMessage() , e);
      return this.buildHttpReslutForException(e);
    } 
  }
  
  @SuppressWarnings("unchecked")
  @ApiOperation(value = "使用指定的第三方数据源，进行正式的分页SQL查询")
  @ApiImplicitParams({@ApiImplicitParam(name = "dataSourceCode", value = "第三方数据源编码", required = true),
    @ApiImplicitParam(name = "json", value = "第三方数据视图dataView和执行参数inputParams", required = true)
  })
  @RequestMapping(value="/executeByPage" , method={RequestMethod.POST})
  public ResponseModel executeByPage(@RequestParam("dataSourceCode")String dataSourceCode , @RequestBody JSONObject json) {
    try { 
      Principal principal = this.getPrincipal();
      Pageable pageable = this.resolvePageable(json);
      DataViewEntity dataView = this.resolveDataView(json);
      InvokeParams inputParams = this.resolveKuiperInvokeParams(json);
      Future<?> futrue = this.dataViewEntityService.execute(dataSourceCode,dataView,inputParams,pageable,principal);
      Object result = futrue.get(TIME_OUT,TimeUnit.SECONDS);
      if(result instanceof Exception) {
        throw new IllegalArgumentException((Exception)result);
      }
      Page<JSONObject> page = (Page<JSONObject>) result;
      return this.buildHttpReslutW(page, new String[]{});
    } catch(Exception e) {
      LOGGER.error(e.getMessage() , e);
      return this.buildHttpReslutForException(e);
    } 
  }
  
  
  @ApiOperation(value = "使用主数据源，进行正式的查询SQL执行")
  @ApiImplicitParams({
    @ApiImplicitParam(name = "json", value = "主数据源数据视图dataView和执行参数inputParams", required = true)
  })
  @RequestMapping(value="/mainExecuteByPage" , method={RequestMethod.POST})
  public ResponseModel mainExecuteByPage(@RequestBody JSONObject json) {
    try { 
      Principal principal = this.getPrincipal();
      Pageable pageable = this.resolvePageable(json);
      DataViewEntity dataView = this.resolveDataView(json);
      InvokeParams inputParams = this.resolveKuiperInvokeParams(json);
      Page<JSONObject> page = this.dataViewEntityService.execute(dataView,inputParams,pageable,principal); 
      return this.buildHttpReslutW(page, new String[]{});
    } catch(Exception e) {
      LOGGER.error(e.getMessage() , e);
      return this.buildHttpReslutForException(e);
    } 
  }
  
  @ApiOperation(value = "使用指定的第三方数据源，进行正式的查询SQL执行")
  @ApiImplicitParams({@ApiImplicitParam(name = "dataSourceCode", value = "第三方数据源编码", required = true),
    @ApiImplicitParam(name = "json", value = "第三方数据视图dataView和执行参数inputParams", required = true)
  })
  @RequestMapping(value="/execute" , method={RequestMethod.POST})
  public ResponseModel execute(@RequestParam("dataSourceCode")String dataSourceCode , @RequestBody JSONObject json) {
    try { 
      Principal principal = this.getPrincipal();
      DataViewEntity dataView = this.resolveDataView(json);
      InvokeParams inputParams = this.resolveKuiperInvokeParams(json);
      Future<?> futrue = this.dataViewEntityService.execute(dataSourceCode,dataView,inputParams,principal);
      Object result = futrue.get(TIME_OUT,TimeUnit.SECONDS);
      if(result instanceof Exception) {
        throw new IllegalArgumentException((Exception)result);
      }
      JSONArray jsonArr = (JSONArray) result;
      return this.buildHttpReslutW(jsonArr, new String[]{});
    } catch(Exception e) {
      LOGGER.error(e.getMessage() , e);
      return this.buildHttpReslutForException(e);
    } 
  }
  
  
  @ApiOperation(value = "使用主数据源，进行正式的查询SQL执行")
  @ApiImplicitParams({
    @ApiImplicitParam(name = "json", value = "主数据源数据视图dataView和执行参数inputParams", required = true)
  })
  @RequestMapping(value="/mainExecute" , method={RequestMethod.POST})
  public ResponseModel mainExecute(@RequestBody JSONObject json) {
    try { 
      Principal principal = this.getPrincipal();
      DataViewEntity dataView = this.resolveDataView(json);
      InvokeParams inputParams = this.resolveKuiperInvokeParams(json);
      JSONArray jsonArr = this.dataViewEntityService.execute(dataView,inputParams,principal); 
      return this.buildHttpReslutW(jsonArr, new String[]{});
    } catch(Exception e) {
      LOGGER.error(e.getMessage() , e);
      return this.buildHttpReslutForException(e);
    } 
  }

  /**
   * 导出数据到excel
   * @param dataSourceCode
   * @param listTemplateId
   * @param json
   * @return
   */
  @PostMapping("exportExcel")
  @ApiOperation("导出数据到excel")
  public void exportExcel(HttpServletRequest request, HttpServletResponse response,
                          @ApiParam("数据源编码") String dataSourceCode,
                          @RequestParam("dataView") String dataViewJson,
                          @RequestParam(value = "inputParams", required = false) String inputParamsJson,
                          @RequestParam("exportExcelParams") String exportExcelParamsJson) {
    try {
      Principal principal = this.getPrincipal();
      DataViewEntity dataView = null;
      if(StringUtils.isNotBlank(dataViewJson)) {
        dataView = JSONObject.parseObject(dataViewJson).toJavaObject(DataViewEntity.class);
      }
      InvokeParams inputParams = null;
      if(StringUtils.isNotBlank(inputParamsJson)) {
        inputParams = this.resolveInvokeParams(JSONObject.parseObject(inputParamsJson));
      }
      List<DataviewExportExcelFieldVo> excelFields = JSONArray.parseArray(exportExcelParamsJson).toJavaList(DataviewExportExcelFieldVo.class);
      byte[] bytes = dataViewEntityService.exportExcel(dataSourceCode, dataView, inputParams, principal, excelFields);
      super.writeResponseFile(request, response, bytes, "列表导出.xlsx");
    } catch (RuntimeException e) {
      LOGGER.error(e.getMessage(), e);
    }
  }

  @ApiOperation(value = "根据已有的视图ID以及系统参数信息（json格式），查找已有的数据信息")
  @ApiImplicitParams({
    @ApiImplicitParam(name = "json", value = "数据源数据视图dataView和执行参数inputParams", required = true)
  })
  @RequestMapping(value="/findByViewAndDataKey" , method={RequestMethod.POST})
  public ResponseModel findByViewAndDataKey(@RequestBody JSONObject json) {
    try { 
      Principal principal = this.getPrincipal();
      DataViewEntity dataView = this.resolveDataView(json);
      InvokeParams inputParams = this.resolveKuiperInvokeParams(json);
      String primaryKey = json.getString("primaryKey");
      JSONArray primaryValue = json.getJSONArray("primaryValue");
      JSONArray jsonArr = this.dataViewEntityService.findByViewAndDataKey(dataView, inputParams,principal,primaryKey,primaryValue); 
      return this.buildHttpReslutW(jsonArr, new String[]{});
    } catch(Exception e) {
      LOGGER.error(e.getMessage() , e);
      return this.buildHttpReslutForException(e);
    } 
  }
  

  @ApiOperation(value = "第三方数据源语句输出分析")
  @RequestMapping(value = "/sqlAnalysis",method = {RequestMethod.PATCH})
  public ResponseModel sqlAnalysis(@RequestParam("dataSourceCode") @ApiParam("第三方数据源编码") String dataSourceCode,@RequestBody @ApiParam() DataViewEntity dataView){
    try {
      DataViewEntity result = this.dataViewEntityService.executeResource(dataSourceCode,dataView);
      return this.buildHttpReslutW(result, FIELD_FILTERS);
    } catch(Exception e) {
      LOGGER.error(e.getMessage() , e);
      return this.buildHttpReslutForException(e);
    }
  }

  @ApiOperation(value = "主数据源语句输出分析")
  @RequestMapping(value = "/mainSqlAnalysis",method = {RequestMethod.PATCH})
  public ResponseModel mainSqlAnalysis(@RequestBody @ApiParam() DataViewEntity dataView){
    try {
      DataViewEntity result = this.dataViewEntityService.executeResource(dataView);
      return this.buildHttpReslutW(result, FIELD_FILTERS);
    } catch(Exception e) {
      LOGGER.error(e.getMessage() , e);
      return this.buildHttpReslutForException(e);
    }
  }
  
  /**
   * 解析视图json ， 并根据前端传入的视图id查询视图详细信息，并验证
   * @param json
   * @return
   */
  private DataViewEntity resolveDataView(JSONObject json) {
    JSONObject jsonDataView = json.getJSONObject(VIEW_PARAM_NAME);
    DataViewEntity dataView =  jsonDataView == null ? null : JSONObject.toJavaObject(jsonDataView, DataViewEntity.class);
    return dataView;
  }
  
  /**
   * 解析KuiperInvokeParams参数
   * @param json
   * @return
   */
  private InvokeParams resolveKuiperInvokeParams(JSONObject json) {
    JSONObject jsonInputParams = json.getJSONObject(INPUT_PARAM_NAME);
    return this.resolveInvokeParams(jsonInputParams);
  }

  /**
   * 解析KuiperInvokeParams参数
   * @param json
   * @return
   */
  private InvokeParams resolveInvokeParams(JSONObject jsonInputParams) {
    InvokeParams inputParams = new InvokeParams();
    if(jsonInputParams != null) {
      JSONObject models = jsonInputParams.getJSONObject(KUIPER_PARAM_NAME);
      for(String e : models.keySet()) {
        inputParams.add(e, models.getObject(e, InvokeOperations.class), false);
      }
    }
    return inputParams;
  }
  
  /**
   * 解析page参数
   */
  private Pageable resolvePageable(JSONObject json) {
    JSONObject jsonPage = json.getJSONObject(PAGE_PARAM_NAME);
    Pageable pageable = null;
    if(jsonPage != null && (jsonPage.getInteger("page") != null && jsonPage.getInteger("size") != null)) {
      pageable = PageRequest.of(jsonPage.getIntValue("page"), jsonPage.getIntValue("size"));
    }else {
      pageable = PageRequest.of(0, 50);
    }
    return pageable;
  }
} 
