package com.bizunited.platform.core.repository;

import java.util.Set;

import org.springframework.data.jpa.repository.JpaRepository;
import org.springframework.data.jpa.repository.JpaSpecificationExecutor;
import org.springframework.data.jpa.repository.Modifying;
import org.springframework.data.jpa.repository.Query;
import org.springframework.data.repository.query.Param;
import org.springframework.stereotype.Repository;

import com.bizunited.platform.core.entity.DynamicTaskSchedulerEntity;
import com.bizunited.platform.core.repository.internal.DynamicTaskSchedulerRepositoryCustom;

@Repository("DynamicTaskSchedulerRepository")
public interface DynamicTaskSchedulerRepository
    extends
      JpaRepository<DynamicTaskSchedulerEntity, String> ,
      JpaSpecificationExecutor<DynamicTaskSchedulerEntity> ,
      DynamicTaskSchedulerRepositoryCustom {
  
  /**
   * 按照指定的任务状态和工作状态查询符合条件的动态任务信息，并且按照任务code进行排序
   * @param tstatus 指定的动态任务状态
   * @param workingStatus 指定的动态任务工作状态（可以是多个）
   * @return
   */
  @Query("from DynamicTaskSchedulerEntity dts "
      + " left join fetch dts.script where dts.tstatus = :tstatus and dts.workingStatus in (:workingStatuses) order by dts.taskCode ")
  public Set<DynamicTaskSchedulerEntity> findByTstatusAndWorkingStatus(@Param("tstatus") int tstatus ,@Param("workingStatuses") int[] workingStatuses);
  /**
   * 按照指定的任务状态查询符合要求的动态任务信息，并按照任务code进行排序
   * @param tstatus 指定的动态任务状态
   * @return
   */
  @Query("from DynamicTaskSchedulerEntity dts where dts.tstatus = :tstatus  order by dts.taskCode ")
  public Set<DynamicTaskSchedulerEntity> findByTstatus(@Param("tstatus") int tstatus);
  /**
   * 按照指定的任务运行状态查询符合要求的动态任务信息，并按照任务code进行排序
   * @param workingStatus 指定的动态任务工作状态（可以是多个）
   * @return
   */
  @Query("from DynamicTaskSchedulerEntity dts where dts.workingStatus in (:workingStatuses) order by dts.taskCode ")
  public Set<DynamicTaskSchedulerEntity> findByWorkingStatus(@Param("workingStatuses") int[] workingStatuses);
  
  /**
   * 查询任务编号对应的动态任务信息
   * @param taskCode 指定的任务编号
   * @return
   */
  @Query("from DynamicTaskSchedulerEntity dts where dts.taskCode = :taskCode ")
  public DynamicTaskSchedulerEntity findByTaskCode(@Param("taskCode") String taskCode);
  /**
   * 查询任务编号对应的动态任务信息，包括动态任务的关联信息
   * @param taskCode 指定的任务编号
   * @return
   */
  @Query("from DynamicTaskSchedulerEntity dts "
      + " left join fetch dts.script sc"
      + " left join fetch dts.creator c "
      + " left join fetch dts.modifier m where dts.taskCode = :taskCode ")
  public DynamicTaskSchedulerEntity findDetailsByTaskCode(@Param("taskCode") String taskCode);
  
  /**
   *查询已存在的所有任务，按照创建时间进行倒序排列
   */
  @Query("from DynamicTaskSchedulerEntity dts order by dts.createTime desc ")
  public Set<DynamicTaskSchedulerEntity> findAlls();
  
  /**
   * 将指定的一个或者多个动态任务的状态变更为“失效”
   * @param taskCodes
   */
  @Modifying
  @Query(value = "update engine_dynamic_task set working_status = 1 , t_status = 0 where task_code in (:taskCodes)" , nativeQuery = true)
  public void updateInvalidByTaskCode(@Param("taskCodes") String[] taskCodes);
  /**
   * 将指定的一个或者多个动态任务的状态变更为“运行中”
   * @param taskCodes
   */
  @Modifying
  @Query(value = "update engine_dynamic_task set working_status = 2 , t_status = 1 where task_code in (:taskCodes)" , nativeQuery = true)
  public void updateRunningByTaskCode(@Param("taskCodes") String[] taskCodes);
}