package com.bizunited.platform.core.repository;

import com.bizunited.platform.core.entity.OrganizationEntity;
import com.bizunited.platform.core.entity.PositionEntity;
import com.bizunited.platform.core.repository.internal.PositionRepositoryCustom;
import org.springframework.data.jpa.repository.JpaRepository;
import org.springframework.data.jpa.repository.JpaSpecificationExecutor;
import org.springframework.data.jpa.repository.Modifying;
import org.springframework.data.jpa.repository.Query;
import org.springframework.data.repository.query.Param;
import org.springframework.stereotype.Repository;

import java.util.List;
import java.util.Set;

/**
 * PositionRepository
 *
 * @description:
 * @author: yanwe
 * @date: 17/Jan/2019 11:16
 */
@Repository("PositionRepository")
public interface PositionRepository
    extends JpaRepository<PositionEntity, String>, JpaSpecificationExecutor<PositionEntity>, PositionRepositoryCustom {

  /**
   * 根据编号查询
   *
   * @param code
   * @return
   */
  PositionEntity findByCode(@Param("code") String code);

  /**
   * 根据名称查询
   * @param name
   * @return
   */
  List<PositionEntity> findByName(@Param("name") String name);

  /**
   * 根据ID查询岗位的详细信息，包括相关组织、角色、用户等
   *
   * @param id
   * @return
   */
  @Query(
      "select distinct position from PositionEntity position "
          + "left join fetch position.organization "
          + "left join fetch position.roles "
          + "left join fetch position.users "
          + "where position.id = :id")
  PositionEntity findDetailsById(@Param("id") String id);

  /**
   * 根据Code查询岗位的详细信息，包括相关组织、角色、用户等
   *
   * @param code
   * @return
   */
  @Query(
      "select distinct position from PositionEntity position "
          + "left join fetch position.organization "
          + "left join fetch position.roles "
          + "left join fetch position.users "
          + "where position.code = :code")
  PositionEntity findDetailsByCode(@Param("code") String code);

  /**
   * 查询某一组织下所有岗位
   *
   * @param organization
   * @return
   */
  List<PositionEntity> findALlByOrAndOrganization(OrganizationEntity organization);

  /**
   * 查询某一角色与某一职位绑定关系
   * @param roleId
   * @param positionId
   * @return
   */
  @Query(value = "SELECT COUNT(*) FROM engine_position_role_mapping WHERE role_id = :roleId AND position_id = :positionId",nativeQuery = true)
  int countByRoleAndPosition(@Param("roleId") String roleId, @Param("positionId") String positionId);

  /**
   * 绑定某一角色与某一职位关系
   *
   * @param roleId
   * @param positionId
   */
  @Modifying
  @Query(
      value =
          "insert into engine_position_role_mapping(role_id,position_id) values (:roleId,:positionId)",
      nativeQuery = true)
  void bindRole(@Param("roleId") String roleId, @Param("positionId") String positionId);

  /**
   * 解绑某一角色与某一职位关系
   *
   * @param roleId
   * @param positionId
   */
  @Modifying
  @Query(
      value =
          "delete from engine_position_role_mapping where role_id = :roleId and position_id = :positionId",
      nativeQuery = true)
  void unbindRole(@Param("roleId") String roleId, @Param("positionId") String positionId);

  /**
   * 解绑某一角色的所有职位绑定关系
   *
   * @param roleId
   */
  @Modifying
  @Query(
      value = "delete from engine_position_role_mapping where role_id = :roleId",
      nativeQuery = true)
  void unbindRoles(@Param("roleId") String roleId);

  /**
   * 查询某一用户与某一职位绑定关系
   * @param userId
   * @param positionId
   * @return
   */
  @Query(value = "SELECT COUNT(*) FROM engine_position_user_mapping WHERE user_id = :userId AND position_id = :positionId",nativeQuery = true)
  int countByUserAndPosition(@Param("userId") String userId, @Param("positionId") String positionId);

  /**
   * 绑定某一用户与某一职位关系
   *
   * @param userId
   * @param positionId
   */
  @Modifying
  @Query(
      value =
          "insert into engine_position_user_mapping(user_id,position_id) values (:userId,:positionId)",
      nativeQuery = true)
  void binduser(@Param("userId") String userId, @Param("positionId") String positionId);

  /**
   * 解绑某一用户与某一职位关系
   *
   * @param userId
   * @param positionId
   */
  @Modifying
  @Query(
      value =
          "delete from engine_position_user_mapping where user_id = :userId and position_id = :positionId",
      nativeQuery = true)
  void unbinduser(@Param("userId") String userId, @Param("positionId") String positionId);

  /**
   * 解绑某一用户的所有职位绑定关系
   *
   * @param userId
   */
  @Modifying
  @Query(
      value = "delete from engine_position_user_mapping where user_id = :userId",
      nativeQuery = true)
  void unbindusers(@Param("userId") String userId);

  /**
   * 查询多个岗位信息
   * @param ids
   * @return
   */
  @Query("from PositionEntity m where m.id in :ids")
  public Set<PositionEntity> findByIds(@Param("ids") List<String> ids);
}
