package com.bizunited.platform.core.repository.dataview;

import com.alibaba.fastjson.JSONArray;
import com.alibaba.fastjson.JSONObject;
import com.bizunited.platform.core.entity.DataViewEntity;
import com.bizunited.platform.core.entity.DataViewFieldEntity;
import com.bizunited.platform.core.entity.DataViewSystemEntity;
import com.bizunited.platform.core.repository.dynamic.DynamicDataSourceManager;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.hibernate.SessionFactory;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.PageImpl;
import org.springframework.data.domain.Pageable;
import org.springframework.orm.hibernate5.SessionFactoryUtils;
import org.springframework.stereotype.Repository;
import org.springframework.util.CollectionUtils;
import javax.sql.DataSource;
import java.sql.Connection;
import java.sql.DatabaseMetaData;
import java.sql.ResultSet;
import java.sql.ResultSetMetaData;
import java.sql.SQLException;
import java.sql.Statement;
import java.util.ArrayList;
import java.util.HashSet;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Set;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import java.util.stream.Collectors;

/**
 * 第三方数据视图数据查询
 */
@Repository("DataViewThirdDataSourceRepositoryImpl")
public class DataViewThirdDataSourceRepositoryImpl implements DataViewThirdDataSourceRepository {
  private static Logger LOGGER = LoggerFactory.getLogger(DataViewThirdDataSourceRepositoryImpl.class);
  @Autowired
  private DynamicDataSourceManager dynamicDataSourceManager;
  private static final String ERROR_SESSION_FACTORY = "未能获取到第三方的sessionFactory信息，请检查!!";
  private static final String ERROR_SQL = "SQL执行异常，请检查!!详情：";
  private static final String SYSTEM_PARAMS_PATTERN = "\\{\\:[a-zA-Z]{1}\\w*\\}";

  /**
   * 执行第三方数据视图预查询，至多只返回一条数据
   */
  @Override
  public JSONArray executeSQLTop1(String dataSourceCode , String sql) {
    SessionFactory sessionFactory = this.dynamicDataSourceManager.getCurrentSessionFactory(dataSourceCode);
    Validate.notNull(sessionFactory, ERROR_SESSION_FACTORY);
    JSONArray jsonArr = new JSONArray();
    try(Connection conn = SessionFactoryUtils.getDataSource(sessionFactory).getConnection();
        Statement st = conn.createStatement();
        ResultSet results = st.executeQuery(sql);){
      ResultSetMetaData  meta = null;
      //只取第一条数据
      if(results.first()) {
        meta = results.getMetaData();
        JSONObject jsonObj = new JSONObject();
        for(int i = 1 ; i <= meta.getColumnCount() ; i++) {
          jsonObj.put(meta.getColumnName(i), results.getObject(i));
        }
        jsonArr.add(jsonObj);
        LOGGER.debug("sql执行结果jsonArr = " + jsonArr.toJSONString());
      }
    }catch (Exception e) {
      throw new IllegalArgumentException(ERROR_SQL + e.getMessage());
    }
    return jsonArr;
  }


  /**
   * 执行第三方数据视图查询
   */
  @Override
  public JSONArray executeSQL(String dataSourceCode, String sql) {
    SessionFactory sessionFactory = this.dynamicDataSourceManager.getCurrentSessionFactory(dataSourceCode);
    Validate.notNull(sessionFactory, ERROR_SESSION_FACTORY);
    JSONArray jsonArr = new JSONArray();
    try(Connection conn = SessionFactoryUtils.getDataSource(sessionFactory).getConnection();
        Statement st = conn.createStatement();
        ResultSet results = st.executeQuery(sql);) {
      ResultSetMetaData  meta = null;
      while(results.next()) {
        meta = results.getMetaData();
        JSONObject jsonObj = new JSONObject();
        for(int i = 1 ; i <= meta.getColumnCount() ; i++) {
          jsonObj.put(meta.getColumnName(i), results.getObject(i));
        }
        jsonArr.add(jsonObj);
      }
      LOGGER.debug("sql执行结果jsonArr = " + jsonArr.toJSONString());
    }catch (Exception e) {
      throw new IllegalArgumentException(ERROR_SQL + e.getMessage());
    }
    return jsonArr;
  }

  @Override
  public DataViewEntity executeResource(String dataSourceCode, DataViewEntity dataView) {
    Validate.notNull(dataView,"数据视图不能为空！");
    Validate.notBlank(dataView.getSourceSql(), "原始SQL不能为空！");
    SessionFactory sessionFactory = this.dynamicDataSourceManager.getCurrentSessionFactory(dataSourceCode);
    Validate.notNull(sessionFactory,"未找到该数据源！");
    String sourceSql = dataView.getSourceSql();
    Set<DataViewFieldEntity> fields = new LinkedHashSet<>();
    DataSource dataSource = SessionFactoryUtils.getDataSource(sessionFactory);
    DataViewEntity currentDataView = this.replaceSysParam(dataView);
    try(Connection connection = dataSource.getConnection();
        Statement statement = connection.createStatement();
        ResultSet result = statement.executeQuery(currentDataView.getSourceSql())) {
      ResultSetMetaData metaData = result.getMetaData();
      //遍历结果集中每一个字段，得到字段信息并存入fields
      Set<String> fieldNames = new HashSet<>();
      //用户参数与系统参数也不能重名
      Set<String> sysParamsStrs = currentDataView.getSystemFilters().stream().map(o -> o.getParamName()).collect(Collectors.toSet());
      fieldNames.addAll(sysParamsStrs);
      for(int i=1;i<=metaData.getColumnCount();i++){
        DataViewFieldEntity field = new DataViewFieldEntity();
        field.setFieldName(metaData.getColumnLabel(i));
        //如果查询出的字段集中含有相同的字段名，那么报错提醒用户
        Validate.isTrue(fieldNames.add(metaData.getColumnLabel(i)),"返回字段集或与系统参数中含有相同字段名：%s,请为其取别名，或做其他调整。" , metaData.getColumnLabel(i));
        field.setFieldType(metaData.getColumnClassName(i));
        field.setPhysical(!StringUtils.isEmpty(metaData.getTableName(i)));
        field.setSchemaName(metaData.getSchemaName(i));
        field.setTargetTable(metaData.getTableName(i));
        field.setTargetField(metaData.getColumnName(i));
        field.setSortIndex(i);
        fields.add(field);
      }
      fieldNames.clear();
      currentDataView.setFields(fields);
    } catch (SQLException e) {
      throw new IllegalArgumentException(ERROR_SQL+e.getMessage());
    }
    currentDataView.setSourceSql(sourceSql);
    return currentDataView;
  }

  /**
   * 分析SQL，获取其中的系统参数，并将系统参数值变为空字符串
   * 替换查询SQL中的系统参数,
   * @param dataView
   * @return
   */
  private DataViewEntity replaceSysParam(DataViewEntity  dataView){
    Validate.notNull(dataView,"数据视图不能为空！");
    Validate.notBlank(dataView.getSourceSql(), "原始SQL不能为空！");
    String sql = dataView.getSourceSql();
    Set<DataViewSystemEntity> systemEntities = new LinkedHashSet<>();
    Set<String> sysParams = this.extractMessageByRegular(sql);
    if(!CollectionUtils.isEmpty(sysParams)){
      for(String param : sysParams){
        //取出系统参数字段
        DataViewSystemEntity dataViewSystem = new DataViewSystemEntity();
        dataViewSystem.setParamName(param);
        systemEntities.add(dataViewSystem);
        //将原SQL字段设置为空
        String replaceStr = new StringBuilder().append("{:").append(param).append("}").toString();
        sql = sql.replace(replaceStr,"''");
      }
    }
    dataView.setSystemFilters(systemEntities);
    dataView.setSourceSql(sql);
    return dataView;
  }


  /**
   * 正则获取特定格式的系统参数
   * @param msg
   * @return
   */
  private Set<String> extractMessageByRegular(String msg){
    Set<String> list=new LinkedHashSet<>();
    Pattern p = Pattern.compile(SYSTEM_PARAMS_PATTERN);
    Matcher m = p.matcher(msg);
    while(m.find()){
      //截取符合规则的系统参数字符串
      list.add(m.group().substring(2, m.group().length()-1));
    }
    return list;
  }


  /**
   * 执行第三方视图查询带分页
   */
  @Override
  public Page<JSONObject> executeSQLPageAble(String dataSourceCode, String sql, Pageable pageable) {
    SessionFactory sessionFactory = this.dynamicDataSourceManager.getCurrentSessionFactory(dataSourceCode);
    Validate.notNull(sessionFactory, ERROR_SESSION_FACTORY);
    Page<JSONObject> page = null;
    List<JSONObject> list = new ArrayList<>();
    try(Connection conn = SessionFactoryUtils.getDataSource(sessionFactory).getConnection();
        Statement st = conn.createStatement();
        ResultSet results = st.executeQuery(sql);){
      ResultSetMetaData  meta = null;
      while(results.next()) {
        meta = results.getMetaData();
        JSONObject jsonObj = new JSONObject();
        for(int i = 1 ; i <= meta.getColumnCount() ; i++) {
          jsonObj.put(meta.getColumnName(i), results.getObject(i));
        }
        list.add(jsonObj);
      }
      page = new PageImpl<>(list, pageable, list.size());
      LOGGER.debug("sql执行结果page = " + JSONObject.toJSONString(page));
    }catch (Exception e) {
      throw new IllegalArgumentException(ERROR_SQL+e.getMessage());
    }
    return page;
  }
  
  /**
   * 检查第三方数据源中基础表是否存在
   */
  @Override
  public void checkTable(String dataSourceCode, String tableName) {
    Validate.notBlank(tableName,"输入数据表名不能为空，请检查！");
    Validate.notBlank(dataSourceCode,"输入第三方数据源编码不能为空，请检查！");
    SessionFactory sessionFactory = this.dynamicDataSourceManager.getCurrentSessionFactory(dataSourceCode);
    try (Connection connection = SessionFactoryUtils.getDataSource(sessionFactory).getConnection();) {
      DatabaseMetaData md = connection.getMetaData();
      try(ResultSet rs = md.getTables(null, null, tableName, null);){
        Validate.isTrue(rs.next(),"所设置第三方数据源中基础表不存在，请检查！");
      }
    } catch (Exception e){
      throw new IllegalArgumentException(ERROR_SQL+e.getMessage());
    }
  }
}
