package com.bizunited.platform.core.repository.dataview.analysis;

import com.bizunited.platform.core.common.enums.SQLCorrelationEnum;
import com.bizunited.platform.core.entity.DataViewAuthVerticalEntity;
import com.bizunited.platform.core.entity.DataViewAuthVerticalRelationEntity;
import com.bizunited.platform.core.entity.DataViewFieldEntity;
import com.bizunited.platform.rbac.server.vo.RoleVo;
import com.bizunited.platform.rbac.server.vo.UserGroupVo;
import com.google.common.collect.Sets;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.util.CollectionUtils;

import java.util.HashMap;
import java.util.HashSet;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;

/**
 * 用于数据视图之纵向数据权限条件的分析以及构造SQL工具类
 */
class SQLAuthVerticalAnalysis {
  /**
   * 通过私有构造函数，让该类暂时不对外开发
   * 因为后期可能会对该类进行重新设计
   */
  private SQLAuthVerticalAnalysis() {
  }

  /**
   * 纵向数据权限 参数筛选条件的分析及构造
   * 默认Prestatement的构建方式(?传参)
   * 注意：使用该类前，需要把显示字段的多对多关系数据 和 经过处理后的系统参数SQL(如果没有，就是视图的原始SQL) 组装好
   *
   * @param auths
   * @param map
   * @param sql   可能是外部已经处理过的SQL，比如经过了系统参数筛选条件处理的SQL
   * @return
   */
  @SuppressWarnings("unchecked")
  static void buildSQLAuthVertical(Set<DataViewAuthVerticalEntity> auths, String sql, Map<SQLCorrelationEnum, Object> map, Set<DataViewFieldEntity> allFileds) {
    if (map == null) {
      map = new HashMap<>();
    }

    if (CollectionUtils.isEmpty(auths)) {
      map.put(SQLCorrelationEnum.RESULT_SQL, String.format("select %s from (%s) %s where 1=1 ", "*", sql, Constants.ALIAS_STRING));
      return;
    }

    //开始处理数据视图的纵向权限逻辑
    Set<String> finalFieldNames = new HashSet<>();
    Map<Integer, Object> presets = (Map<Integer, Object>) map.get(SQLCorrelationEnum.VERTICAL_PRESETS);
    Map<Integer, Set<String>> verticalDispalyFields = new HashMap<>();
    for (int i = 1; i <= Constants.AUTH_TYPE_COUNT; i++) {
      final Integer authType = i;
      //获取相同权限类型的数据
      Set<DataViewAuthVerticalEntity> sameAuths = auths.stream().filter(e -> authType.compareTo(e.getAuthType()) == 0).collect(Collectors.toSet());
      if (CollectionUtils.isEmpty(sameAuths)) {
        continue;
      }
      processAuths(sameAuths, presets, i, finalFieldNames, verticalDispalyFields, allFileds);
    }
    processUnion(verticalDispalyFields, map);
    Set<String> displayFields = (HashSet<String>) map.get(SQLCorrelationEnum.DISPLAY_FIELDS);
    String verticalFields = String.join(",", displayFields);
    map.put(SQLCorrelationEnum.RESULT_SQL, String.format("select %s from (%s) %s where 1=1 ", verticalFields, sql, Constants.ALIAS_STRING));
  }


  private static void processUnion(Map<Integer, Set<String>> verticalDispalyFields, Map<SQLCorrelationEnum, Object> map) {
    Set<String> userSets = verticalDispalyFields.get(1) == null ? new HashSet<>() : verticalDispalyFields.get(1);
    Set<String> roleSets = verticalDispalyFields.get(2) == null ? new HashSet<>() : verticalDispalyFields.get(2);
    Set<String> positionSets = verticalDispalyFields.get(3) == null ? new HashSet<>() : verticalDispalyFields.get(3);
    Set<String> groupSets = verticalDispalyFields.get(4) == null ? new HashSet<>() : verticalDispalyFields.get(4);
    Set<String> orgSets = verticalDispalyFields.get(5) == null ? new HashSet<>() : verticalDispalyFields.get(5);

    Sets.SetView<String> v1 = Sets.union(roleSets, userSets);
    Sets.SetView<String> v2 = Sets.union(positionSets, groupSets);
    Sets.SetView<String> v3 = Sets.union(v1.copyInto(new HashSet<>()), v2.copyInto(new HashSet<>()));
    Sets.SetView<String> v4 = Sets.union(orgSets, v3.copyInto(new HashSet<>()));
    map.put(SQLCorrelationEnum.DISPLAY_FIELDS, v4.copyInto(new HashSet<>()));
  }


  @SuppressWarnings("unchecked")
  private static void processAuths(Set<DataViewAuthVerticalEntity> sameAuths, Map<Integer, Object> presets, Integer index, Set<String> finalFieldNames, Map<Integer, Set<String>> verticalDispalyFields, Set<DataViewFieldEntity> allFileds) {
    for (DataViewAuthVerticalEntity e : sameAuths) {
      Set<DataViewFieldEntity> displayFields = e.getDisplayFields();
      Validate.notNull(displayFields, "显示字段不能为空！！");
      Validate.isTrue(!CollectionUtils.isEmpty(displayFields), "纵向权限中没有指定显示字段信息，请检查!!");
      Set<DataViewAuthVerticalRelationEntity> relations = e.getAuthRelations();
      Validate.notNull(relations, "权限领域不能为空！！");
      Validate.isTrue(!CollectionUtils.isEmpty(relations), "纵向权限中没有指定权限领域信息，请检查!!");

      Object o = presets.get(index);
      Validate.isTrue(o != null, "纵向权限处理时，没有获取到当前预制信息！！");
      switch (index) {
        case 1://用户
        case 3://岗位
        case 5://组织机构
          Boolean flag1 = relations.stream().anyMatch(x -> StringUtils.equals(x.getAuthRelationIds(), o.toString()));
          if (flag1) {
            finalFieldNames.addAll(displayFields.stream().map(DataViewFieldEntity::getFieldName).collect(Collectors.toSet()));
          }
          break;
        case 2://角色
          Set<RoleVo> roles = (HashSet<RoleVo>) o;
          Set<String> roleStrs = roles.stream().map(RoleVo::getId).collect(Collectors.toSet());
          Set<String> relationStrs = relations.stream().map(DataViewAuthVerticalRelationEntity::getId).collect(Collectors.toSet());
          Sets.SetView<String> roleSets = Sets.intersection(roleStrs, relationStrs);
          Boolean flag2 = CollectionUtils.isEmpty(roleSets);
          if (flag2) {
            finalFieldNames.addAll(displayFields.stream().map(DataViewFieldEntity::getFieldName).collect(Collectors.toSet()));
          }
          break;
        case 4://用户组
          Set<UserGroupVo> groups = (HashSet<UserGroupVo>) o;
          Set<String> groupStrs = groups.stream().map(UserGroupVo::getId).collect(Collectors.toSet());
          Set<String> grelationStrs = relations.stream().map(DataViewAuthVerticalRelationEntity::getId).collect(Collectors.toSet());
          Sets.SetView<String> groupSets = Sets.intersection(groupStrs, grelationStrs);
          Boolean flag3 = CollectionUtils.isEmpty(groupSets);
          if (flag3) {
            finalFieldNames.addAll(displayFields.stream().map(DataViewFieldEntity::getFieldName).collect(Collectors.toSet()));
          }
          break;
        default:
          break;
      }
    }

    if (CollectionUtils.isEmpty(finalFieldNames)) {
      finalFieldNames = allFileds.stream().map(DataViewFieldEntity::getFieldName).collect(Collectors.toSet());
    }
    verticalDispalyFields.put(index, finalFieldNames);
  }
}
