package com.bizunited.platform.core.repository.dataview.analysis;

import com.bizunited.platform.core.common.enums.SQLCorrelationEnum;
import com.bizunited.platform.core.common.utils.ParamsAnalysisUtil;
import com.bizunited.platform.core.entity.DataViewFilterEntity;
import com.bizunited.platform.core.service.dataview.model.SQLParamModel;
import com.bizunited.platform.core.service.invoke.model.InvokeOperations;
import com.bizunited.platform.core.service.invoke.model.InvokeParams;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.cglib.beans.BeanCopier;
import org.springframework.util.CollectionUtils;

import java.util.LinkedHashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;

/**
 * 用于数据视图filter用户筛选条件的分析以及构造SQL工具类
 */
class SQLConditionFilterAnalysis {
  /**
   * 通过私有构造函数，让该类暂时不对外开发
   * 因为后期可能会对该类进行重新设计
   */
  private SQLConditionFilterAnalysis() {
  }

  /**
   * 数据视图filter筛选条件的分析以及构造SQL
   * 默认Prestatement的构建方式(?传参)
   *
   * @param filters
   * @param params
   * @param mysqlAnalysis
   * @return
   */
  static Map<SQLCorrelationEnum, Object> buildSQLConditions(Set<DataViewFilterEntity> filters, Map<SQLCorrelationEnum, Object> map, InvokeParams params, AbstractSqlAnalysis sqlAnalysis) {
    Validate.isTrue(!CollectionUtils.isEmpty(map), "SQL分析时，上下文信息不能为空，请检查!!");
    if (CollectionUtils.isEmpty(filters)) {
      return map;
    }

    Set<DataViewFilterEntity> conditionSets = filters.stream().filter(e -> params.containsKey(e.getField().getFieldName())).collect(Collectors.toSet());
    if (CollectionUtils.isEmpty(conditionSets)) {
      return map;
    }
    //开始处理数据视图的filter筛选条件
    for (DataViewFilterEntity filter : conditionSets) {
      String fieldName = filter.getField().getFieldName();
      InvokeOperations model = params.get(fieldName);
      model.setOprtType(filter.getOpType());
      model.setParamName(fieldName);
      analysisConditions(model, map, sqlAnalysis);
    }
    return map;
  }

  /**
   * 解析的条件包括where字段拼接
   *
   * @param model
   * @param map
   */
  private static void analysisConditions(InvokeOperations model, Map<SQLCorrelationEnum, Object> map, AbstractSqlAnalysis sqlAnalysis) {
    Object compareValue = model.getCompareValue();
    Validate.isTrue(!(compareValue == null && StringUtils.isBlank(model.getOprtType())), "%s参数值不能为空或不是排序字段", model.getParamName());
    if (compareValue == null) {
      return;
    }

    //先处理where字段拼接的情况
    if (InvokeOperations.isLikeOprt(model.getOprtType())) {//like操作
      analysisLikeConditions(model, map, sqlAnalysis);
    } else if (InvokeOperations.isBetweenOprt(model.getOprtType())) {//between操作
      analysisBetweenConditions(model, map);
    } else if (InvokeOperations.isInOprt(model.getOprtType())) {//in操作
      analysisInConditions(model, map);
    } else if (InvokeOperations.nullOprt(model.getOprtType())) {//is null操作
      analysisNullConditions(model, map);
    } else if (InvokeOperations.isRegionOprt(model.getOprtType())) {//区间范围操作 如：大于且小于
      analysisRegionConditions(model, map);
    } else {//等于 不等于 小于 大于 大于等于 小于等于
      analysisOtherConditions(model, map);
    }
  }


  /**
   * 存储各个解析步骤中的参数值
   * 用户筛选条件的参数传入方式默认为1，外部传入
   *
   * @param model
   * @param map
   */
  @SuppressWarnings("unchecked")
  private static void addConditionValue(InvokeOperations model, Map<SQLCorrelationEnum, Object> map) {
    Set<SQLParamModel> values = (LinkedHashSet<SQLParamModel>) map.get(SQLCorrelationEnum.CONDITION_VALUES);
    SQLParamModel spm = new SQLParamModel();
    spm.setParamName(model.getParamName());
    spm.setValue(model.getCompareValue());
    spm.setMappingType(model.getMappingType());
    spm.setTransferType(1);//外部传入
    //这里设置参数的下标计数
    setCouter(spm, map);
    values.add(spm);
  }

  private static void setCouter(SQLParamModel spm, Map<SQLCorrelationEnum, Object> map) {
    int count = (int) map.remove(SQLCorrelationEnum.COUNTER);
    spm.setIndex(count++);
    map.put(SQLCorrelationEnum.COUNTER, count);
  }

  /**
   * 存储各个解析步骤中返回的sql
   *
   * @param conditionSQL
   * @param map
   */
  private static void addConditionSQL(String conditionSQL, Map<SQLCorrelationEnum, Object> map) {
    String oldVal = map.remove(SQLCorrelationEnum.RESULT_SQL).toString();
    String newVal = oldVal + conditionSQL;
    map.put(SQLCorrelationEnum.RESULT_SQL, newVal);
  }


  /**
   * 解析Like操作 , 并存储参数值
   *
   * @param model
   * @param map
   */
  private static void analysisLikeConditions(InvokeOperations model, Map<SQLCorrelationEnum, Object> map, AbstractSqlAnalysis sqlAnalysis) {
    String sql = Constants.EMPTY_CHAR;
    String likeCondition;
    switch (model.getOprtType()) {
      case "LIKE":
        likeCondition = sqlAnalysis.concatSql("'%%'", Constants.PLACE_HOLDER, "'%%'");
        sql = String.format(" and %s.%s like %s", Constants.ALIAS_STRING, model.getParamName(), likeCondition);
        break;
      case "LLIKE":
        likeCondition = sqlAnalysis.concatSql("'%%'", Constants.PLACE_HOLDER);
        sql = String.format(" and %s.%s like %s", Constants.ALIAS_STRING, model.getParamName(), likeCondition);
        break;
      case "RLIKE":
        likeCondition = sqlAnalysis.concatSql(Constants.PLACE_HOLDER, "'%%'");
        sql = String.format(" and %s.%s like %s", Constants.ALIAS_STRING, model.getParamName(), likeCondition);
        break;
      default:
        break;
    }
    addConditionSQL(sql, map);
    addConditionValue(model, map);
  }


  /**
   * 解析IN 或 NOT IN 操作 , 并存储参数值
   *
   * @param model
   * @param map
   */
  @SuppressWarnings("unchecked")
  private static void analysisInConditions(InvokeOperations model, Map<SQLCorrelationEnum, Object> map) {
    List<Object> list = (List<Object>) model.getCompareValue();
    Validate.notNull(list, "前端传入的数组值不能为空");
    Validate.isTrue(list.size() >= 1, "前端传入的数组值有误");
    StringBuilder inPatternBuilder = new StringBuilder(Constants.EMPTY_CHAR);
    for (int i = 0; i < list.size(); i++) {
      inPatternBuilder.append(Constants.PLACE_HOLDER + ",");
    }
    String inPattern = inPatternBuilder.toString();
    String sql = " and " + Constants.ALIAS_STRING + "." + model.getParamName() + " " + getOprtTypeValue(model.getOprtType()) + "(" + StringUtils.left(inPattern, inPattern.length() - 1) + ")";
    addConditionSQL(sql, map);
    addConditionArray(model, map, list.size());
  }


  /**
   * 解析Between操作 , 并存储参数值
   *
   * @param model
   * @param map
   */
  private static void analysisBetweenConditions(InvokeOperations model, Map<SQLCorrelationEnum, Object> map) {
    String sql = String.format(" and %s.%s between %s and %s", Constants.ALIAS_STRING, model.getParamName(), Constants.PLACE_HOLDER, Constants.PLACE_HOLDER);
    addConditionSQL(sql, map);
    addConditionArray(model, map, 2);
  }

  /**
   * 解析IS NULL或IS NOT NULL操作
   *
   * @param model
   * @param map
   */
  private static void analysisNullConditions(InvokeOperations model, Map<SQLCorrelationEnum, Object> map) {
    String sql = String.format(" and %s.%s %s", Constants.ALIAS_STRING, model.getParamName(), getOprtTypeValue(model.getOprtType()));
    addConditionSQL(sql, map);
  }

  /**
   * 解析区间范围操作 , 并存储参数值
   *
   * @param model
   * @param map
   */
  private static void analysisRegionConditions(InvokeOperations model, Map<SQLCorrelationEnum, Object> map) {
    String sql = Constants.EMPTY_CHAR;
    switch (model.getOprtType()) {
      case "LC_RC":
        sql = String.format(" and %s.%s >= %s and %s.%s <= %s", Constants.ALIAS_STRING, model.getParamName(),
            Constants.PLACE_HOLDER, Constants.ALIAS_STRING, model.getParamName(),
            Constants.PLACE_HOLDER);
        break;
      case "LO_RO":
        sql = String.format(" and %s.%s > %s and %s.%s < %s", Constants.ALIAS_STRING, model.getParamName(),
            Constants.PLACE_HOLDER, Constants.ALIAS_STRING, model.getParamName(),
            Constants.PLACE_HOLDER);
        break;
      case "LC_RO":
        sql = String.format(" and %s.%s >= %s and %s.%s < %s", Constants.ALIAS_STRING, model.getParamName(),
            Constants.PLACE_HOLDER, Constants.ALIAS_STRING, model.getParamName(),
            Constants.PLACE_HOLDER);
        break;
      case "LO_RC":
        sql = String.format(" and %s.%s > %s and %s.%s <= %s", Constants.ALIAS_STRING, model.getParamName(),
            Constants.PLACE_HOLDER, Constants.ALIAS_STRING, model.getParamName(),
            Constants.PLACE_HOLDER);
        break;
      default:
        break;
    }
    addConditionSQL(sql, map);
    addConditionArray(model, map, 2);
  }


  /**
   * 解析其他情况的操作 , 并存储参数值
   * 包括：等于 不等于 大于 小于 大于等于 小于等于
   *
   * @param model
   * @param map
   */
  private static void analysisOtherConditions(InvokeOperations model, Map<SQLCorrelationEnum, Object> map) {
    String sql = String.format(" and %s.%s %s %s", Constants.ALIAS_STRING, model.getParamName(), getOprtTypeValue(model.getOprtType()), Constants.PLACE_HOLDER);
    addConditionSQL(sql, map);
    addConditionValue(model, map);
  }


  /**
   * 根据情况拆解前端传入的数组值
   * 注：In操作数组长度最小为1
   * 区间范围操作和between操作，数组固定长度为2
   *
   * @param model
   * @param map
   * @param arrLength
   */
  @SuppressWarnings("unchecked")
  private static void addConditionArray(InvokeOperations model, Map<SQLCorrelationEnum, Object> map, int arrLength) {
    //因为前端传入的model中，compareValue值为数组，这里需要拆解
    //注意：需要根据model中的mappingType判断类型
    Validate.isTrue(SQLParamModel.isSupportBaseClassType(model.getMappingType()), "%s传入的mappingType值不是Nebula支持的类型", model.getMappingType());
    List<Object> list = (List<Object>) model.getCompareValue();
    Validate.notNull(list, "传入数组不能为空");
    Validate.isTrue(list.size() == arrLength, "传入的数组值有错误");
    for (int index = 0; index < list.size(); index++) {
      InvokeOperations kipm = new InvokeOperations();
      BeanCopier b = BeanCopier.create(InvokeOperations.class, InvokeOperations.class, false);
      b.copy(model, kipm, null);
      kipm.setCompareValue(list.get(index));
      matchBetween(kipm, index);
      addConditionValue(kipm, map);
    }
  }

  private static String getOprtTypeValue(String oprtType) {
    return InvokeParams.OprtCharType.getOprtValue(oprtType);
  }


  private static void matchBetween(InvokeOperations model, int index) {
    if (!InvokeOperations.isBetweenOprt(model.getOprtType())) {
      return;
    }
    Object value = ParamsAnalysisUtil.doTrans(model.getCompareValue(), model.getMappingType()).toString();;
    if(model.getMappingType().equals("java.util.Date")) {
      if (index == 0) {
        value += " 00:00:00";
      } else {
        value += " 23:59:59";
      }
      model.setMappingType("java.lang.String");
    } else {
      model.setMappingType(model.getMappingType());
    }
    model.setCompareValue(value);
  }
}
