package com.bizunited.platform.core.repository.dataview.analysis;

import com.bizunited.platform.core.common.constant.DataViewSystemParamKey;
import com.bizunited.platform.core.entity.DataViewAuthHorizontalEntity;
import com.bizunited.platform.core.entity.DataViewAuthVerticalEntity;
import com.bizunited.platform.core.entity.DataViewSystemEntity;
import com.bizunited.platform.rbac.server.service.OrganizationService;
import com.bizunited.platform.rbac.server.service.PositionService;
import com.bizunited.platform.rbac.server.service.RoleService;
import com.bizunited.platform.rbac.server.service.UserGroupService;
import com.bizunited.platform.rbac.server.service.UserService;
import com.bizunited.platform.rbac.server.vo.OrganizationVo;
import com.bizunited.platform.rbac.server.vo.PositionVo;
import com.bizunited.platform.rbac.server.vo.RoleVo;
import com.bizunited.platform.rbac.server.vo.UserGroupVo;
import com.bizunited.platform.rbac.server.vo.UserVo;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;
import org.springframework.util.CollectionUtils;

import java.security.Principal;
import java.util.Date;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;

@Component("SQLPresetValueAnalysis")
public class SQLPresetValueAnalysis {

  @Autowired
  private UserService userService;
  @Autowired
  private PositionService positionService;
  @Autowired
  private UserGroupService userGroupService;
  @Autowired
  private OrganizationService organizationService;
  @Autowired
  private RoleService roleService;
  private static final String ANALYSIS_ERROR = "系统参数分析时，Principal信息不能为空，请检查!!";
  private static final String ANALYSIS_H_ERROR = "横向权限分析时，Principal信息不能为空，请检查!!";
  private static final String ANALYSIS_V_ERROR = "纵向权限分析时，Principal信息不能为空，请检查!!";
  private static final String ANALYSIS_P_ERROR = "预置信息分析时，Principal信息不能为空，请检查!!";

  /**
   * 系统参数筛选时，获取指定的预制标识值的预制信息
   *
   * @param systems
   * @param principal
   * @return
   */
  public Map<String, Object> systemPresetValues(Set<DataViewSystemEntity> systems, Principal principal) {
    if (CollectionUtils.isEmpty(systems)) {
      return null;
    }
    Validate.notNull(principal, ANALYSIS_ERROR);
    Validate.notBlank(principal.getName(), ANALYSIS_ERROR);
    Map<String, Object> presets = new HashMap<>();
    Set<DataViewSystemEntity> presetSystems = systems.stream().filter(e -> e.getParamSourceType() == 3).collect(Collectors.toSet());
    for (DataViewSystemEntity element : presetSystems) {
      //当系统参数设置为预制信息的方式时，需要遍历预制标识信息ParamValue
      Object o = this.getPresetValueByParamKey(element.getParamValue(), principal);
      String key = element.getParamName() + "|DataViewSystemEntity";
      presets.put(key, o);
    }
    return presets;
  }


  /**
   * 横向权限筛选时，获取指定的预制标识值的预制信息
   *
   * @param authHorizontals
   * @param principal
   * @return
   */
  public Map<String, Object> authHorizontalPresetValues(Set<DataViewAuthHorizontalEntity> authHorizontals, Principal principal) {
    if (CollectionUtils.isEmpty(authHorizontals)) {
      return null;
    }
    Validate.notNull(principal, ANALYSIS_H_ERROR);
    Validate.notBlank(principal.getName(), ANALYSIS_H_ERROR);
    Map<String, Object> presets = new HashMap<>();
    Set<DataViewAuthHorizontalEntity> presetAuths = authHorizontals.stream().filter(e -> e.getParamSourceType() == 3).collect(Collectors.toSet());
    for (DataViewAuthHorizontalEntity auth : presetAuths) {
      //当系统参数设置为预制信息的方式时，需要遍历预制标识信息ParamValue
      if (auth.getParamSourceType() == 3) {
        Object o = this.getPresetValueByParamKey(auth.getParamKey(), principal);
        String key = auth.getField().getFieldName() + "|DataViewAuthHorizontalEntity";
        presets.put(key, o);
      }
    }
    return presets;
  }

  /**
   * 纵向权限时，获取当前人信息
   *
   * @param authVerticals
   * @param principal
   * @return
   */
  public Map<Integer, Object> authVerticalPresetValues(Set<DataViewAuthVerticalEntity> authVerticals, Principal principal) {
    if (CollectionUtils.isEmpty(authVerticals)) {
      return null;
    }
    Validate.notNull(principal, ANALYSIS_V_ERROR);
    Validate.notBlank(principal.getName(), ANALYSIS_V_ERROR);
    Map<Integer, Object> presets = new HashMap<>();
    for (int i = 1; i <= Constants.AUTH_TYPE_COUNT; i++) {
      final Integer authType = i;
      //获取相同权限类型的数据
      Set<DataViewAuthVerticalEntity> sameAuths = authVerticals.stream().filter(e -> authType.compareTo(e.getAuthType()) == 0).collect(Collectors.toSet());
      if (CollectionUtils.isEmpty(sameAuths)) {
        continue;
      }
      Integer key = i;
      Object value = this.getPresetValueByAuthType(principal, i);
      presets.put(key, value);
    }
    return presets;
  }


  /**
   * 用于查询固定值情况，纵向才会使用
   * 因为纵向的本身就是当前用户人
   */
  private Object getPresetValueByAuthType(Principal principal, int authType) {
    Validate.notNull(principal, ANALYSIS_P_ERROR);
    Validate.notBlank(principal.getName(), ANALYSIS_P_ERROR);
    String account = principal.getName();
    Validate.notBlank(account, "未能获取到用户的账户信息，请检查!!");
    UserVo currentUser = this.userService.findByAccount(account);
    Validate.notNull(currentUser, "非法的用户信息，请检查!!");
    switch (authType) {
      case 1://用户
        return currentUser.getId();
      case 2://角色
        List<RoleVo> roles = this.roleService.findAllByUserId(currentUser.getId());
        Validate.isTrue(!CollectionUtils.isEmpty(roles), "当前用户没有角色信息！！");
        return new HashSet<>(roles);
      case 3://岗位
        PositionVo p1 = this.positionService.findByUserId(currentUser.getId());
        Validate.isTrue(p1 != null, "当前用户没有岗位信息！！");
        return p1.getId();
      case 4://用户组
        Set<UserGroupVo> groups = userGroupService.findByUserId(currentUser.getId());
        Validate.isTrue(!CollectionUtils.isEmpty(groups), "当前用户没有用户组信息！！");
        return groups;
      case 5://组织机构
        OrganizationVo o1 = organizationService.findOrgByUserId(currentUser.getId());
        Validate.isTrue(o1 != null, "当前用户没有组织机构信息！！");
        return o1.getId();
      default:
        return null;
    }
  }

  /**
   * 根据预置值的标识信息，获取对应的预置值
   * 当控件设置了日期的预置值时，暂时忽略前端设置参数类型，统一返回日期Date类型，以备后续的逻辑处理
   */
  private Object getPresetValueByParamKey(String paramKey, Principal principal) {
    Validate.notNull(principal, ANALYSIS_P_ERROR);
    Validate.notBlank(principal.getName(), ANALYSIS_P_ERROR);
    Validate.notBlank(paramKey, "预置控件中，未能发现预置值标识信息，请检查!!");
    String account = principal.getName();
    Validate.notBlank(account, "未能获取到用户的账户信息，请检查!!");
    UserVo currentUser = this.userService.findByAccount(account);
    Validate.notNull(currentUser, "非法的用户信息，请检查!!");
    Object result = null;
    switch (paramKey) {
      //用户相关
      case DataViewSystemParamKey.OPT_USER_ID:
        result = currentUser.getId();
        break;
      case DataViewSystemParamKey.OPT_USER_NAME:
        result = currentUser.getUserName();
        break;
      case DataViewSystemParamKey.OPT_USER_ACCOUNT:
        result = currentUser.getAccount();
        break;
      case DataViewSystemParamKey.OPT_USER_CODE:
        result = currentUser.getId();
        break;
      case DataViewSystemParamKey.OPT_USER_PHONE:
        result = currentUser.getPhone();
        break;
      case DataViewSystemParamKey.OPT_USER_STATUS:
        result = currentUser.getUseStatus();
        break;
      case DataViewSystemParamKey.OPT_USER_IDCARD:
        result = currentUser.getIdcard();
        break;

      //岗位相关
      case DataViewSystemParamKey.OPT_USER_POSITION_ID:
      case DataViewSystemParamKey.OPT_USER_POSITION:
      case DataViewSystemParamKey.OPT_USER_POSITION_CODE:
      case DataViewSystemParamKey.OPT_USER_POSITION_STATUS:
        Map<String, Object> res = this.getPositionInfoByUserId(currentUser.getId());
        Validate.notEmpty(res, "预制信息处理时，没有获取到岗位信息，请检查!!");
        result = res.get(paramKey);
        break;

      //组织机构相关
      case DataViewSystemParamKey.OPT_USER_ORG_ID:
      case DataViewSystemParamKey.OPT_USER_ORG:
      case DataViewSystemParamKey.OPT_USER_ORG_TYPE:
      case DataViewSystemParamKey.OPT_USER_ORG_DESC:
      case DataViewSystemParamKey.OPT_USER_ORG_STATUS:
        Map<String, Object> res2 = this.getOrgInfoByUserId(currentUser.getId());
        Validate.notEmpty(res2, "预制信息处理时，没有获取到组织信息，请检查!!");
        result = res2.get(paramKey);
        break;

      //用户组相关
      case DataViewSystemParamKey.OPT_USER_GROUP:
        Set<UserGroupVo> groups = userGroupService.findByUserId(currentUser.getId());
        result = CollectionUtils.isEmpty(groups) ? null : StringUtils.join(groups.stream().map(UserGroupVo::getGroupName).collect(Collectors.toList()), ",");
        break;

      //用户角色相关
      case DataViewSystemParamKey.OPT_USER_ROLE:
        List<RoleVo> roles = this.roleService.findAllByUserId(currentUser.getId());
        result = CollectionUtils.isEmpty(roles) ? null : StringUtils.join(roles.stream().map(RoleVo::getRoleName).collect(Collectors.toList()), ",");
        break;


      //时间日期相关
      case DataViewSystemParamKey.CURRENT_USER_LOG_OUT_TIME:
        result = currentUser.getLastloginTime();
        break;
      case DataViewSystemParamKey.CURRENT_USER_LOGIN_TIME:
        result = currentUser.getEntryTime();
        break;
      case DataViewSystemParamKey.NOW_DATE_MONTH:
      case DataViewSystemParamKey.NOW_DATE_DAY:
      case DataViewSystemParamKey.NOW_DATE_MINUTE:
      case DataViewSystemParamKey.NOW_DATE_SECOND:
        result = new Date();
        break;
      default:
        break;
    }
    return result;
  }


  private Map<String, Object> getPositionInfoByUserId(String id) {
    Validate.notBlank("获取岗位信息时，传入的用户id不能为空，请检查!!");
    PositionVo p1 = this.positionService.findByUserId(id);
    Map<String, Object> result = new HashMap<>();
    result.put(DataViewSystemParamKey.OPT_USER_POSITION_ID, p1 == null ? null : p1.getId());
    result.put(DataViewSystemParamKey.OPT_USER_POSITION, p1 == null ? null : p1.getName());
    result.put(DataViewSystemParamKey.OPT_USER_POSITION_CODE, p1 == null ? null : p1.getCode());
    result.put(DataViewSystemParamKey.OPT_USER_POSITION_STATUS, p1 == null ? null : p1.getTstatus());
    return result;
  }

  private Map<String, Object> getOrgInfoByUserId(String id) {
    Validate.notBlank("获取组织信息时，传入的用户id不能为空，请检查!!");
    OrganizationVo o1 = organizationService.findOrgByUserId(id);
    Map<String, Object> result = new HashMap<>();
    result.put(DataViewSystemParamKey.OPT_USER_ORG_ID, o1 == null ? null : o1.getId());
    result.put(DataViewSystemParamKey.OPT_USER_ORG, o1 == null ? null : o1.getOrgName());
    result.put(DataViewSystemParamKey.OPT_USER_ORG_TYPE, o1 == null ? null : o1.getType());
    result.put(DataViewSystemParamKey.OPT_USER_ORG_DESC, o1 == null ? null : o1.getDescription());
    result.put(DataViewSystemParamKey.OPT_USER_POSITION_CODE, o1 == null ? null : o1.getCode());
    result.put(DataViewSystemParamKey.OPT_USER_POSITION_STATUS, o1 == null ? null : o1.getTstatus());
    return result;
  }
}
