package com.bizunited.platform.core.repository.dynamic;

import java.util.concurrent.BlockingQueue;
import java.util.concurrent.RejectedExecutionHandler;
import java.util.concurrent.ThreadFactory;
import java.util.concurrent.ThreadPoolExecutor;
import java.util.concurrent.TimeUnit;

/**
 * 用来执行基于第三方数据源的查询任务的线程池，由于需要在线程执行前和执行后做相关处理，
 * 所以需要使用java原生线程池。
 * @author yinwenjie
 */
public class DynamicDataSourceThreadPoolExecutor extends ThreadPoolExecutor {
  
  // 为了保证锁的对应性，这里采用的拒绝策略只能是AbortPolicy
  public DynamicDataSourceThreadPoolExecutor(int corePoolSize, int maximumPoolSize, long keepAliveTime,
      TimeUnit unit, BlockingQueue<Runnable> workQueue, ThreadFactory threadFactory,
      RejectedExecutionHandler handler) {
    super(corePoolSize, maximumPoolSize, keepAliveTime, unit, workQueue, threadFactory, handler);
  }

  /* (non-Javadoc)
   * @see java.util.concurrent.ThreadPoolExecutor#beforeExecute(java.lang.Thread, java.lang.Runnable)
   */
  @Override
  protected void beforeExecute(Thread t, Runnable r) {
    /*
     * 当基于动态数据源的查询任务被激活时，DynamicDataSourceManager中将加上读锁
     * */
    DynamicDataSourceThread dynamicThread = (DynamicDataSourceThread)t;
    dynamicThread.getDynamicDataSourceManger().readLock();
  }

  /* (non-Javadoc)
   * @see java.util.concurrent.ThreadPoolExecutor#afterExecute(java.lang.Runnable, java.lang.Throwable)
   */
  @Override
  protected void afterExecute(Runnable r, Throwable t) {
    /*
     * 1、当基于动态数据源的查询任务正常结束或者异常结束时，DynamicDataSourceManager中的读锁将被取消
     * 2、当前线程所绑定运行的session将被取出，被确认其已经关闭（如果没有关闭，这里将强制关闭，并且视是否有异常确定）
     * TODO 第二个问题目前看来只能在异步方法内部去做了
     * */
    Thread currentThread = Thread.currentThread();
    DynamicDataSourceThread dynamicThread = (DynamicDataSourceThread)currentThread;
    dynamicThread.getDynamicDataSourceManger().unReadLock();
  }
}