package com.bizunited.platform.core.service.dataview.internal;

import com.bizunited.platform.core.entity.DataViewAuthVerticalEntity;
import com.bizunited.platform.core.entity.DataViewAuthVerticalRelationEntity;
import com.bizunited.platform.core.entity.DataViewEntity;
import com.bizunited.platform.core.entity.DataViewFieldEntity;
import com.bizunited.platform.core.repository.dataview.DataViewAuthVerticalRelationRepository;
import com.bizunited.platform.core.repository.dataview.DataViewAuthVerticalRepository;
import com.bizunited.platform.core.repository.dataview.DataViewRepository;
import com.bizunited.platform.core.service.NebulaToolkitService;
import com.bizunited.platform.core.service.dataview.DataViewAuthVerticalRelationService;
import com.bizunited.platform.core.service.dataview.DataViewAuthVerticalService;
import com.google.common.collect.Sets;

import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;

import javax.transaction.Transactional;
import java.util.*;
import java.util.stream.Collectors;

/**
 * 数据视图纵向权限，DataViewAuthVerticalEntity业务模型的服务层接口实现
 * @author saturn
 */
//TODO 该类下的数据视图权限部分，需要优化
@Service("DataViewAuthVerticalServiceImpl")
public class DataViewAuthVerticalServiceImpl implements DataViewAuthVerticalService {
  @Autowired
  private DataViewRepository dataViewRepository;
  @Autowired
  private DataViewAuthVerticalRepository dataViewAuthVerticalRepository;
  @Autowired
  private DataViewAuthVerticalRelationService dataViewAuthVerticalRelationService;
  @Autowired
  private DataViewAuthVerticalRelationRepository dataViewAuthVerticalRelationRepository;
  /** 
   * Kuiper表单引擎用于减少编码工作量的工具包 
   */ 
  @Autowired
  private NebulaToolkitService nebulaToolkitService;
  @Transactional
  @Override
  public Set<DataViewAuthVerticalEntity> create(Set<DataViewAuthVerticalEntity> authVerticals) {
    Validate.notNull(authVerticals,"新增时，传入的纵向权限信息不能为空！");
    Validate.isTrue(!CollectionUtils.isEmpty(authVerticals),"新增时，传入的纵向权限信息不能为空！");

    //验证数据正确性以及常规校验
    for(DataViewAuthVerticalEntity e : authVerticals){
      this.createValidation(e);
    }

    //批量保存
    this.dataViewAuthVerticalRepository.saveAll(authVerticals);

    //当传入参数方式是固定值时
    Set<DataViewAuthVerticalEntity> auths = authVerticals.stream().filter(e -> e.getParamSourceType() == 2).collect(Collectors.toSet());
    if(!CollectionUtils.isEmpty(auths)){
      for(DataViewAuthVerticalEntity e : auths){
        Set<DataViewAuthVerticalRelationEntity> authRelations = e.getAuthRelations();
        authRelations.stream().forEach(x -> x.setAuthVertical(e));
        authRelations.forEach(x -> dataViewAuthVerticalRelationService.createValidation(x));
        dataViewAuthVerticalRelationRepository.saveAll(authRelations);
      }
    }

    // TODO manyToMany关联需要显示的字段集目前版本下还需要开发人员自行处理

    // 返回最终处理的结果，里面带有详细的关联信息
    return authVerticals;
  }


  /**
   * 在创建一个新的DataViewAuthVerticalEntity模型对象之前，检查对象各属性的正确性，其主键属性必须没有值
   */
  public void createValidation(DataViewAuthVerticalEntity dataViewAuthVerticalEntity) {
    Validate.notNull(dataViewAuthVerticalEntity , "纵向数据权限的对象信息必须传入!!");
    Validate.isTrue(StringUtils.isBlank(dataViewAuthVerticalEntity.getId()), "添加纵向数据权限信息时，（主键）不能有值！");
    dataViewAuthVerticalEntity.setId(null);

    Validate.notNull(dataViewAuthVerticalEntity.getParamSourceType(), "参数来源不能为空！");
    Validate.notNull(dataViewAuthVerticalEntity.getAuthType(), "权限类型不能为空！");
    Validate.notNull(dataViewAuthVerticalEntity.getOprtType(), "操作符不能为空！");

    DataViewEntity dataView = dataViewAuthVerticalEntity.getDataView();
    Validate.notNull(dataView,"添加纵向数据权限信息时，必须传入数据视图信息，请检查!!");
    Set<DataViewFieldEntity> displayFields = dataViewAuthVerticalEntity.getDisplayFields();
    Validate.isTrue(!CollectionUtils.isEmpty(displayFields),"显示字段必须传入!!");

    switch (dataViewAuthVerticalEntity.getParamSourceType()){
      case 1 : //外部传入
        Validate.isTrue(CollectionUtils.isEmpty(dataViewAuthVerticalEntity.getAuthRelations()),"添加纵向数据权限信息时，且参数来源不是固定值，权限关系必须为空，请检查!!");
        Validate.notNull(dataViewAuthVerticalEntity.getField(),"添加纵向数据权限信息时，且参数来源是外部传入方式，指定的权限字段信息不能为空，请检查!!");
        break;
      case 2 : //固定值
        Validate.isTrue(!CollectionUtils.isEmpty(dataViewAuthVerticalEntity.getAuthRelations()),"添加纵向数据权限信息时，且参数来源为固定值，必须传入指定的权限关系，请检查!!");
        break;
      case 3 : //预制值
        Validate.isTrue(dataViewAuthVerticalEntity.getParamSourceType() != 3 ,"纵向数据权限参数来源没有预制值方式！！");
        break;
      default:
        break;
    }
  }

  @Transactional
  @Override
  public Set<DataViewAuthVerticalEntity> update(Set<DataViewAuthVerticalEntity> authVerticals , String dataViewId) {
    Validate.notBlank(dataViewId,"更新时，传入的视图ID不能为空！！");
    Optional<DataViewEntity> op = dataViewRepository.findById(dataViewId);
    DataViewEntity dataViewEntity = op.orElse(null);
    Validate.notNull(dataViewEntity,"根据视图ID，没有获取到相应的视图信息，请检查!!");
    if(CollectionUtils.isEmpty(authVerticals)){
      Set<DataViewAuthVerticalEntity> authVers = dataViewAuthVerticalRepository.findDetailsByDataViewCode(dataViewEntity.getCode());
      if(!CollectionUtils.isEmpty(authVers)){
        for(DataViewAuthVerticalEntity x : authVers){
          dataViewAuthVerticalRelationRepository.deleteByAuthVertical(x.getId());
        }
        dataViewAuthVerticalRepository.deleteByDataView(dataViewId);
      }
      return Sets.newHashSet();
    }
    
    Set<DataViewAuthVerticalEntity> needInsertSets = authVerticals.stream().filter(e -> StringUtils.isBlank(e.getId())).collect(Collectors.toSet());
    Set<DataViewAuthVerticalEntity> needUpdateSets = authVerticals.stream().filter(e -> StringUtils.isNotBlank(e.getId())).collect(Collectors.toSet());
    Set<DataViewAuthVerticalEntity> dbAuthVerticals = dataViewAuthVerticalRepository.findDetailsByDataViewCode(dataViewEntity.getCode());
    Set<String> needDeleteFieldKeys = nebulaToolkitService.collectionDiffent(dbAuthVerticals,authVerticals, DataViewAuthVerticalEntity::getId);

    //可能需要增加的
    if(!CollectionUtils.isEmpty(needInsertSets)){
      this.create(needInsertSets);
    }

    //可能需要删除的
    if(!CollectionUtils.isEmpty(needDeleteFieldKeys)){
      for(String authVerticalKey : needDeleteFieldKeys){
        //删除明细
        dataViewAuthVerticalRelationRepository.deleteByAuthVertical(authVerticalKey);
        //删除主数据
        dataViewAuthVerticalRepository.deleteById(authVerticalKey);
      }
    }

    if(CollectionUtils.isEmpty(needUpdateSets)){
      return authVerticals;
    }

    //可能需要更新的
    //更新前需过验证
    for(DataViewAuthVerticalEntity e : needUpdateSets){
      this.updateValidation(e);

      // ===================基本信息
      String currentId = e.getId();
      Optional<DataViewAuthVerticalEntity> opt = this.dataViewAuthVerticalRepository.findById(currentId);
      DataViewAuthVerticalEntity currentDataViewAuthVerticalEntity = opt.orElse(null);
      Validate.notNull(currentDataViewAuthVerticalEntity ,"未发现指定的原始模型对象信");
      // 开始重新赋值——一般属性
      currentDataViewAuthVerticalEntity.setAuthType(e.getAuthType());
      currentDataViewAuthVerticalEntity.setOprtType(e.getOprtType());
      currentDataViewAuthVerticalEntity.setDataView(e.getDataView());

      //当参数来源传入方式不是固定值时，清空权限关系
      if(e.getParamSourceType() != 2){
        e.setAuthRelations(null);
      }

      this.dataViewAuthVerticalRepository.save(currentDataViewAuthVerticalEntity);

      Set<DataViewAuthVerticalRelationEntity> authRelations = e.getAuthRelations();
      if(CollectionUtils.isEmpty(authRelations) || e.getParamSourceType() == 3){
        return authVerticals;
      }
      //这里的关系权限集可能包含新增、更新与删除，所以需分情况验证，但是删除的验证不用理会
      //1.可能的新增
      Set<DataViewAuthVerticalRelationEntity> needInsertAuthRelations = authRelations.stream().filter(x -> StringUtils.isBlank(x.getId())).collect(Collectors.toSet());
      if(!CollectionUtils.isEmpty(needInsertAuthRelations)){
        needInsertAuthRelations.forEach(y -> {
          y.setAuthVertical(currentDataViewAuthVerticalEntity);
          dataViewAuthVerticalRelationService.createValidation(y);
        });
        dataViewAuthVerticalRelationRepository.saveAll(needInsertAuthRelations);
      }

      //2.可能的更新
      Set<DataViewAuthVerticalRelationEntity> needUpdateAuthRelations = authRelations.stream().filter(x -> StringUtils.isNotBlank(x.getId())).collect(Collectors.toSet());
      if(!CollectionUtils.isEmpty(needUpdateAuthRelations)){
        needUpdateAuthRelations.forEach(y -> {
          y.setAuthVertical(currentDataViewAuthVerticalEntity);
          dataViewAuthVerticalRelationService.updateValidation(y);
        });
        dataViewAuthVerticalRelationRepository.saveAll(needUpdateAuthRelations);
      }

      //3.可能的删除
      Set<DataViewAuthVerticalRelationEntity> dbAuthVerticalRelations = dataViewAuthVerticalRelationRepository.findDetailsByAuthVertical(currentDataViewAuthVerticalEntity.getId());
      Set<String> needDeleteAuthVerticalRelationKeys = nebulaToolkitService.collectionDiffent(dbAuthVerticalRelations,authRelations, DataViewAuthVerticalRelationEntity::getId);
      if(!CollectionUtils.isEmpty(needDeleteAuthVerticalRelationKeys)){
        needDeleteAuthVerticalRelationKeys.forEach(x -> dataViewAuthVerticalRelationRepository.deleteById(x));
      }
    }
    return authVerticals;
  }
  
  /**
   * 在更新一个已有的DataViewAuthVerticalEntity模型对象之前，该私有方法检查对象各属性的正确性，其id属性必须有值
   */
  public void updateValidation(DataViewAuthVerticalEntity dataViewAuthVerticalEntity) {
    Validate.notNull(dataViewAuthVerticalEntity,"更新时，纵向权限信息必须传入！！");
    Validate.isTrue(StringUtils.isNotBlank(dataViewAuthVerticalEntity.getId()), "修改纵向数据权限信息时，（主键）必须有值！");

    // 基础信息判断，基本属性，需要满足not null
    Validate.notNull(dataViewAuthVerticalEntity.getParamSourceType(), "参数来源不能为空！");
    Validate.notNull(dataViewAuthVerticalEntity.getAuthType(), "权限类型不能为空！");
    Validate.notNull(dataViewAuthVerticalEntity.getOprtType(), "操作符不能为空！");

    DataViewEntity dataView = dataViewAuthVerticalEntity.getDataView();
    Validate.notNull(dataView,"更新纵向数据权限信息时，必须传入数据视图信息，请检查!!");

    switch (dataViewAuthVerticalEntity.getParamSourceType()){
      case 1 : //外部传入
        Validate.isTrue(CollectionUtils.isEmpty(dataViewAuthVerticalEntity.getAuthRelations()),"更新纵向数据权限信息时，且参数来源不是固定值，权限关系必须为空，请检查!!");
        Validate.notNull(dataViewAuthVerticalEntity.getField(),"更新纵向数据权限信息时，且参数来源是外部传入方式，指定的权限字段信息不能为空，请检查!!");
        break;
      case 2 : //固定值
        Validate.isTrue(!CollectionUtils.isEmpty(dataViewAuthVerticalEntity.getAuthRelations()),"更新纵向数据权限信息时，且参数来源为固定值，必须传入指定的权限关系，请检查!!");
        break;
      case 3 : //预制值
        Validate.isTrue(dataViewAuthVerticalEntity.getParamSourceType() != 3 ,"纵向数据权限参数来源没有预制值方式！！");
        break;
      default:
        break;
    }
  }

  @Override
  public Set<DataViewAuthVerticalEntity> findDetailsByDataViewCode(String dataView) {
    if(StringUtils.isBlank(dataView)) { 
      return Sets.newHashSet();
    }
    Set<DataViewAuthVerticalEntity> auths = this.dataViewAuthVerticalRepository.findDetailsByDataViewCode(dataView);
    Collection<DataViewAuthVerticalEntity> collection = this.nebulaToolkitService.copyCollectionByWhiteList(auths,DataViewAuthVerticalEntity.class,DataViewAuthVerticalEntity.class, HashSet.class, ArrayList.class,new String[]{"authRelations","authRelations.authVertical","displayFields","dataView","field"});
    return (Set<DataViewAuthVerticalEntity>)collection;
  }
  @Override
  public Set<DataViewAuthVerticalEntity> findDetailsByDataViewIds(String[] dataViewIds) {
    if(dataViewIds == null || dataViewIds.length == 0) {
      return Sets.newHashSet();
    }
    
    Set<DataViewAuthVerticalEntity> auths = this.dataViewAuthVerticalRepository.findDetailsByDataViewIds(dataViewIds);
    Collection<DataViewAuthVerticalEntity> collection = this.nebulaToolkitService.copyCollectionByWhiteList(auths,DataViewAuthVerticalEntity.class,DataViewAuthVerticalEntity.class, HashSet.class, ArrayList.class,new String[]{"authRelations","authRelations.authVertical","displayFields","dataView","field"});
    return (Set<DataViewAuthVerticalEntity>)collection;
  }
  
  @Override
  public DataViewAuthVerticalEntity findDetailsById(String id) { 
    if(StringUtils.isBlank(id)) { 
      return null;
    }
    return this.dataViewAuthVerticalRepository.findDetailsById(id);
  }
  @Override
  public DataViewAuthVerticalEntity findById(String id) { 
    if(StringUtils.isBlank(id)) { 
      return null;
    }
    Optional<DataViewAuthVerticalEntity> op = dataViewAuthVerticalRepository.findById(id);
    return op.orElse(null);
  }
  @Override
  @Transactional
  public void deleteById(String id) {
    // 只有存在才进行删除
    Validate.notBlank(id , "进行删除时，必须给定主键信息!!");
    Optional<DataViewAuthVerticalEntity> op = this.dataViewAuthVerticalRepository.findById(id);
    op.ifPresent(dataViewAuthVerticalEntity -> this.dataViewAuthVerticalRepository.delete(dataViewAuthVerticalEntity));
  }
} 
