package com.bizunited.platform.core.service.dataview.internal;

import com.alibaba.fastjson.JSONArray;
import com.alibaba.fastjson.JSONObject;
import com.bizunited.platform.core.common.utils.ChineseCharUtil;
import com.bizunited.platform.core.entity.DataSourceEntity;
import com.bizunited.platform.core.entity.DataViewEntity;
import com.bizunited.platform.core.entity.DataViewGroupEntity;
import com.bizunited.platform.core.repository.dataview.DataSourceRepository;
import com.bizunited.platform.core.repository.dataview.DataViewGroupRepository;
import com.bizunited.platform.core.repository.dataview.DataViewRepository;
import com.bizunited.platform.core.repository.dataview.DataViewThirdDataSourceRepository;
import com.bizunited.platform.core.repository.dataview.DataViewRepositoryCustom;
import com.bizunited.platform.core.service.dataview.DataViewGroupService;
import com.google.common.collect.Lists;

import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;

import javax.transaction.Transactional;
import java.util.*;
import java.util.stream.Collectors;

/**
 * DataViewGroupEntity业务模型的服务层接口实现
 * @author saturn
 */
@Service("DataViewGroupEntityServiceImpl")
public class DataViewGroupServiceImpl implements DataViewGroupService {
  @Autowired
  private DataViewGroupRepository dataViewGroupEntityRepository;
  @Autowired
  private DataSourceRepository dataSourceRepository;
  @Autowired
  private DataViewRepository dataViewRepository;
  @Autowired
  private DataViewRepositoryCustom dataViewRepositoryCustom;
  @Autowired
  private DataViewThirdDataSourceRepository dataViewThirdDataSourceRepository;

  /**
   * 在创建一个新的DataViewGroupEntity模型对象之前，检查对象各属性的正确性，其主键属性必须没有值
   */
  @Override
  public void createValidation(DataViewGroupEntity dataViewGroupEntity) {
    Validate.notNull(dataViewGroupEntity , "进行当前操作时，信息对象必须传入!!");
    // 判定那些不能为null的输入值：条件为 caninsert = true，且nullable = false
    Validate.isTrue(StringUtils.isBlank(dataViewGroupEntity.getId()), "添加信息时，当期信息的数据编号（主键）不能有值！");
    dataViewGroupEntity.setId(null);
    Validate.notBlank(dataViewGroupEntity.getGroupName(), "分组名(中文名)不能为空！");
    Validate.notBlank(dataViewGroupEntity.getCode(), "编码不能为空！");
    Validate.notBlank(dataViewGroupEntity.getTargetTable(), "基准数据表不能为空！");
    // 验证长度，被验证的这些字段符合特征: 字段类型为String，且不为PK （注意连续空字符串的情况） 
    Validate.isTrue(dataViewGroupEntity.getGroupName() == null || dataViewGroupEntity.getGroupName().length() < 64 , "分组名(中文名),填入值超过了限定长度(64)，请检查!");
    Validate.isTrue(dataViewGroupEntity.getTargetTable() == null || dataViewGroupEntity.getTargetTable().length() < 64 , "基准数据表,填入值超过了限定长度(64)，请检查!");
    //找出同一数据源下所有分组
    List<DataViewGroupEntity> existGroups = dataViewGroupEntityRepository.findDetails();
    Set<DataViewGroupEntity> mainGroups = new HashSet<>();//主数据源组
    Set<DataViewGroupEntity> threeGroups = new HashSet<>();//第三方数据源组
    //分别提取主数据源组和第三方数据源组
    if(!CollectionUtils.isEmpty(existGroups)) {
      existGroups.stream().forEach(e -> {
        if(e.getDataSource() == null) {
          mainGroups.add(e);
        }else {
          threeGroups.add(e);
        }
      });
    }else {
      return;
    }
    //验证基准表再各自分组中，是否重复
    if(dataViewGroupEntity.getDataSource() == null) {
      Validate.isTrue(mainGroups.stream().allMatch(e -> !StringUtils.equals(e.getTargetTable(), dataViewGroupEntity.getTargetTable())), 
        "新增基准表已存在于主数据源分组:%s,请检查", dataViewGroupEntity.getTargetTable());
    }else {
      Validate.isTrue(threeGroups.stream().allMatch(e -> !StringUtils.equals(e.getTargetTable(), dataViewGroupEntity.getTargetTable())), 
        "新增基准表已存在于数据源分组:%s,请检查", dataViewGroupEntity.getTargetTable());
    }
  }
  @Transactional
  @Override
  public DataViewGroupEntity create(DataViewGroupEntity dataViewGroupEntity , boolean ignoreValidate) {
    // 为了新增字段后不让前端报错，新增时如果没有CODE，默认添加一个
    if(StringUtils.isBlank(dataViewGroupEntity.getCode())){
      dataViewGroupEntity.setCode(UUID.randomUUID().toString());
    }
    if(!ignoreValidate) {
      this.createValidation(dataViewGroupEntity);
    }
    // ===============================
    //  和业务有关的验证填写在这个区域    
    // ===============================
    if(null!=dataViewGroupEntity.getDataSource()){
      DataSourceEntity dataSource = dataViewGroupEntity.getDataSource();
      Validate.notNull(dataSource.getId(),"所选数据源ID不能为空");
      Optional<DataSourceEntity> op = dataSourceRepository.findById(dataSource.getId());
      Validate.isTrue(op.isPresent(),"未在数据库中找到第三方数据源");
      Validate.isTrue(!ChineseCharUtil.hasChinese(dataViewGroupEntity.getTargetTable()),"基准表表名不能含有中文，请检查！");
    }
    //验证该基础数据表是否存在
    //区分是主数据源还是第三方数据源
    if (null == dataViewGroupEntity.getDataSource()) {
      // 主数据源
      dataViewRepositoryCustom.checkTable(dataViewGroupEntity.getTargetTable());
    }else {
      // 第三方
      dataViewThirdDataSourceRepository.checkTable(dataViewGroupEntity.getDataSource().getCode(),dataViewGroupEntity.getTargetTable());
    }
    this.dataViewGroupEntityRepository.save(dataViewGroupEntity);
    return dataViewGroupEntity;
  }

  @Transactional
  @Override
  public DataViewGroupEntity update(DataViewGroupEntity dataViewGroupEntity) { 
    this.updateValidation(dataViewGroupEntity);
    // ===================基本信息
    String currentId = dataViewGroupEntity.getId();
    Optional<DataViewGroupEntity> op = this.dataViewGroupEntityRepository.findById(currentId);
    DataViewGroupEntity currentDataViewGroupEntity = op.orElse(null);
    Validate.notNull(currentDataViewGroupEntity ,"未发现指定的原始模型对象信");

    // 开始重新赋值，首先是一般对象，updateable = true形式的
    currentDataViewGroupEntity.setGroupName(dataViewGroupEntity.getGroupName());
    currentDataViewGroupEntity.setTargetTable(dataViewGroupEntity.getTargetTable());
    
    this.dataViewGroupEntityRepository.saveAndFlush(currentDataViewGroupEntity);
    // =============
    // ManyToMany 结构的关联关系，需要开发人员自行处理
    // 分组下数据视图的移动与更新，由其他接口负责。
    // =============
    return currentDataViewGroupEntity;
  }
  /**
   * 在更新一个已有的DataViewGroupEntity模型对象之前，检查对象各属性的正确性，其id属性必须有值
   */
  @Override
  public void updateValidation(DataViewGroupEntity dataViewGroupEntity) { 
    Validate.isTrue(!StringUtils.isBlank(dataViewGroupEntity.getId()), "修改信息时，当期信息的数据编号（主键）必须有值！");
    
    // 基础信息判断，基本属性，需要满足not null 且 updateable == true
    Validate.notBlank(dataViewGroupEntity.getGroupName(), "分组名(中文名)不能为空！");
    Validate.notBlank(dataViewGroupEntity.getTargetTable(), "基准数据表不能为空！");
    // 验证长度，被验证的这些字段符合特征: 字段类型为String，且不为PK，且canupdate = true
    Validate.isTrue(dataViewGroupEntity.getGroupName() == null || dataViewGroupEntity.getGroupName().length() < 64 , "分组名(中文名),填入值超过了限定长度(64)，请检查!");
    Validate.isTrue(dataViewGroupEntity.getTargetTable() == null || dataViewGroupEntity.getTargetTable().length() < 64 , "基准数据表,填入值超过了限定长度(64)，请检查!");
    //因为此处前端传回实体中没有分组信息，需要重新查询
    DataViewGroupEntity dataViewGroup = dataViewGroupEntityRepository.findDetailsById(dataViewGroupEntity.getId());
    //找出同一数据源下所有分组
    List<DataViewGroupEntity> existGroups = dataViewGroupEntityRepository.findDetails();
    if(!CollectionUtils.isEmpty(existGroups)){
      List<DataViewGroupEntity> sameSourceGroups = existGroups.stream().filter(o -> {
        //主数据源
        if(null == dataViewGroup.getDataSource()){
          return null == o.getDataSource();
        }else {
          //第三方数据源
          return dataViewGroup.getDataSource().getId().equals(o.getDataSource().getId());
        }
      }).collect(Collectors.toList());
      if(!CollectionUtils.isEmpty(sameSourceGroups)){
        sameSourceGroups.stream().forEach(o -> {
          //修改组名不能重复
          Validate.isTrue(!o.getGroupName().equals(dataViewGroupEntity.getGroupName()),"同一数据源下，修改数据视图分组名不能重复！请检查");
        });
      }

    }
  } 
  @Override
  public List<DataViewGroupEntity> findByDataSource(String dataSourceId) { 
    if(StringUtils.isBlank(dataSourceId)) { 
      return Lists.newArrayList();
    }
    return this.dataViewGroupEntityRepository.findByDataSource(dataSourceId);
  }
  @Override
  public DataViewGroupEntity findDetailsById(String id) { 
    if(StringUtils.isBlank(id)) { 
      return null;
    }
    return this.dataViewGroupEntityRepository.findDetailsById(id);
  }
  @Override
  @Transactional
  public void deleteById(String id) {
    // 只有存在才进行删除
    Validate.notBlank(id , "进行删除时，必须给定主键信息!!");
    Optional<DataViewGroupEntity> op = this.dataViewGroupEntityRepository.findById(id);
    op.ifPresent(dataViewGroupEntity -> this.dataViewGroupEntityRepository.delete(dataViewGroupEntity));
  }

  @Override
  @Transactional
  public void moveGroup(String dataViewId, String dataViewGroupId) {
    Validate.notBlank(dataViewId,"所选视图ID不能为空！");
    Validate.notBlank(dataViewGroupId,"所选新分组ID不能为空！");
    Optional<DataViewEntity> opView = dataViewRepository.findById(dataViewId);
    Validate.isTrue(opView.isPresent(),"未找到所选视图信息，请检查！");
    DataViewEntity dataViewEntity = opView.get();
    Optional<DataViewGroupEntity> opGroup = dataViewGroupEntityRepository.findById(dataViewGroupId);
    DataViewGroupEntity dataViewGroupEntity = opGroup.orElse(null);
    Validate.notNull(dataViewGroupEntity,"未找到所选分组信息，请检查！");

    dataViewEntity.setDataViewGroup(dataViewGroupEntity);
    dataViewRepository.saveAndFlush(dataViewEntity);
  }
  @Override
  public List<DataViewGroupEntity> findByDataSourceCode(String code) {
    List<DataViewGroupEntity> gourpList = null;
    if(StringUtils.isBlank(code)) {
      gourpList = dataViewGroupEntityRepository.findByDataSource();
    }else {
      gourpList = dataViewGroupEntityRepository.findByDataSourceCode(code);
    }
    return gourpList;
  }
  @Override
  public List<JSONObject> findDetailsByDataSource(String dataSourceId) {
    //dataSourceId可以为空，为空代表主数据源
    //根据数据源id查询分组
    List<DataViewGroupEntity> firstLayer = null;
    if(StringUtils.isBlank(dataSourceId)) {
      firstLayer = this.dataViewGroupEntityRepository.findByDataSource();
    }else {
      firstLayer = this.dataViewGroupEntityRepository.findByDataSource(dataSourceId);
    }

    List<JSONObject> list = new ArrayList<>();
    if(CollectionUtils.isEmpty(firstLayer)){
      return list;
    }

    //第一层（数据源的分组）
    for(DataViewGroupEntity group : firstLayer) {
      group.setDataSource(null);
      String jsonStr = JSONObject.toJSONString(group);
      JSONObject json = JSONObject.parseObject(jsonStr);
      json.put("title", group.getGroupName());
      List<DataViewEntity> views = dataViewRepository.findByDataViewGroup(group.getId());
      JSONArray jaArray = new JSONArray();
      if(!CollectionUtils.isEmpty(views)) {
        //第二层（当前数据源下的视图信息）
        for(DataViewEntity view : views) {
          view.setDataViewGroup(null);
          view.setFields(null);
          view.setFilters(null);
          view.setSystemFilters(null);
          view.setDataSource(null);
          String vs = JSONObject.toJSONString(view);
          JSONObject jvs = JSONObject.parseObject(vs);
          jvs.put("title", view.getName());
          jaArray.add(jvs);
        }
      }
      json.put("children", jaArray);
      list.add(json);
    }
    return list;
  }

  @Override
  public DataViewGroupEntity findByCode(String code) {
    if(StringUtils.isBlank(code)){
      return null;
    }
    return dataViewGroupEntityRepository.findByCode(code);
  }
} 
