package com.bizunited.platform.core.service.image;

import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.Calendar;
import java.util.Date;
import java.util.LinkedList;
import java.util.List;
import java.util.Optional;
import java.util.Random;
import java.util.UUID;
import javax.transaction.Transactional;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.stereotype.Service;
import org.springframework.util.Base64Utils;
import org.springframework.web.multipart.MultipartFile;
import com.bizunited.platform.core.entity.OrdinaryFileEntity;
import com.bizunited.platform.core.repository.OrdinaryFileRepository;
import com.bizunited.platform.core.service.file.NebulaFileService;

/**
 * 文件上传服务
 * @author yinwenjie
 */
@Service("ImageUpdateServiceImpl")
public class FileUpdateServiceImpl implements FileUpdateService {
  /**
   * 打印日志.
   */
  private final Logger LOGGER = LoggerFactory.getLogger(FileUpdateServiceImpl.class);
  /**
   * 最大允许的单个文件上传大小（单位MB）
   */
  @Value("${kuiper.file.maxFileSize}")
  private Integer maxFileSize;
  /**
   * 文件扩展名白名单
   */
  @Value("${kuiper.file.whitePrefixs}")
  private String[] whitePrefixs;
  @Autowired
  private NebulaFileService nebulaFileService;
  
  @Autowired
  private OrdinaryFileRepository ordinaryFileRepository;
  
  /* (non-Javadoc)
   * @see com.bizunited.platform.core.service.image.FileUpdateService#fileUpload(java.lang.String, java.lang.String, java.lang.Integer, org.springframework.web.multipart.MultipartFile[])
   */
  @Override
  @Transactional
  public List<OrdinaryFileEntity> fileUpload(String subSystem , String creator , Integer effective , MultipartFile[] files) throws IllegalArgumentException {
    Validate.notNull(files != null && files.length > 0 , "上传的文件信息至少有一个，请检查!!");
    Validate.notBlank(creator , "附件的提交人必须传入，请检查!!");
    /*
     * 1、开始保存文件，注意，文件都要重命名。 为了简单起见重命名使用java自带的UUID工具完成即可
     * 2、正式写入文件，如果以上所有步骤都成功，则向上传者返回文件存储的提示信息
     * 注意，这些文件在上传时，由于不清楚这些图片在显示时会使用什么样的特效 所以在文件上传时就进行redis缓存存储是没有意义的。
     * 
     * 最后，本工程没有提供上传的测试页面，测试是使用postman等软件完成的
     */
    List<OrdinaryFileEntity> ordinaryFileEntitys = new LinkedList<>();
    for (MultipartFile file : files) {
      // 文件内容
      byte[] fileContext = null;
      try {
        fileContext = file.getBytes(); 
      } catch(Exception e) {
        LOGGER.error(e.getMessage() , e);
        throw new IllegalArgumentException(e);
      }
      String originalFilename = file.getOriginalFilename();
      int prefixIndex = originalFilename.lastIndexOf('.');
      OrdinaryFileEntity result = this.saveOrdinaryFile(prefixIndex, originalFilename, subSystem, effective, creator, fileContext);
      ordinaryFileEntitys.add(result);
    }
    
    ordinaryFileRepository.saveAll(ordinaryFileEntitys);
    return ordinaryFileEntitys;
  }
  
  private OrdinaryFileEntity saveOrdinaryFile(int prefixIndex , String originalFilename , String subSystem, Integer effective , String creator , byte[] fileContext) {
    String prefix = null;
    if (prefixIndex != -1) {
      prefix = originalFilename.substring(prefixIndex + 1);
      prefix = prefix.toLowerCase();
    }
    
    // 1、======
    // 可以使用日期作为文件夹的名字
    Date nowDate = new Date();
    String folderName = new SimpleDateFormat("yyyyMMdd").format(nowDate);
    String renameImage = UUID.randomUUID().toString();
    String relativePath = null;
    if (!StringUtils.isBlank(subSystem)) {
      relativePath = StringUtils.join("/" , subSystem , "/" , folderName , "/" , (new Random().nextInt(100) % 10));
    } else {
      relativePath = StringUtils.join("/" , folderName , "/" , (new Random().nextInt(100) % 10));
    }
    String fileRename = StringUtils.join(renameImage , "." , prefix);
    
    // 保存文件
    OrdinaryFileEntity result = this.fileUpload(relativePath, originalFilename, fileRename, prefix, effective, creator, fileContext);
    return result;
  }

  /* (non-Javadoc)
   * @see com.bizunited.platform.core.service.image.FileUpdateService#fileUpload(java.lang.String, java.lang.String, java.lang.Integer, java.lang.String[], java.lang.String[])
   */
  @Override
  @Transactional
  public List<OrdinaryFileEntity> fileUpload(String subSystem, String creator, Integer effective, String[] fileNanmes, String[] base64Contents) {
    Validate.notBlank(creator , "文件的提交人必须传入，请检查!!");
    if(base64Contents == null || base64Contents.length == 0) {
      throw new IllegalArgumentException("错误的base64编码文件内容!!");
    }
    if(fileNanmes == null || fileNanmes.length == 0) {
      throw new IllegalArgumentException("错误的文件名数量信息!!");
    }
    Validate.isTrue(fileNanmes.length == base64Contents.length , "文件内容数量和文件名数量不匹配，请检查!!");
    
    // 一个文件一个文件的保存
    List<OrdinaryFileEntity> ordinaryFileEntitys = new LinkedList<>();
    for(int index = 0 ; index < fileNanmes.length ; index++) {
      String originalFilename = fileNanmes[index];
      String base64Content = base64Contents[index];
      byte[] fileContext = Base64Utils.decodeFromString(base64Content);
      
      // 保存文件
      int prefixIndex = originalFilename.lastIndexOf('.');
      OrdinaryFileEntity result = this.saveOrdinaryFile(prefixIndex, originalFilename, subSystem, effective, creator, fileContext);
      ordinaryFileEntitys.add(result);
    }
    
    ordinaryFileRepository.saveAll(ordinaryFileEntitys);
    return ordinaryFileEntitys;
  }

  /**
   * 该私有方法保存文件内容
   * @param relativePath 指定的相对路径
   * @param originalFilename 原始文件名，包括扩展名
   * @param fileRename 重命名后的文件名，包括扩展名
   * @param effective 指定的文件有效期时长，单位“天”
   * @param fileContext 文件内容
   * @return 文件的对象描述
   */
  private OrdinaryFileEntity fileUpload(String relativePath , String originalFilename, String fileRename , String prefix ,Integer effective , String creator , byte[] fileContext) {
    nebulaFileService.saveFile(relativePath , originalFilename, fileRename , fileContext);
    
    // 计算有效期
    SimpleDateFormat simpleDateFormat = new SimpleDateFormat("yyyy-MM-dd HH:mm:ss");
    Date effectiveDate = null;
    try {
      effectiveDate = simpleDateFormat.parse("3999-01-01 00:00:00");
    } catch (ParseException e) {
      LOGGER.error(e.getMessage() , e);
      throw new IllegalArgumentException(e.getMessage());
    }
    // 如果条件成立，则需要按照当前时间从新计算时间
    if(effective != null) {
      Validate.isTrue(effective > 0 , "错误的有效期时长（单位“天”）, 请检查!!");
      Calendar calendar = Calendar.getInstance();
      calendar.add(Calendar.DATE, effective);
      effectiveDate = calendar.getTime();
    }
    
    // 构造信息保存数据库，并进行返回
    OrdinaryFileEntity result = new OrdinaryFileEntity();
    result.setCommitUser(creator);
    result.setRelativeLocal(relativePath);
    result.setFileName(fileRename);
    result.setEffectiveDate(effectiveDate);
    result.setCreateDate(new Date());
    result.setOriginalFileName(originalFilename);
    result.setPrefix(prefix);
    return result;
  }
  
  /* (non-Javadoc)
   * @see com.bizunited.platform.core.service.image.FileUpdateService#fileImageUpload(java.lang.String, java.lang.String, java.lang.Integer, org.springframework.web.multipart.MultipartFile[])
   */
  @Override
  @Transactional
  public List<OrdinaryFileEntity> fileImageUpload(String subsystem, String creator, Integer effective, MultipartFile[] files) throws IllegalArgumentException {
    Validate.notNull(files != null && files.length > 0 , "上传的图片文件信息至少有一个，请检查!!");
    Validate.notBlank(creator , "图片文件的提交人必须传入，请检查!!");
    /*
     * 处理过程为： 
     * 1、首先判断文件后缀格式是否为支持的格式。 支持jpg、png、gif格式的图片上传
     * 2、为了保证网络畅通，要控制文件大小在1MB以下，所以也要进行控制（当然也可以通过spring mvc的配置实现限制）
     * 其它步骤请参见fileUpload
     * */
    for (MultipartFile file : files) {
      String originalFilename = file.getOriginalFilename();
      String prefix = null;
      int prefixIndex = originalFilename.lastIndexOf('.');
      if (prefixIndex != -1) {
        prefix = originalFilename.substring(prefixIndex + 1);
        prefix = prefix.toLowerCase();
      }
      // 只有位图文件和在扩展名白名单设定中的文件，才能进行上传管理
      if (StringUtils.isBlank(prefix) || (!StringUtils.equals(prefix, "png") 
          && !StringUtils.equals(prefix, "gif") && !StringUtils.equals(prefix, "jpg") 
          && !StringUtils.equals(prefix, "jpeg") && !StringUtils.equals(prefix, "bmp") 
          && !StringUtils.equalsAnyIgnoreCase(prefix, whitePrefixs))) {
        throw new IllegalArgumentException("file prefix not be support it(bmp/jpg/png/gif/white prefixs)!");
      }
    }
    
    return this.fileUpload(subsystem, creator , effective, files);
  }

  /* (non-Javadoc)
   * @see com.bizunited.platform.core.service.image.FileUpdateService#fileImageUpload(java.lang.String, java.lang.String, java.lang.Integer, java.lang.String[], java.lang.String[])
   */
  @Override
  public List<OrdinaryFileEntity> fileImageUpload(String subsystem, String creator, Integer effective, String[] fileNanmes, String[] base64Contents) {
    Validate.notBlank(creator , "图片文件的提交人必须传入，请检查!!");
    if(base64Contents == null || base64Contents.length == 0) {
      throw new IllegalArgumentException("错误的base64编码文件内容!!");
    }
    if(fileNanmes == null || fileNanmes.length == 0) {
      throw new IllegalArgumentException("错误的文件名数量信息!!");
    }
    Validate.isTrue(fileNanmes.length == base64Contents.length , "文件内容数量和文件名数量不匹配，请检查!!");
    
    for (int index = 0 ; index < fileNanmes.length ; index++) {
      String originalFilename = fileNanmes[index];
      String prefix = null;
      int prefixIndex = originalFilename.lastIndexOf('.');
      if (prefixIndex != -1) {
        prefix = originalFilename.substring(prefixIndex + 1);
        prefix = prefix.toLowerCase();
      }
      // 只有位图文件和在扩展名白名单设定中的文件，才能进行上传管理
      if (StringUtils.isBlank(prefix) || (!StringUtils.equals(prefix, "png") 
          && !StringUtils.equals(prefix, "gif") && !StringUtils.equals(prefix, "jpg") 
          && !StringUtils.equals(prefix, "jpeg") && !StringUtils.equals(prefix, "bmp") 
          && !StringUtils.equalsAnyIgnoreCase(prefix, whitePrefixs))) {
        throw new IllegalArgumentException("file prefix not be support it(bmp/jpg/png/gif/white prefixs)!");
      }
    }
    
    return this.fileUpload(subsystem, creator, effective, fileNanmes, base64Contents);
  }

  /* (non-Javadoc)
   * @see com.bizunited.platform.core.service.image.FileUpdateService#findByEffectiveDate(java.util.Date)
   */
  @Override
  public List<OrdinaryFileEntity> findByEffectiveDate(Date currentDate) {
    if(currentDate == null) {
      return null;
    }
    return this.ordinaryFileRepository.findByEffectiveDate(currentDate);
  }

  /* (non-Javadoc)
   * @see com.bizunited.platform.core.service.image.FileUpdateService#updateEffective(java.lang.String[])
   */
  @Override
  @Transactional
  public void updateEffective(String[] fileReNames) {
    if(fileReNames == null || fileReNames.length == 0) {
      return;
    }
    SimpleDateFormat simpleDateFormat = new SimpleDateFormat("yyyy-MM-dd HH:mm:ss");
    Date effectiveDate = null;
    try {
      effectiveDate = simpleDateFormat.parse("3999-01-01 00:00:00");
    } catch (ParseException e) {
      LOGGER.error(e.getMessage() , e);
      throw new IllegalArgumentException(e.getMessage());
    }
    
    // 开始一个一个更新
    for (int index = 0; index < fileReNames.length; index++) {
      String fileReNameItem = fileReNames[index];
      int targetIndex = fileReNameItem.lastIndexOf('/');
      String fileReName = fileReNameItem.substring(targetIndex + 1);
      String relativeLocal = fileReNameItem.substring(0, targetIndex);
      OrdinaryFileEntity currentOrdinaryFile = this.ordinaryFileRepository.findByFileNameAndRelativeLocal(fileReName, relativeLocal);
      if(currentOrdinaryFile == null) {
        continue;
      }
      
      // 文件有效时间更新为3999-01-01 00:00:00
      currentOrdinaryFile.setEffectiveDate(effectiveDate);
      this.ordinaryFileRepository.save(currentOrdinaryFile);
    }
  }

  /* (non-Javadoc)
   * @see com.bizunited.platform.core.service.image.FileUpdateService#deleteFiles(java.lang.String[])
   */
  @Override
  @Transactional
  public void deleteFiles(String[] fileIds) {
    Validate.notNull(fileIds , "至少传入一个文件的数据库编号信息!!");
    Validate.isTrue(fileIds.length > 0 , "至少传入一个文件的数据库编号信息!!");
    for (int index = 0; index < fileIds.length; index++) {
      String fileId = fileIds[index];
      Optional<OrdinaryFileEntity> op = this.ordinaryFileRepository.findById(fileId);
      OrdinaryFileEntity currentOrdinaryFile = op.orElse(null);
      if(currentOrdinaryFile == null) {
        continue;
      }
      // 首先删除磁盘文件
      this.nebulaFileService.deleteFile(currentOrdinaryFile.getRelativeLocal(), currentOrdinaryFile.getOriginalFileName(), currentOrdinaryFile.getFileName());
      // 然后删除数据库信息
      this.ordinaryFileRepository.deleteById(fileId);
    }
  }

  @Override
  @Transactional
  public void deleteFile(String filePath, String fileName) {
    Validate.notNull(filePath, "文件路径不能为空");
    Validate.notBlank(fileName, "文件名称不能为空");
    OrdinaryFileEntity file = ordinaryFileRepository.findByFileNameAndRelativeLocal(fileName, StringUtils.trim(filePath));
    Validate.notNull(file, "删除的文件不存在");
    // 首先删除磁盘文件
    this.nebulaFileService.deleteFile(file.getRelativeLocal(), file.getOriginalFileName(), file.getFileName());
    // 然后删除数据库信息
    this.ordinaryFileRepository.delete(file);
  }

}
