package com.bizunited.platform.core.service.init;

import com.alibaba.fastjson.JSONArray;
import com.alibaba.fastjson.JSONObject;
import com.bizunited.platform.core.entity.CompetenceEntity;
import com.bizunited.platform.core.repository.CompetenceRepository;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;

import java.io.BufferedReader;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.util.Date;
import java.util.Iterator;

/**
 * 初始化菜单抽象类
 * @Author: Paul Chan
 * @Date: 2019-10-15 16:00
 */
public abstract class AbstractCompetenceInitProcess implements InitProcessService {

  private static final Logger LOGGER = LoggerFactory.getLogger(AbstractCompetenceInitProcess.class);

  @Autowired
  private CompetenceRepository competenceRepository;

  @Override
  public int sort() {
    return 0;
  }

  /**
   * 初始化核心菜单,inputStream是一个菜单数组的文件输入流内容的格式见当前包下的nebula-competences.json资源文件
   * @param inputStream
   */
  protected void initCompetences(InputStream inputStream) throws IOException {
    if(inputStream == null) {
      return;
    }
    StringBuilder sb = new StringBuilder();
    try (BufferedReader bufferedReader = new BufferedReader(new InputStreamReader(inputStream))) {
      String line = bufferedReader.readLine();
      while (StringUtils.isNotBlank(line)){
        sb.append(line);
        line = bufferedReader.readLine();
      }
    }
    String json = sb.toString();
    if(StringUtils.isBlank(json)) {
      LOGGER.warn("初始化菜单json为空！！");
      return;
    }
    JSONArray competences = JSONArray.parseArray(json);
    this.initCompetences(null, competences);
  }

  /**
   * 初始化核心菜单,parent是菜单的上级，如果菜单没有上级，则parent为null。
   * competences是CompetenceEntity对应的json数组，具体内容格式见nebula-competences.json资源文件，
   * 菜单地址根据菜单名称更新，若菜单不存在，则初始化菜单，且菜单默认启用。
   * 请保证json对象的字段跟nebula-competences.json资源文件一样
   * @param parent
   * @param competences
   */
  protected void initCompetences(CompetenceEntity parent, JSONArray competences) {
    if(competences == null || competences.isEmpty()) {
      return;
    }
    Iterator<Object> iterator = competences.iterator();
    while (iterator.hasNext()) {
      Object object = iterator.next();
      JSONObject competenceJson = JSONObject.parseObject(object.toString());
      CompetenceEntity competence = JSONObject.parseObject(competenceJson.toJSONString(), CompetenceEntity.class);
      CompetenceEntity oldCompetence = competenceRepository.findByComment(competence.getComment());
      if(oldCompetence == null) {
        competence.setCreateDate(new Date());
        competence.setModifyDate(new Date());
        competence.setTstatus(1);
        competence.setParent(parent);
        competenceRepository.save(competence);
        oldCompetence = competence;
      }
      if(competenceJson.containsKey("competences")) {
        this.initCompetences(oldCompetence, competenceJson.getJSONArray("competences"));
      }
    }
  }

}
