package com.bizunited.platform.core.service.init;

import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.regex.Pattern;
import java.util.stream.Collectors;

import javax.transaction.Transactional;

import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;
import org.springframework.web.bind.annotation.RequestMethod;
import org.springframework.web.method.HandlerMethod;
import org.springframework.web.servlet.mvc.method.RequestMappingInfo;
import org.springframework.web.servlet.mvc.method.annotation.RequestMappingHandlerMapping;

import com.bizunited.platform.core.entity.CompetenceEntity;
import com.bizunited.platform.core.repository.CompetenceRepository;
import com.google.common.collect.Sets;
import com.google.common.collect.Sets.SetView;

/**
 * 该初始化过程完成Spring MVC中关于controller接口方法的扫描和持久化记录（每次启动都要刷新记录）
 * @author yinwenjie
 */
@Component("frameworkEndpointProcess")
public class FrameworkEndpointInitProcess implements InitProcessService {
  /**
   * 日志
   */
  private static final Logger LOGGER = LoggerFactory.getLogger(FrameworkEndpointInitProcess.class);
  @Autowired
  private RequestMappingHandlerMapping frameworkEndpointHandler;
  @Autowired
  private CompetenceRepository competenceRepository;
  
  @Override
  public int sort() {
    // 紧接着权限初始化完成，就开始进行http接口初始化
    return 2;
  }

  @Override
  public boolean doProcess() {
    // 每次进程启动都要处理
    return true;
  }

  @Override
  @Transactional
  public void init() {
    this.initFrameworkEndpoint();
  }
  
  /**
   * 检测当前在spring mvc中完成注册的各个requestMapping是否已经被写入到表单引擎的功能表中<br>
   * 如果没有完成写入，则在这里进行判定后写入
   */
  private void initFrameworkEndpoint() {
    /*
     * 操作步骤为
     * 1：遍历目前已经注册的requestMapping信息，并生成url$method这种形式的list
     * 2：取得当前competence数据表中所有的功能信息 并生成url$method这种形式的list
     * 3：比较两个list，并从第一个集合中取出那些不存在于数据库中的信息，最后进行添加操作
     * */
    LOGGER.info("正在检测和调整requestMapping和数据库中Competence的对应关系======");
    Map<RequestMappingInfo , HandlerMethod> pathMapping = frameworkEndpointHandler.getHandlerMethods();
    // 1、=====
    Set<String> currentRequstUrls = Sets.newHashSet();
    Set<RequestMappingInfo> requestMappings = pathMapping.keySet();
    for (RequestMappingInfo requestMappingInfo : requestMappings) {
      Set<RequestMethod> requestMethods = requestMappingInfo.getMethodsCondition().getMethods();
      Set<String> patterns = requestMappingInfo.getPatternsCondition().getPatterns();
      // 注意，有可能开发人员在同一个requestMapping中定义了多个Mapping地址（但至少会有一个）
      String[] methodValues = requestMethods.stream().map(Object::toString).collect(Collectors.toList()).toArray(new String[]{});
      if (methodValues != null && methodValues.length > 0) {
        patterns.forEach(item -> currentRequstUrls.add(item + "+" + StringUtils.join(methodValues, "|")));
      }
    }
    
    // 2、====
    List<CompetenceEntity> dbCompetenes = this.competenceRepository.findAll();
    Set<String> dbRequstUrls = null;
    if(dbCompetenes != null) {
      dbRequstUrls = dbCompetenes.stream().map(item -> item.getResource() + "+" + item.getMethods()).collect(Collectors.toSet());
    } else {
      dbRequstUrls = Sets.newHashSet();
    }
    SetView<String> difference = Sets.difference(currentRequstUrls, dbRequstUrls);
    
    // 3、====
    Pattern pattern = Pattern.compile("\\{[\\w]+\\}");
    if(difference != null) {
      List<CompetenceEntity> newCompetences = difference.stream().map(item -> {
        String[] urlAndMethod = item.split("\\+");
        String url = urlAndMethod[0];
        String method = urlAndMethod[1];
        CompetenceEntity currentCompetence = new CompetenceEntity();
        currentCompetence.setComment(url);
        currentCompetence.setMethods(method);
        currentCompetence.setResource(url);
        currentCompetence.setSortIndex(100);
        currentCompetence.setViewItem(0);
        // 是否是一个动态路径
        if(pattern.matcher(url).find()) {
          currentCompetence.setExtractUri(1);
        } else {
          currentCompetence.setExtractUri(0);
        }
        return currentCompetence;
      }).collect(Collectors.toList());
      // 批量添加
      if(newCompetences !=null && !newCompetences.isEmpty()) {
        this.competenceRepository.saveAll(newCompetences);
        this.competenceRepository.flush();
      }
    }
  }
}
