package com.bizunited.platform.core.service.init;

import com.bizunited.platform.core.entity.CompetenceEntity;
import com.bizunited.platform.core.entity.RoleEntity;
import com.bizunited.platform.core.entity.UserEntity;
import com.bizunited.platform.core.repository.CompetenceRepository;
import com.bizunited.platform.core.repository.RoleRepository;
import com.bizunited.platform.core.repository.UserRepository;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.context.annotation.Lazy;
import org.springframework.security.crypto.password.PasswordEncoder;
import org.springframework.stereotype.Component;
import org.springframework.util.CollectionUtils;

import java.util.ArrayList;
import java.util.Date;
import java.util.List;
import java.util.stream.Collectors;

/**
 * 该初始化过程
 * @author yinwenjie
 */
@Component("rbacInitProcess")
public class RbacInitProcess implements InitProcessService {
  private static final String ADMIN_NAME = "admin";
  private static final String ADMIN_ROLE_NAME = "ADMIN";
  private static final String BASEROLE_NAME = "BASEROLE";
  @Autowired
  private RoleRepository roleRepository;
  @Autowired
  private UserRepository userRepository;
  @Autowired
  private CompetenceRepository competenceRepository;
  @Autowired
  @Qualifier("passwordEncoder")
  @Lazy
  private PasswordEncoder passwordEncoder;
  
  @Override
  public boolean doProcess() {
    // 如果当前系统中能够找到一个名叫BASEROLE的角色，就认为整个基础权限的初始化做完了，那么就没有必要继续做了
    return this.roleRepository.findByRoleName(ADMIN_ROLE_NAME) == null || this.roleRepository.findByRoleName(ADMIN_NAME) == null;
  }
  
  @Override
  public int sort() {
    // 要求在系统启动时，第一时间检验并完成权限基本信息的初始化
    return 1;
  }

  @Override
  public boolean stopOnException() {
    // 如果该初始化过程出现异常，则直接抛出，并终止后续处理
    return true;
  }

  @Override
  public void init() {
    // 如果条件成立，则初始化基本角色
    if (this.roleRepository.findByRoleName(BASEROLE_NAME) == null) {
      this.initBaseRole();
      this.initBaseRoleCompetence();
    }
    // 如果条件成立，则初始化管理员角色
    if (this.roleRepository.findByRoleName(ADMIN_ROLE_NAME) == null) {
      this.initAdminRole();
    }
    // 如果条件成立，则初始化管理员
    if (this.userRepository.findByAccount(ADMIN_NAME) == null) {
      this.initAdmin();
    }
  }
  
  /** 初始化基本角色 */
  private void initBaseRole() {
    /*
     * 向系统初始化BASEROLE角色信息
     * 保证系统中至少固定持有BASEROLE（用户基础角色）角色
     * */
    RoleEntity role = new RoleEntity();
    role.setComment("用户基础角色");
    role.setCreateDate(new Date());
    role.setRoleName(BASEROLE_NAME);
    role.setRoleCode(BASEROLE_NAME);
    role.setTstatus(1);
    role.setIsDeny(true);
    this.roleRepository.saveAndFlush(role);
  }

  /** 初始化用户基础角色的功能 */
  private void initBaseRoleCompetence() {
    RoleEntity baserole = roleRepository.findByRoleName(BASEROLE_NAME);
    Validate.notNull(baserole, "初始化用户基础角色时未找到基础角色!");
    ArrayList<String> resources = new ArrayList<>();
    
    /*
     * {
     * "登录接口":"/v1/rbac/login",
     * "图片上传路径":/v1/nebula/files/{subsystem}/fileImageUpload
     * "文件上传路径":/v1/nebula/files/{subsystem}/fileUpload
     * "文件base64上传路径"：/v1/nebula/files/{subsystem}/fileUploadBase64
     * "获取当前用户菜单栏权限":"/v1/nebula/competences/findByCurrentUser",
     * "查询当前页面用户的按钮权限":"/v1/nebula/competences/findByUrlResource",
     * "用户id查询关联的组织机构":"/v1/nebula/orgs/findOrgByUserId"
     * "根据用户id查询关联的岗位":"/v1/nebula/position/findByUserId"
     * "根据用户id查询用户信息":"/v1/nebula/users/findByUserId"
     * "根据key查询环境变量":"/v1/nebula/environmentVariables/findByKey"
     * "获取登录人信息":"/v1/nebula/users/findByPrincipal"
     * "获取主题":"/v1/nebula/theme/findTheme"
     * "根据ID获取用户信息":"/v1/nebula/users/findByUserId"
     * "根据用户ID获取组织":"/v1/nebula/orgs/findOrgByUserId"
     * "根据key获取系统变量":"/v1/nebula/environmentVariables/findByKey"
     * "按照条件查询表单模板列表":"/v1/kuiper/templates/findByConditions"
     * "查询管理员权限用户":"/v1/kuiper/templates/maintainer/loadAuthorProps"
     * "根据模板id查询模板详情":"/v1/kuiper/templates/findDetailsById"
     * "根据模板id查询布局信息":"/v1/kuiper/templateLayouts/findDetailsByTemplateId"
     * "根据模板id查询事件":"/v1/kuiper/templateEvents/findByTemplateId"
     * "根据模板id查询可见性":"/v1/kuiper/templatevisibilities/findDetailsByTemplateId"
     * "查询所有表单样式":"/v1/kuiper/templateStyles/findAll"
     * "根据条件查询列表功能列表":"/v1/kuiper/listTemplates/findByConditions"
     * "根据id查询列表功能详细信息":"/v1/kuiper/listTemplates/findDetailsById"
     * "根据id查询列表功能配置内容":"/v1/kuiper/listTemplates/findConentById"
     * "根据数据视图id执行数据视图":"/v1/kuiper/dataViews/mainExecuteByPage"
     * "流程模板列表":"/v1/titan/templates/findByConditions"
     * "查询流程模板详情":"/v1/titan/templates/findDetailsById"
     * "查询流程监听器列表":"/v1/titan/listeners/findByConditions"
     * "按钮功能操作功能列表":"/v1/titan/template/node/permissions/findByConditions"
     * "添加按钮功能":"/v1/titan/template/node/permissions"
     * "查询可发起的流程列表":"/v1/titan/templates/findStartableByConditions"
     * "发起的流程":"/v1/titan/instances/startProcess"
     * "根据流程id，查询流程详情":"/v1/titan/instances/findDetailsByProcessInstanceId"
     * "根据流程id，节点id查询流程":"/v1/titan/tasks/findTaskInfoByInstanceIdAndTaskKey"
     * "根据流程id查询流程留言":"/v1/titan/instance/msgs/findByProcessInstanceId"
     * "根据流程id查询流程图详情":"/v1/titan/instances/findImageInfoByProcessInstanceId"
     * "我发起的流程列表":"/v1/titan/instances/findByConditions"
     * "我的已办":"/v1/titan/tasks/findDoneByConditions"
     * "我的待办":"/v1/titan/tasks/findMyTasksByConditions"
     * "我的抄送":"/v1/titan/carbonCopies/findMyByConditions"
     * "我的实例列表":"/v1/titan/instances/findByConditions"
     * "创建、修改表单模板":"/v1/kuiper/templates"
     * "表单引擎列表":"/template"
     * "列表功能":"/listTemplate"
     * "流程模板列表":"/flowcontrol"
     * "监听器管理":"/monitor"
     * "按钮功能操作":"/operation"
     * "我可发起的流程":"/flowInstance/useableFlowTemp"
     * "我发起的流程":"/flowInstance/launchBySelf"
     * "我的已办":"/flowInstance/done"
     * "我的待办":"/flowInstance/waiting"
     * "我的抄送":"/flowInstance/copy"
     * "流程实例列表":"/flowInstance/list"
     * }
     * */
    resources.add("/v1/rbac/login");
    resources.add("/v1/nebula/files/{subsystem}/fileImageUpload");
    resources.add("/v1/nebula/files/{subsystem}/fileUpload");
    resources.add("/v1/nebula/files/{subsystem}/fileUploadBase64");
    resources.add("/v1/nebula/competences/findByCurrentUser");
    resources.add("/v1/nebula/competences/findByUrlResource");
    resources.add("/v1/nebula/orgs/findOrgByUserId");
    resources.add("/v1/nebula/position/findByUserId");
    resources.add("/v1/nebula/users/findByUserId");
    resources.add("/v1/nebula/environmentVariables/findByKey");
    resources.add("/v1/nebula/users/findByPrincipal");
    resources.add("/v1/nebula/theme/findTheme");
    resources.add("/v1/kuiper/templates/findByConditions");
    resources.add("/v1/kuiper/templates/maintainer/loadAuthorProps");
    resources.add("/v1/kuiper/templates/findDetailsById");
    resources.add("/v1/kuiper/templateLayouts/findDetailsByTemplateId");
    resources.add("/v1/kuiper/templateEvents/findByTemplateId");
    resources.add("/v1/kuiper/templatevisibilities/findDetailsByTemplateId");
    resources.add("/v1/kuiper/templateStyles/findAll");
    resources.add("/v1/kuiper/listTemplates/findByConditions");
    resources.add("/v1/kuiper/listTemplates/findDetailsById");
    resources.add("/v1/kuiper/listTemplates/findConentById");
    resources.add("/v1/kuiper/dataViews/mainExecuteByPage");
    resources.add("/v1/titan/templates/findByConditions");
    resources.add("/v1/titan/templates/findDetailsById");
    resources.add("/v1/titan/listeners/findByConditions");
    resources.add("/v1/titan/template/node/permissions/findByConditions");
    resources.add("/v1/titan/template/node/permissions");
    resources.add("/v1/titan/templates/findStartableByConditions");
    resources.add("/v1/titan/instances/startProcess");
    resources.add("/v1/titan/instances/findDetailsByProcessInstanceId");
    resources.add("/v1/titan/tasks/findTaskInfoByInstanceIdAndTaskKey");
    resources.add("/v1/titan/instance/msgs/findByProcessInstanceId");
    resources.add("/v1/titan/instances/findImageInfoByProcessInstanceId");
    resources.add("/v1/titan/instances/findByConditions");
    resources.add("/v1/titan/tasks/findDoneByConditions");
    resources.add("/v1/titan/tasks/findMyTasksByConditions");
    resources.add("/v1/titan/carbonCopies/findMyByConditions");
    resources.add("/v1/kuiper/templates");
    resources.add("/template");
    resources.add("/listTemplate");
    resources.add("/flowcontrol");
    resources.add("/monitor");
    resources.add("/operation");
    resources.add("/flowInstance/useableFlowTemp");
    resources.add("/flowInstance/launchBySelf");
    resources.add("/flowInstance/done");
    resources.add("/flowInstance/waiting");
    resources.add("/flowInstance/copy");
    resources.add("/flowInstance/list");
    resources.add("/v1/kuiper/dataViews/mainExecute");
    resources.add("/v1/kuiper/frontFiles/findByConditions");
    resources.add("/v1/kuiper/instances/findByConditions");

    for(String resource : resources){
      List<CompetenceEntity> competenceEntities = competenceRepository.findByResource(resource);
      if (CollectionUtils.isEmpty(competenceEntities)) {
        continue;
      }

      List<String> ids = competenceEntities.stream().map(p -> p.getId()).collect(Collectors.toList());
      for (String id : ids) {
        if(competenceRepository.countByRoleIdAndCompetenceId(baserole.getId(), id) == 0) {
          competenceRepository.bindCompetence(baserole.getId(), id);
        }
      }
    }
  }
  
  /**
   * 初始化管理员角色
   */
  private void initAdminRole() {
    /*
     * 向系统初始化role角色信息
     * 保证系统中至少固定持有ADMIN（管理员）角色
     * */
    RoleEntity role = new RoleEntity();
    role.setComment("管理员角色");
    role.setCreateDate(new Date());
    role.setRoleName(ADMIN_ROLE_NAME);
    role.setRoleCode(ADMIN_ROLE_NAME);
    role.setTstatus(1);
    role.setIsDeny(true);
    this.roleRepository.saveAndFlush(role);
  }

  /** 初始化系统管理员 */
  private void initAdmin() {
    // 保证系统中至少有一个系统管理员
    UserEntity admin = new UserEntity();
    admin.setAccount(ADMIN_NAME);
    admin.setGender(0);
    admin.setPassword(this.passwordEncoder.encode("12345678"));
    admin.setPhone("12345678");
    admin.setUserName("后台超级管理员");
    admin.setUseStatus(1);
    this.userRepository.saveAndFlush(admin);
    // 绑定角色
    RoleEntity adminRole = this.roleRepository.findByRoleName(ADMIN_ROLE_NAME);
    this.roleRepository.bindUser(admin.getId(), adminRole.getId());
  }

}
