package com.bizunited.platform.core.service.internal;

import com.bizunited.platform.core.common.utils.ChineseCharUtil;
import com.bizunited.platform.core.entity.CodeRuleEntity;
import com.bizunited.platform.core.entity.ScriptEntity;
import com.bizunited.platform.core.entity.UserEntity;
import com.bizunited.platform.core.repository.CodeRuleRepository;
import com.bizunited.platform.core.service.CodeRuleService;
import com.bizunited.platform.core.service.ScriptService;
import com.bizunited.platform.core.service.invoke.InvokeProxyException;
import com.bizunited.platform.rbac.server.service.UserService;
import com.bizunited.platform.rbac.server.vo.UserVo;
import com.google.common.collect.Sets;

import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.domain.Pageable;
import org.springframework.security.core.Authentication;
import org.springframework.security.core.context.SecurityContext;
import org.springframework.security.core.context.SecurityContextHolder;
import org.springframework.stereotype.Service;

import javax.transaction.Transactional;
import java.util.Date;
import java.util.HashMap;
import java.util.Map;
import java.util.Optional;
import java.util.Set;

/**
 * CodeRuleServiceImpl
 *
 * @description:
 * @author: yanwe
 * @date: 12/Jun/2019 15:44
 */
@Service("CodeRuleServiceImpl")
public class CodeRuleServiceImpl implements CodeRuleService {

  @Autowired private CodeRuleRepository codeRuleRepository;
  @Autowired private UserService userService;
  @Autowired private ScriptService scriptService;

  @Override
  @Transactional
  public CodeRuleEntity create(CodeRuleEntity entity) {
    this.validEntity(entity, true);
    UserEntity userEntity = this.getAccount();
    entity.setCreateUser(userEntity);
    entity.setModifyUser(userEntity);
    entity.setCreateTime(new Date());
    entity.setModifyTime(new Date());
    return codeRuleRepository.save(entity);
  }

  @Override
  @Transactional
  public CodeRuleEntity update(CodeRuleEntity entity) {
    this.validEntity(entity, false);
    UserEntity userEntity = this.getAccount();
    Validate.notBlank(entity.getId() , "当前业务编码规则的ID信息必须传入");
    Optional<CodeRuleEntity> exsit = codeRuleRepository.findById(entity.getId());
    Validate.isTrue(exsit.isPresent(), "编码生成规则未查询到！");
    CodeRuleEntity current = exsit.orElse(null);
    current.setDescription(entity.getDescription());
    current.setLanguage(entity.getLanguage());
    current.setRuleType(entity.getRuleType());
    current.setRuleName(entity.getRuleName());
    current.setModifyUser(userEntity);
    current.setModifyTime(new Date());
    return codeRuleRepository.save(current);
  }

  @Override
  @Transactional
  public CodeRuleEntity updateContent(String ruleId, String content) {
    Validate.notBlank(ruleId, "编码生成规则ID不能为空！");
    Validate.notBlank(content, "脚本内容不能为空！");
    Optional<CodeRuleEntity> exsit = codeRuleRepository.findById(ruleId);
    Validate.isTrue(exsit.isPresent(), "编码生成规则未查询到！");
    CodeRuleEntity entity = exsit.get();
    ScriptEntity scriptEntity = entity.getScript();
    if (null == scriptEntity) {
      // 新增脚本内容
      ScriptEntity script = new ScriptEntity();
      script.setLanguage(entity.getLanguage());
      // 默认保存的脚本实体名称为： "cgr-编码生成编码"
      script.setName(String.format("%s%s", "cgr", entity.getRuleCode()));
      script = this.scriptService.create(script, content);
      entity.setScript(script);
    } else {
      // 修改脚本内容
      scriptEntity = this.scriptService.update(scriptEntity, content);
      entity.setScript(scriptEntity);
    }
    return codeRuleRepository.save(entity);
  }

  @Override
  public String invoke(String ruleCode) {
    Validate.notBlank(ruleCode,"规则编码不能为空！");

    CodeRuleEntity codeRuleEntity = codeRuleRepository.findDetailByCode(ruleCode);
    Validate.notNull(codeRuleEntity,"未根据该编码查询到编码规则！");
    ScriptEntity scriptEntity = codeRuleEntity.getScript();
    Validate.notNull(scriptEntity,"未查询到该规则绑定的脚本信息！");
    Validate.notBlank(scriptEntity.getId(),"脚本id不能为空！");

    try{
      Map<String,Object> result = scriptService.invoke(scriptEntity.getId(),new HashMap<>());
      String resultStr = String.valueOf(result.get("resultKey"));
      return resultStr;
    }catch (InvokeProxyException e){
      throw new IllegalArgumentException("执行生成编码脚本错误！");
    }
  }

  /**
   * 验证新增/修改实体
   *
   * @param entity
   * @param isCreate 是否新增
   */
  private void validEntity(CodeRuleEntity entity, Boolean isCreate) {
    Validate.notNull(entity, "编码生成规则基础信息不能为空！");
    Validate.notBlank(entity.getRuleName(), "编码生成规则名称不能为空!");
    Validate.notBlank(entity.getRuleCode(), "编码生成规则编码不能为空!");
    Validate.isTrue(!ChineseCharUtil.hasChinese(entity.getRuleCode()), "编码生成规则编码不能含有中文！");
    // 验证编码是否重复
    CodeRuleEntity existName =
        codeRuleRepository.findByRuleName(entity.getRuleName());
    CodeRuleEntity existCode =
        codeRuleRepository.findByRuleCode(entity.getRuleCode());
    if (isCreate) {
      Validate.isTrue(StringUtils.isBlank(entity.getId()), "新增时ID必须为空！");
      Validate.isTrue(null == existName, "编码规则名称已有重复，请检查！");
      Validate.isTrue(null == existCode, "编码规则编码已有重复，请检查！");
    } else {
      Validate.notBlank(entity.getId(), "修改时id不能为空！");
      Optional<CodeRuleEntity> op = codeRuleRepository.findById(entity.getId());
      CodeRuleEntity oldEntity = op.orElse(null);
      Validate.notNull(oldEntity, "未根据ID查询到编码生成规则实体！");
      Validate.isTrue(StringUtils.equals(entity.getRuleCode(),oldEntity.getRuleCode()),"规则编码不能修改！");
      Validate.isTrue(StringUtils.equals(entity.getRuleName(),oldEntity.getRuleName()),"规则编码不能修改！");
    }
  }

  /**
   * 获取当前登录账户ID，注入新的UserEntity
   * @return
   */
  private UserEntity getAccount(){
    // 获取当前登录账户
    SecurityContext securityContext = SecurityContextHolder.getContext();
    Validate.notNull(securityContext, "未发现任何用户权限信息!!");
    Authentication authentication = securityContext.getAuthentication();
    Validate.notNull(authentication, "未发现任何用户登录信息!!");
    UserVo userVo = userService.findByAccount(authentication.getName());
    Validate.notNull(userVo, "未获取到当前登录账号！");
    UserEntity userEntity = new UserEntity();
    userEntity.setId(userVo.getId());
    return userEntity;
  }

  @Override
  public String findContentByCode(String code) {
  
    Validate.notBlank(code, "编码生成规则的业务编号不能为空！");
    CodeRuleEntity entity = codeRuleRepository.findDetailByCode(code);
    if (entity.getScript() == null) {
      return null;
    }
    String scriptId = entity.getScript().getId();
    String content = scriptService.findContentById(scriptId);
    return content;
  }

  @Override
  public Page<CodeRuleEntity> findByConditions(
      Pageable pageable, String ruleName, String ruleCode, String ruleType, String language) {
    Map<String, Object> params = new HashMap<>();
    if (StringUtils.isNotEmpty(ruleName)) {
      params.put("ruleName", ruleName);
    }
    if (StringUtils.isNotEmpty(ruleCode)) {
      params.put("ruleCode", ruleCode);
    }
    if (StringUtils.isNotEmpty(ruleType)) {
      params.put("ruleType", ruleType);
    }
    if (StringUtils.isNotEmpty(language)) {
      params.put("language", language);
    }
    if (null == pageable) {
      pageable = PageRequest.of(0, 50);
    }
    return codeRuleRepository.findByConditions(pageable, params);
  }

  @Override
  public Set<CodeRuleEntity> findDetailsByIds(String[] ids) {
    if(ids == null || ids.length <= 0) { 
      return Sets.newHashSet();
    }
    return this.codeRuleRepository.findDetailsByIds(ids);
  }

  @Override
  public int countByIds(String[] ids) {
    if(ids == null || ids.length <= 0) {
      return 0;
    }
    return this.codeRuleRepository.countByIds(ids);
  }

  @Override
  public CodeRuleEntity findByRuleCode(String ruleCode) {
    if(StringUtils.isBlank(ruleCode)){
      return null;
    }
    return codeRuleRepository.findByRuleCode(ruleCode);
  }

  @Override
  public CodeRuleEntity findByRuleName(String ruleName) {
    if(StringUtils.isBlank(ruleName)){
      return null;
    }
    return codeRuleRepository.findByRuleName(ruleName);
  }
}
