package com.bizunited.platform.core.service.internal;

import com.bizunited.platform.core.entity.DictCategoryEntity;
import com.bizunited.platform.core.entity.DictEntity;
import com.bizunited.platform.core.entity.DictItemEntity;
import com.bizunited.platform.core.repository.DictItemRepository;
import com.bizunited.platform.core.repository.DictRepository;
import com.bizunited.platform.core.service.DictCategoryService;
import com.bizunited.platform.core.service.DictService;
import com.google.common.collect.Sets;

import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.domain.Pageable;
import org.springframework.stereotype.Service;

import javax.transaction.Transactional;
import java.util.*;

/**
 * DictServiceImpl
 *
 * @description:
 * @author: yanwe
 * @date: 11/Feb/2019 10:33
 */
@Service("dictServiceImpl")
public class DictServiceImpl implements DictService {
  private static final String ERROR_CODE_MESS = "查询字典编码不能为空！";
  private static final String ERROR_MESS = "未找到该编码匹配字典！";
  @Autowired 
  private DictRepository dictRepository;
  @Autowired 
  private DictItemRepository dictItemRepository;
  @Autowired 
  private DictCategoryService dictCategoryService;

  @Override
  public List<DictEntity> findAll() {
    return dictRepository.findAll();
  }

  @Override
  public List<DictEntity> findByStatus(Boolean dictStatus) {
    return dictRepository.findByDictStatus(dictStatus);
  }

  @Override
  public Page<DictEntity> findByConditions(
      String dictCode, String dictTitle, String dictType, Boolean dictStatus, Pageable pageable) {
    Map<String, Object> conditions = new HashMap<>();
    if (StringUtils.isNotBlank(dictCode)) {
      conditions.put("dictCode", dictCode);
    }
    if (StringUtils.isNotBlank(dictTitle)) {
      conditions.put("dictTitle", dictTitle);
    }
    if (StringUtils.isNotBlank(dictType)) {
      conditions.put("dictType", dictType);
    }
    // 如果为空则默认查询上架状态的字典
    if (dictStatus != null) {
      conditions.put("dictStatus", dictStatus);
    }

    if (pageable == null) {
      pageable = PageRequest.of(0, 50);
    }
    return dictRepository.queryPage(pageable, conditions);
  }

  @Override
  public Set<DictEntity> findDetailsByIds(String[] dictIds) {
    if(dictIds == null || dictIds.length == 0) {
      return Sets.newHashSet();
    }
    
    return this.dictRepository.findDetailsByIds(dictIds);
  }

  @Override
  public int countByIds(String[] ids) {
    if(ids == null || ids.length == 0) {
      return 0;
    }
    return this.dictRepository.countByIds(ids);
  }

  @Override
  @Transactional
  public DictEntity create(DictEntity dictEntity, String categoryId) {
    this.validEntity(dictEntity, categoryId);
    DictCategoryEntity categoryEntity = dictCategoryService.findById(categoryId);
    // 判断CODE是否有重复
    DictEntity existDict = dictRepository.findByDictCode(dictEntity.getDictCode());
    Validate.isTrue(null == existDict, "该编码已有重复！");
    // 新增字典默认明细项数量为0
    dictEntity.setItemNum(0);
    // 默认启用状态为FALSE
    dictEntity.setDictStatus(false);
    dictEntity.setCategory(categoryEntity);
    return dictRepository.saveAndFlush(dictEntity);
  }

  @Override
  @Transactional
  public DictEntity update(DictEntity dictEntity, String categoryId) {
    this.validEntity(dictEntity, categoryId);
    DictCategoryEntity categoryEntity = dictCategoryService.findById(categoryId);
    Optional<DictEntity> op = dictRepository.findById(dictEntity.getId());
    DictEntity existDict = op.orElse(null);
    Validate.notNull(existDict, "该字典不存在，请检查！");

    Validate.isTrue(!existDict.getDictStatus(), "该字典已经启用，不可被编辑");
    Validate.notNull(existDict, "未找到改修改字典");
    existDict.setDescription(dictEntity.getDescription());
    existDict.setDictType(dictEntity.getDictType());
    existDict.setDictTitle(dictEntity.getDictTitle());
    existDict.setCategory(categoryEntity);
    return dictRepository.saveAndFlush(existDict);
  }

  @Override
  @Transactional
  public DictEntity upgrade(String newDictCode, String dictCode) {
    Validate.notBlank(dictCode, ERROR_CODE_MESS);
    DictEntity dictEntity = dictRepository.findByDictCode(dictCode);
    Validate.notNull(dictEntity, ERROR_MESS);
    // 复制字典
    DictEntity newDict = new DictEntity();
    newDict.setDictCode(newDictCode);
    newDict.setDictTitle(dictEntity.getDictTitle());
    newDict.setItemNum(dictEntity.getItemNum());
    newDict.setDescription(dictEntity.getDescription());
    newDict.setDictType(dictEntity.getDictType());
    newDict.setDictStatus(false);
    newDict.setDictItemEntities(new HashSet<>());
    newDict.setCategory(dictEntity.getCategory());
    newDict = dictRepository.saveAndFlush(newDict);
    // 复制字典明细
    Set<DictItemEntity> newItems = new HashSet<>();
    for (DictItemEntity item : dictEntity.getDictItemEntities()) {
      DictItemEntity newItem = new DictItemEntity();
      newItem.setDictSort(item.getDictSort());
      newItem.setDictKey(item.getDictKey());
      newItem.setDictValue(item.getDictValue());
      newItem.setDictEntity(newDict);
      newItems.add(newItem);
    }
    dictItemRepository.saveAll(newItems);
    return newDict;
  }

  @Override
  @Transactional
  public DictEntity onshelf(String dictCode) {
    Validate.notBlank(dictCode, ERROR_CODE_MESS);
    DictEntity dictEntity = dictRepository.findByDictCode(dictCode);
    Validate.notNull(dictEntity, ERROR_MESS);
    dictEntity.setDictStatus(true);
    return dictRepository.saveAndFlush(dictEntity);
  }

  @Override
  @Transactional
  public void delete(String dictCode) {
    Validate.notBlank(dictCode, ERROR_CODE_MESS);
    DictEntity dictEntity = dictRepository.findByDictCode(dictCode);
    Validate.notNull(dictEntity, ERROR_MESS);
    Validate.isTrue(!dictEntity.getDictStatus(), "已上架的字典无法删除！");
    Set<DictItemEntity> dictItemEntities = dictEntity.getDictItemEntities();
    dictItemRepository.deleteAll(dictItemEntities);
    dictRepository.delete(dictEntity);
  }

  @Override
  public DictEntity findByDictCode(String dictCode) {
    if(StringUtils.isBlank(dictCode)){
      return null;
    }
    return dictRepository.findByDictCode(dictCode);
  }

  /**
   * 验证实体
   *
   * @param dictEntity
   * @param categoryId
   */
  private void validEntity(DictEntity dictEntity, String categoryId) {
    Validate.notNull(dictEntity, "输入字典不可为空!");
    // 判断该字典是否已经是启用状态，若为启用，则不可再编辑
    Validate.notBlank(dictEntity.getDictCode(), "dict code 不能为空!");
    Validate.notBlank(dictEntity.getDictTitle(), "dict title 不能为空!");
    Validate.notBlank(categoryId, "字典分组ID不能为空");
    DictCategoryEntity categoryEntity = dictCategoryService.findById(categoryId);
    Validate.notNull(categoryEntity, "未查询到字典分组");
  }
}
