package com.bizunited.platform.core.service.internal;

import com.bizunited.platform.core.common.utils.ChineseCharUtil;
import com.bizunited.platform.core.entity.EnvironmentVariableEntity;
import com.bizunited.platform.core.entity.UserEntity;
import com.bizunited.platform.core.repository.EnvironmentVariableRepository;
import com.bizunited.platform.core.service.EnvironmentVariableService;

import com.bizunited.platform.rbac.server.service.UserService;
import com.bizunited.platform.rbac.server.vo.UserVo;
import com.google.common.collect.Sets;

import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.domain.Pageable;
import org.springframework.security.core.Authentication;
import org.springframework.security.core.context.SecurityContext;
import org.springframework.security.core.context.SecurityContextHolder;
import org.springframework.stereotype.Service;

import javax.transaction.Transactional;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;

/**
 * EnvironmentVariableServiceImpl 参数
 *
 * @description:
 * @author: yanwe
 * @date: 15/May/2019 14:34
 */
@Service("EnvironmentVariableServiceImpl")
public class EnvironmentVariableServiceImpl implements EnvironmentVariableService {

  @Autowired private EnvironmentVariableRepository environmentVariableRepository;
  @Autowired private UserService userService;

  @Override
  public List<EnvironmentVariableEntity> findAll() {
    return environmentVariableRepository.findAll();
  }

  @Override
  public Page<EnvironmentVariableEntity> findByConditions(String paramCode, String paramKey, String paramValue, String paramType, Integer paramStatus, Pageable pageable) {
    Map<String,Object> conditions = new HashMap<>();
    if(StringUtils.isNotEmpty(paramCode)){
      conditions.put("paramCode",paramCode);
    }
    if(StringUtils.isNotEmpty(paramKey)){
      conditions.put("paramKey",paramKey);
    }
    if(StringUtils.isNotEmpty(paramValue)){
      conditions.put("paramValue",paramValue);
    }
    if(StringUtils.isNotEmpty(paramType)){
      conditions.put("paramType",paramType);
    }
    if(null != paramStatus){
      conditions.put("paramStatus",paramStatus);
    }
    if(pageable ==null){
      pageable = PageRequest.of(0,50);
    }
    return environmentVariableRepository.queryPage(pageable,conditions);
  }

  @Override
  public EnvironmentVariableEntity findByKey(String key) {
    if(StringUtils.isBlank(key)) {
      return null;
    }
    return environmentVariableRepository.findByParamKey(key);
  }

  @Override
  public EnvironmentVariableEntity findByCode(String code) {
    if(StringUtils.isBlank(code)) {
      return null;
    }
    return environmentVariableRepository.findByParamCode(code);
  }

  @Override
  public Set<EnvironmentVariableEntity> findDetailsByIds(String[] ids) {
    if(ids == null || ids.length == 0) {
      return Sets.newHashSet();
    }
    return this.environmentVariableRepository.findDetailsByIds(ids);
  }

  @Override
  @Transactional
  public EnvironmentVariableEntity enable(String id) {
    Validate.notBlank(id,"启用时必须传入ID");
    Optional<EnvironmentVariableEntity> op = environmentVariableRepository.findById(id);
    EnvironmentVariableEntity entity = op.orElse(null);
    Validate.notNull(entity,"未找到该参数！");

    entity.setParamStatus(1);
    return environmentVariableRepository.save(entity);
  }

  @Override
  @Transactional
  public EnvironmentVariableEntity disable(String id) {
    Validate.notBlank(id,"禁用时必须传入ID");
    Optional<EnvironmentVariableEntity> op = environmentVariableRepository.findById(id);
    EnvironmentVariableEntity entity = op.orElse(null);
    Validate.notNull(entity,"未找到该参数！");

    entity.setParamStatus(0);
    return environmentVariableRepository.save(entity);
  }

  @Override
  @Transactional
  public EnvironmentVariableEntity save(EnvironmentVariableEntity entity) {
    Validate.notNull(entity, "参数不能为空！");
    Validate.notBlank(entity.getParamKey(), "参数的键不能为空！");
    Validate.notBlank(entity.getParamValue(),"参数的值不能为Null！");
    Validate.notBlank(entity.getParamCode(),"参数的编码不能为Null！");
    Validate.isTrue(!ChineseCharUtil.hasChinese(entity.getParamKey()),"参数键不能含有中文！");
    Validate.notBlank(entity.getParamType(),"参数类型不能为空！");
    Validate.notNull(entity.getParamStatus(),"参数状态不能为空！");
    //获取当前登录账户
    UserEntity userEntity = this.getAccount();
    if (StringUtils.isEmpty(entity.getId())) {
      // 新增
      EnvironmentVariableEntity exist = environmentVariableRepository.findByParamKey(entity.getParamKey());
      Validate.isTrue(null == exist, "保存参数的键已有重复，请检查！");
      entity.setCreateDate(new Date());
      entity.setModifyDate(new Date());
      entity.setCreateUser(userEntity);
      entity.setModifyUser(userEntity);
      return environmentVariableRepository.save(entity);
    } else {
      // 修改
      Optional<EnvironmentVariableEntity> op = environmentVariableRepository.findById(entity.getId());
      EnvironmentVariableEntity exist = op.orElse(null);
      Validate.notNull(exist,"根据ID未获取到该参数！");

      //验证修改后的名称，是否有重复
      EnvironmentVariableEntity existCode = environmentVariableRepository.findByParamCode(entity.getParamCode());
      Boolean notSameCode = null == existCode || exist.getParamCode().equals(existCode.getParamCode());
      Validate.isTrue(notSameCode,"修改后的参数编码已有重复，请检查！");
      //验证修改后的键，是否有重复
      EnvironmentVariableEntity existKey = environmentVariableRepository.findByParamKey(entity.getParamKey());
      Boolean notSameKey = null == existKey || exist.getParamKey().equals(existKey.getParamKey());
      Validate.isTrue(notSameKey,"修改后的参数键已有重复，请检查！");
      // 其余的属性变更
      exist.setParamCode(entity.getParamCode());
      exist.setParamKey(entity.getParamKey());
      exist.setParamValue(entity.getParamValue());
      exist.setParamType(entity.getParamType());
      exist.setParamDesc(entity.getParamDesc());
      exist.setModifyUser(userEntity);
      exist.setModifyDate(new Date());
      return environmentVariableRepository.save(exist);
    }
  }

  @Override
  @Transactional(rollbackOn = Exception.class)
  public void delete(String id) {
    Validate.notBlank(id, "删除参数ID不能为空！");
    Optional<EnvironmentVariableEntity> op = environmentVariableRepository.findById(id);
    EnvironmentVariableEntity exist = op.orElse(null);
    Validate.notNull(exist,"未找到参数，请检查！");
    environmentVariableRepository.delete(exist);
  }


  /**
   * 获取当前登录账户
   * @return
   */
  private UserEntity getAccount(){
    //获取当前登录账户
    SecurityContext securityContext = SecurityContextHolder.getContext();
    Validate.notNull(securityContext , "未发现任何用户权限信息!!");
    Authentication authentication = securityContext.getAuthentication();
    Validate.notNull(authentication , "未发现任何用户登录信息!!");
    UserVo userVo = userService.findByAccount(authentication.getName());
    Validate.notNull(userVo,"未获取到当前登录账号！");
    UserEntity userEntity = new UserEntity();
    userEntity.setId(userVo.getId());
    return userEntity;
  }

  @Override
  public int countByIds(String[] ids) {
    if(ids == null || ids.length <= 0) {
      return 0;
    }
    return this.environmentVariableRepository.countByIds(ids);
  }
}
