package com.bizunited.platform.core.service.scheduler;

import java.security.Principal;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Date;
import java.util.HashMap;
import java.util.HashSet;
import java.util.LinkedHashSet;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ScheduledFuture;

import javax.transaction.Transactional;

import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.context.ApplicationContext;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.domain.Pageable;
import org.springframework.scheduling.concurrent.ThreadPoolTaskScheduler;
import org.springframework.scheduling.support.CronTrigger;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;

import com.bizunited.platform.core.entity.DynamicTaskSchedulerEntity;
import com.bizunited.platform.core.entity.ScriptEntity;
import com.bizunited.platform.core.entity.UserEntity;
import com.bizunited.platform.core.repository.DynamicTaskSchedulerRepository;
import com.bizunited.platform.core.service.DynamicTaskSchedulerService;
import com.bizunited.platform.core.service.NebulaToolkitService;
import com.bizunited.platform.core.service.ScriptService;
import com.bizunited.platform.rbac.server.service.UserService;
import com.bizunited.platform.rbac.server.vo.UserVo;

/**
 * 技术中台动态任务管理服务的实现
 * @author yinwenjie
 *
 */
@Service("DynamicTaskSchedulerServiceImpl")
public class DynamicTaskSchedulerServiceImpl implements DynamicTaskSchedulerService {
  @Autowired
  @Qualifier("platform_dynamicTaskScheduler")
  private ThreadPoolTaskScheduler threadPoolTaskScheduler; 
  @Autowired
  private DynamicTaskSchedulerRepository dynamicTaskSchedulerRepository; 
  @Autowired
  private ApplicationContext applicationContext; 
  @Autowired
  private NebulaToolkitService nebulaToolkitService; 
  @Autowired
  private ScriptService scriptService;
  @Autowired
  private UserService userService; 
  /**
   * 该映射关系用来记录当前的定时器任务运行情况
   */
  private Map<String, ScheduledFuture<?>> taskMapping = new ConcurrentHashMap<>();
  private static final String ERROR_MESS = "周期性任务执行表单式,填入值超过了限定长度(32)，请检查!";
  private static final String ERROR_MESS_CHARSET = "周期性表单时的中文,填入值超过了限定长度(256)，请检查!";
  
  @Override
  @Transactional
  public void interrupt(String taskCode) {
    Validate.notBlank(taskCode , "停止动态任务时，必须指定任务编号");
    DynamicTaskSchedulerEntity current =  this.dynamicTaskSchedulerRepository.findByTaskCode(taskCode);
    Validate.notNull(current , "未发现任务编号对应的任务信息，请检查!!");
    
    current.setWorkingStatus(1);
    this.dynamicTaskSchedulerRepository.save(current);
  }

  @Override
  @Transactional
  public void start(String taskCode) {
    Validate.notBlank(taskCode , "停止动态任务时，必须指定任务编号");
    DynamicTaskSchedulerEntity current =  this.dynamicTaskSchedulerRepository.findByTaskCode(taskCode);
    Validate.notNull(current , "未发现任务编号对应的任务信息，请检查!!");
    current.setWorkingStatus(0);
    current.setTstatus(1);
    this.dynamicTaskSchedulerRepository.save(current);
  }

  @Transactional
  public void loading() {
    /*
     * - 整个处理过程描述如下：
     * 1、初始化过程第一步是查询当前系统中状态为“有效任务”，且运行状态不为“人工停止”所有动态任务。
     * 2、使用applicationContext，在IOC容器中没有这个任务对应的bean的情况下，创建新的bean。如果已经存在，则忽略对于这个任务的处理
     * 3、根据这些任务bean，使用ThreadPoolTaskScheduler对象进行加载，并在加载完成后设
     * 4、这些启动的信息，其数据表的信息将被更新
     * 5、由于集群中的其它节点可能要求终止某些动态任务，所以还要对可能的终止任务进行操作
     * */
    
    // 1、=====
    // 工作状态不能包括“人工停止”
    int[] workingStatuses = new int[] {0 , 2};
    Set<DynamicTaskSchedulerEntity> tasks = dynamicTaskSchedulerRepository.findByTstatusAndWorkingStatus(1, workingStatuses);
    // 2、======
    List<String> runningTaskCodes = new ArrayList<>(100);
    List<String> invalidTaskCodes = new ArrayList<>(100);
    if(!CollectionUtils.isEmpty(tasks)) {
      Date currentTime = new Date();
      for (DynamicTaskSchedulerEntity task : tasks) {
        String taskCode = task.getTaskCode();
        Date validityTime = task.getValidityTime();
        // 如果条件成立，说明有效期已过，不再进行处理，且如果有任务正在运行，也必须取消
        if(validityTime != null && validityTime.getTime() < currentTime.getTime()) {
          ScheduledFuture<?> scheduledFuture = taskMapping.get(taskCode);
          if(scheduledFuture != null) {
            // true的意思是如果当前正在运行，则发送interrupt信号
            scheduledFuture.cancel(true);
            taskMapping.remove(taskCode);
          }
          invalidTaskCodes.add(taskCode);
          continue;
        }
        // 以下视情况进行添加
        // 如果条件成立，则无需进行添加，因为这个bean已经存在了，且当前数据库可能的不同步情况要进行修正
        if(taskMapping.get(taskCode) != null) {
          runningTaskCodes.add(taskCode);
          continue;
        }
        // 开始初始化该任务
        DynamicTaskSchedulerEntity currentTask = this.nebulaToolkitService.copyObjectByWhiteList(task, DynamicTaskSchedulerEntity.class, LinkedHashSet.class, LinkedList.class, new String[] {"script"});
        DynamicTask dynamicTask = new DynamicTask(currentTask , this.applicationContext);
        
        // 3、======这里注意，如果当前task已经在运行内存中了，就不需要再进行加载了
        // 如果条件成立，表示是一个一次性任务
        if(task.getTaskType() == 1) {
          Date executePoint = task.getExecutePoint();
          String cornExpression = task.getExecuteExpression();
          Validate.isTrue(!(executePoint == null && StringUtils.isBlank(cornExpression)) , "指定任务[%s]是一个一次性任务，但是却没有指定任务执行时间，请检查" , task.getTaskCode());
          ScheduledFuture<?> scheduledFuture = null;
          if(executePoint != null) {
            scheduledFuture = this.threadPoolTaskScheduler.schedule(dynamicTask, executePoint);
          } else {
            CronTrigger cronTrigger = new CronTrigger(cornExpression);
            scheduledFuture = this.threadPoolTaskScheduler.schedule(dynamicTask, cronTrigger);
          }
          this.taskMapping.put(taskCode, scheduledFuture);
          runningTaskCodes.add(taskCode);
        } 
        // 如果条件成立，标识是一个周期性任务
        else if(task.getTaskType() == 2) {
          String cornExpression = task.getExecuteExpression();
          Validate.isTrue(StringUtils.isNotBlank(cornExpression) , "指定动态任务[%s]是一个周期性任务，但是却没有指定有效的corn表达式，请检查" , task.getTaskCode());
          CronTrigger cronTrigger = new CronTrigger(cornExpression);
          ScheduledFuture<?> scheduledFuture = this.threadPoolTaskScheduler.schedule(dynamicTask, cronTrigger);
          this.taskMapping.put(taskCode, scheduledFuture);
          runningTaskCodes.add(taskCode);
        } 
        // 其它情况下，直接抛出异常
        else {
          throw new IllegalArgumentException("错误的动态任务类型[" + task.getTaskCode() + "]，请检查");
        }
      }
    }
    
    // 4、========进行数据库中的状态更新
    if(!invalidTaskCodes.isEmpty()) {
      this.dynamicTaskSchedulerRepository.updateInvalidByTaskCode(invalidTaskCodes.toArray(new String[] {}));
    }
    if(!runningTaskCodes.isEmpty()) {
      this.dynamicTaskSchedulerRepository.updateRunningByTaskCode(runningTaskCodes.toArray(new String[] {}));
    }
    
    // 5、========扫描在这个周期可能已经被其它进程操作，在数据库层中要求被停止的进程，这里将进行停止操作
    Set<DynamicTaskSchedulerEntity> statusTasks = this.dynamicTaskSchedulerRepository.findByTstatus(0);
    if(CollectionUtils.isEmpty(statusTasks)) {
      statusTasks = new HashSet<>();
    }
    Set<DynamicTaskSchedulerEntity> invalidTasks = this.dynamicTaskSchedulerRepository.findByWorkingStatus(new int[] {1});
    if(!CollectionUtils.isEmpty(invalidTasks)) {
      statusTasks.addAll(invalidTasks);
    }
    // 开始停止
    for (DynamicTaskSchedulerEntity statusTask : statusTasks) {
      String invalidTaskCode = statusTask.getTaskCode();
      ScheduledFuture<?> scheduledFuture = taskMapping.remove(invalidTaskCode);
      if(scheduledFuture != null) {
        scheduledFuture.cancel(true);
      }
    }
  }
  
  public void unloading() {
    /*
     * - 该方法用于停止当前正在进行的服务，一般情况下是当前进程失去了代表master的锁。处理过程为：
     * 1、直接通过taskMapping中的任务信息进行停止即可
     * */ 
    if(this.taskMapping.isEmpty()) {
      return;
    }
    Collection<ScheduledFuture<?>> tasks = this.taskMapping.values();
    for (ScheduledFuture<?> task : tasks) {
      task.cancel(true);
    }
  }

  @Override
  @Transactional
  public DynamicTaskSchedulerEntity create(DynamicTaskSchedulerEntity task , String scriptContent , Principal principal) {
    // 1、=========首先进行动态任务基本信息的添加
    Validate.notNull(task , "进行当前操作时，动态任务对象必须传入!!");
    String userAccount = principal.getName();
    UserVo uservo = userService.findByAccount(userAccount);
    UserEntity user = new UserEntity();
    user.setId(uservo.getId());
    Date currentTime = new Date();
    // 设置自填充值
    task.setCreator(user);
    task.setCreateTime(currentTime);
    task.setModifier(user);
    task.setLastModifyTime(currentTime);
    task.setTstatus(1);
    task.setWorkingStatus(0);
    // 进行对象信息验证
    this.createValidation(task);
    
    // 2、=======然后进行动态任务脚本内容的创建
    ScriptEntity script = new ScriptEntity();
    // 目前只能是groovy
    script.setLanguage("groovy");
    script.setName("dynamicTask_" + task.getTaskCode());
    script = this.scriptService.create(script, scriptContent);
    task.setScript(script);
    // 进行更新
    this.dynamicTaskSchedulerRepository.save(task);
    
    return task;
  }
  
  /**
   * 在创建一个新的DynamicTaskSchedulerEntity模型对象之前，检查对象各属性的正确性，其主键属性必须没有值
   */
  private void createValidation(DynamicTaskSchedulerEntity dynamicTaskSchedulerEntity) { 
    // 判定那些不能为null的输入值：条件为 caninsert = true，且nullable = false
    Validate.isTrue(StringUtils.isBlank(dynamicTaskSchedulerEntity.getId()), "添加信息时，当期信息的数据编号（主键）不能有值！");
    dynamicTaskSchedulerEntity.setId(null);
    Validate.notBlank(dynamicTaskSchedulerEntity.getTaskCode(), "任务唯一编号（只能由英文、数字、下杠构成）不能为空！");
    Validate.notBlank(dynamicTaskSchedulerEntity.getTaskDesc(), "任务中文说明不能为空！");
    Validate.notNull(dynamicTaskSchedulerEntity.getTaskType(), "任务类型不能为空！");
    Validate.notNull(dynamicTaskSchedulerEntity.getCreateTime(), "创建时间不能为空！");
    Validate.notNull(dynamicTaskSchedulerEntity.getLastModifyTime(), "最后修改时间不能为空！");
    // 验证长度，被验证的这些字段符合特征: 字段类型为String，且不为PK （注意连续空字符串的情况） 
    Validate.isTrue(dynamicTaskSchedulerEntity.getTaskCode() == null || dynamicTaskSchedulerEntity.getTaskCode().length() < 128 , "任务唯一编号（只能由英文、数字、下杠构成）,填入值超过了限定长度(128)，请检查!");
    Validate.isTrue(dynamicTaskSchedulerEntity.getTaskDesc() == null || dynamicTaskSchedulerEntity.getTaskDesc().length() < 256 , "任务中文说明,填入值超过了限定长度(256)，请检查!");
    // 如果当前是一次性任务
    Date currentTime = new Date();
    if(dynamicTaskSchedulerEntity.getTaskType() == 1) {
      Validate.notNull(dynamicTaskSchedulerEntity.getExecutePoint(), "一次性任务的开始时间必须填写");
      Validate.isTrue(dynamicTaskSchedulerEntity.getExecutePoint().getTime() > currentTime.getTime() , "一次性任务的开始时间，必须设定在当前时间点以后");
    }
    // 如果当前是周期性任务
    else if(dynamicTaskSchedulerEntity.getTaskType() == 2) {
      Validate.notBlank(dynamicTaskSchedulerEntity.getExecuteExpression() , "周期性任务必须给定有效的corn表达式!!");
      Validate.notBlank(dynamicTaskSchedulerEntity.getExpressionDesc() , "周期性任务必须传入corn表达式的中文描述!!");
      Validate.isTrue(dynamicTaskSchedulerEntity.getExecuteExpression() == null || dynamicTaskSchedulerEntity.getExecuteExpression().length() < 32 , ERROR_MESS);
      Validate.isTrue(dynamicTaskSchedulerEntity.getExpressionDesc() == null || dynamicTaskSchedulerEntity.getExpressionDesc().length() < 256 , ERROR_MESS_CHARSET);
    }
    // 其它情况报错
    else {
      throw new IllegalArgumentException("动态任务只能是一次性任务或者周期性任务，请检查!!");
    }
    DynamicTaskSchedulerEntity currentDynamicTaskSchedulerEntity = this.dynamicTaskSchedulerRepository.findByTaskCode(dynamicTaskSchedulerEntity.getTaskCode());
    Validate.isTrue(currentDynamicTaskSchedulerEntity == null, "任务唯一编号（只能由英文、数字、下杠构成）已存在,请检查");
  }

  @Override
  @Transactional
  public DynamicTaskSchedulerEntity update(DynamicTaskSchedulerEntity task, String scriptContent , Principal principal) {
    Validate.notNull(task , "进行当前操作时，动态任务对象必须传入!!");
    String userAccount = principal.getName();
    UserVo uservo = userService.findByAccount(userAccount);
    UserEntity user = new UserEntity();
    user.setId(uservo.getId());
    Date currentTime = new Date();
    // 开始检查
    this.updateValidation(task);
    
    // 1、===================更新基本信息
    String currentId = task.getId();
    Optional<DynamicTaskSchedulerEntity> op = this.dynamicTaskSchedulerRepository.findById(currentId);
    DynamicTaskSchedulerEntity currentTask = op.orElse(null);
    Validate.notNull(currentTask ,"未发现指定的原始模型对象信");

    // 如果当前任务的状态不是“人工停止”，则不允许进行信息修改，
    Validate.isTrue(currentTask.getTstatus() == 0 , "只有在调度任务被禁用/无效的情况下（tstatus == 0）,才能进行修改!!!");
    Validate.isTrue(currentTask.getWorkingStatus() == 1 , "只有首先停止任务运行（workingStatus == 1），才能进行修改!!");
    
    // 开始重新赋值——一般属性
    currentTask.setTaskDesc(task.getTaskDesc());
    currentTask.setExecutePoint(task.getExecutePoint());
    currentTask.setExecuteExpression(task.getExecuteExpression());
    currentTask.setExpressionDesc(task.getExpressionDesc());
    currentTask.setLastModifyTime(task.getLastModifyTime());
    currentTask.setValidityTime(task.getValidityTime());
    currentTask.setLastModifyTime(currentTime);
    currentTask.setModifier(user);
    
    // 2、===================更新脚本信息
    ScriptEntity script = task.getScript();
    Validate.notNull(script , "进行更新时，脚本内容必须传入!!");
    script = this.scriptService.update(script, scriptContent);
    task.setScript(script);
    this.dynamicTaskSchedulerRepository.saveAndFlush(currentTask);
    return currentTask;
  }
  /**
   * 在更新一个已有的DynamicTaskSchedulerEntity模型对象之前，该私有方法检查对象各属性的正确性，其id属性必须有值
   */
  private void updateValidation(DynamicTaskSchedulerEntity dynamicTaskSchedulerEntity) { 
    Validate.isTrue(!StringUtils.isBlank(dynamicTaskSchedulerEntity.getId()), "修改信息时，当期信息的数据编号（主键）必须有值！");
    // 基础信息判断，基本属性，需要满足not null
    Validate.notBlank(dynamicTaskSchedulerEntity.getTaskCode(), "任务唯一编号（只能由英文、数字、下杠构成）不能为空！");
    Validate.notBlank(dynamicTaskSchedulerEntity.getTaskDesc(), "任务中文说明不能为空！");
    Validate.notNull(dynamicTaskSchedulerEntity.getTaskType(), "任务类型不能为空！");
    // 如果当前是一次性任务
    Date currentTime = new Date();
    if(dynamicTaskSchedulerEntity.getTaskType() == 1) {
      Validate.notNull(dynamicTaskSchedulerEntity.getExecutePoint(), "一次性任务的开始时间必须填写");
      Validate.isTrue(dynamicTaskSchedulerEntity.getExecutePoint().getTime() > currentTime.getTime() , "一次性任务的开始时间，必须设定在当前时间点以后");
    }
    // 如果当前是周期性任务
    else if(dynamicTaskSchedulerEntity.getTaskType() == 2) {
      Validate.notBlank(dynamicTaskSchedulerEntity.getExecuteExpression() , "周期性任务必须给定有效的corn表达式!!");
      Validate.notBlank(dynamicTaskSchedulerEntity.getExpressionDesc() , "周期性任务必须传入corn表达式的中文描述!!");
      Validate.isTrue(dynamicTaskSchedulerEntity.getExecuteExpression() == null || dynamicTaskSchedulerEntity.getExecuteExpression().length() < 32 , ERROR_MESS);
      Validate.isTrue(dynamicTaskSchedulerEntity.getExpressionDesc() == null || dynamicTaskSchedulerEntity.getExpressionDesc().length() < 256 , ERROR_MESS_CHARSET);
    }
    // 其它情况报错
    else {
      throw new IllegalArgumentException("动态任务只能是一次性任务或者周期性任务，请检查!!");
    }
    
    // 重复性判断，基本属性，需要满足unique = true
    DynamicTaskSchedulerEntity currentForTaskCode = this.dynamicTaskSchedulerRepository.findByTaskCode(dynamicTaskSchedulerEntity.getTaskCode());
    Validate.isTrue(currentForTaskCode == null || StringUtils.equals(currentForTaskCode.getId() , dynamicTaskSchedulerEntity.getId()) , "任务唯一编号（只能由英文、数字、下杠构成）已存在,请检查"); 
    // 验证长度，被验证的这些字段符合特征: 字段类型为String，且不为PK，且canupdate = true
    Validate.isTrue(dynamicTaskSchedulerEntity.getTaskCode() == null || dynamicTaskSchedulerEntity.getTaskCode().length() < 128 , "任务唯一编号（只能由英文、数字、下杠构成）,填入值超过了限定长度(128)，请检查!");
    Validate.isTrue(dynamicTaskSchedulerEntity.getTaskDesc() == null || dynamicTaskSchedulerEntity.getTaskDesc().length() < 256 , "任务中文说明,填入值超过了限定长度(256)，请检查!");
    Validate.isTrue(dynamicTaskSchedulerEntity.getExecuteExpression() == null || dynamicTaskSchedulerEntity.getExecuteExpression().length() < 32 , ERROR_MESS);
    Validate.isTrue(dynamicTaskSchedulerEntity.getExpressionDesc() == null || dynamicTaskSchedulerEntity.getExpressionDesc().length() < 256 , ERROR_MESS_CHARSET);
  } 
  @Override
  @Transactional
  public void updateInvalidByTaskCode(String[] taskCodes) {
    Validate.notNull(taskCodes , "在进行强制停止时，至少传入一个动态定时任务的编号code!!");
    Validate.isTrue(taskCodes.length > 0 , "在进行强制停止时，至少传入一个动态定时任务的编号code!!");
    this.dynamicTaskSchedulerRepository.updateInvalidByTaskCode(taskCodes);
  }

  @Override
  public Set<DynamicTaskSchedulerEntity> findAll() {
    Set<DynamicTaskSchedulerEntity> results = this.dynamicTaskSchedulerRepository.findAlls();
    return results;
  }
  @Override
  public DynamicTaskSchedulerEntity findDetailsByTaskCode(String taskCode) {
    if(StringUtils.isBlank(taskCode)) {
      return null;
    }
    DynamicTaskSchedulerEntity task = this.dynamicTaskSchedulerRepository.findDetailsByTaskCode(taskCode);
    return task;
  }
  @Override
  public DynamicTaskSchedulerEntity findByTaskCode(String taskCode) {
    if(StringUtils.isBlank(taskCode)) {
      return null;
    }
    DynamicTaskSchedulerEntity task = this.dynamicTaskSchedulerRepository.findByTaskCode(taskCode);
    return task;
  }

  @Override
  public Page<DynamicTaskSchedulerEntity> findByConditions(Pageable pageable, Integer taskType, String taskCode, Integer tstatus, Integer workingStatus, String taskDesc) {
    Map<String,Object> conditions = new HashMap<>();
    if(taskType != null){
      conditions.put("taskType",taskType);
    }
    if(StringUtils.isNotEmpty(taskCode)){
      conditions.put("taskCode",taskCode);
    }
    if(tstatus != null){
      conditions.put("tstatus",tstatus);
    }
    if(workingStatus != null){
      conditions.put("workingStatus",workingStatus);
    }
    if(StringUtils.isNotEmpty(taskDesc)){
      conditions.put("taskDesc",taskDesc);
    }
    if(pageable ==null){
      pageable = PageRequest.of(0,50);
    }
    
    return this.dynamicTaskSchedulerRepository.queryPage(pageable, conditions);
  }
}