package com.bizunited.platform.core.service.security;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Date;
import java.util.HashMap;
import java.util.HashSet;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;

import com.bizunited.platform.core.entity.OrganizationEntity;
import com.bizunited.platform.core.entity.PositionEntity;
import com.bizunited.platform.core.entity.RoleEntity;
import com.bizunited.platform.core.entity.UserEntity;
import com.bizunited.platform.core.repository.OrganizationRepository;
import com.bizunited.platform.core.repository.PositionRepository;
import com.bizunited.platform.core.repository.RoleRepository;
import com.bizunited.platform.core.repository.UserRepository;
import com.bizunited.platform.core.service.NebulaToolkitService;
import com.bizunited.platform.rbac.server.service.PositionService;
import com.bizunited.platform.rbac.server.vo.PositionVo;
import com.google.common.collect.Lists;

import com.google.common.collect.Sets;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.PageImpl;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.domain.Pageable;
import org.springframework.util.CollectionUtils;

import javax.transaction.Transactional;

/**
 * @description:
 * @author: yanwe
 * @date: 17/Jan/2019 15:01
 */
public class PositionServiceImpl implements PositionService {

  @Autowired 
  private PositionRepository positionRepository;
  @Autowired 
  private OrganizationRepository organizationRepository;
  @Autowired 
  private UserRepository userRepository;
  @Autowired 
  private RoleRepository roleRepository;
  @Autowired
  @Qualifier("nebulaToolkitService")
  private NebulaToolkitService nebulaToolkitService;

  /* (non-Javadoc)
   * @see com.bizunited.platform.rbac.server.service.PositionService#create(com.bizunited.platform.rbac.server.vo.PositionVo)
   */
  @Override
  @Transactional
  public PositionVo create(PositionVo positionVo) {
    Validate.notNull(positionVo, "岗位信息不能为空，请检查");
    Validate.isTrue(StringUtils.isBlank(positionVo.getId()), "添加岗位时，不能传入岗位id,请重新添加!!");
    Validate.notBlank(positionVo.getName(), "岗位名称不能为空，请检查!!");
    Validate.notNull(positionVo.getCode(), "岗位编号不能为空");
    positionVo.setCreateTime(new Date());
    PositionEntity currentPosition = positionRepository.findByCode(positionVo.getCode());
    Validate.isTrue(null == currentPosition,"存在重复的code,请重新输入岗位编码!!");
    
    // 转换后进行保存
    PositionEntity positionEntity = this.nebulaToolkitService.copyObjectByWhiteList(positionVo, PositionEntity.class, HashSet.class, ArrayList.class, new String[]{});
    positionRepository.save(positionEntity);
    positionVo.setId(positionEntity.getId());
    return positionVo;
  }

  /* (non-Javadoc)
   * @see com.bizunited.platform.rbac.server.service.PositionService#update(com.bizunited.platform.rbac.server.vo.PositionVo)
   */
  @Override
  @Transactional
  public PositionVo update(PositionVo positionVo) {
    // 验证
    Validate.notNull(positionVo, "岗位信息不能为空，请检查");
    Validate.isTrue(!StringUtils.isBlank(positionVo.getId()), "修改岗位时，必须传入岗位id,请重新修改!!");
    Validate.notBlank(positionVo.getName(), "岗位名称不能为空，请检查!!");
    Optional<PositionEntity> op = positionRepository.findById(positionVo.getId());
    PositionEntity position = op.orElse(null);
    Validate.notNull(position, "未在数据层找到对应的岗位信息");

    // 赋值并保存
    position.setName(positionVo.getName());
    positionRepository.save(position);
    // 转换后返回
    return this.nebulaToolkitService.copyObjectByWhiteList(position, PositionVo.class, HashSet.class, ArrayList.class, new String[]{});
  }

  /* (non-Javadoc)
   * @see com.bizunited.platform.rbac.server.service.PositionService#updateStatus(java.lang.String)
   */
  @Override
  @Transactional
  public PositionVo updateStatus(String positionId) {
    Validate.notBlank(positionId, "岗位id不能为空，请检查");
    Optional<PositionEntity> op = positionRepository.findById(positionId);
    PositionEntity position = op.orElse(null);
    Validate.notNull(position, "没有该岗位，请检查!!");
    Integer status = position.getTstatus();
    
    // 反转状态，0:禁用,1:启用
    status = (status == 1 ? 0 : 1);
    position.setTstatus(status);
    positionRepository.save(position);
    
    // 转换后返回
    return this.nebulaToolkitService.copyObjectByWhiteList(position, PositionVo.class, HashSet.class, ArrayList.class, new String[]{});
  }

  /* (non-Javadoc)
   * @see com.bizunited.platform.rbac.server.service.PositionService#findById(java.lang.String)
   */
  @Override
  public PositionVo findDetailsById(String positionId) {
    if (StringUtils.isBlank(positionId)) {
      return null;
    }
    PositionEntity positionEntity = positionRepository.findDetailsById(positionId);
    Validate.notNull(positionEntity , "未找到指定的岗位信息，请检查传参!!");
    
    // 转换后返回
    return this.nebulaToolkitService.copyObjectByWhiteList(positionEntity, PositionVo.class, HashSet.class, ArrayList.class, new String[]{"organization" , "roles" , "users"});
  }

  /* (non-Javadoc)
   * @see com.bizunited.platform.rbac.server.service.PositionService#bindOrgAndPosition(java.lang.String, java.lang.String)
   */
  @Override
  @Transactional
  public void bindOrgAndPosition(String orgId, String positionId) {
    Validate.notBlank(orgId, "组织机构ID不能为空!");
    Validate.notBlank(positionId, "岗位ID不能为空!");
    Optional<OrganizationEntity> optional = organizationRepository.findById(orgId);
    Validate.isTrue(optional.isPresent(), "未找到指定的组织机构!");
    OrganizationEntity organization = optional.get();
    Optional<PositionEntity> op = positionRepository.findById(positionId);
    PositionEntity position = op.orElse(null);
    Validate.notNull(position, "未找到指定的岗位!");
    Validate.isTrue(null == position.getOrganization(), "该岗位已绑定有组织机构，请检查！");
    position.setOrganization(organization);
    positionRepository.saveAndFlush(position);
  }

  /* (non-Javadoc)
   * @see com.bizunited.platform.rbac.server.service.PositionService#unbindOrgAndPosition(java.lang.String, java.lang.String)
   */
  @Override
  @Transactional
  public void unbindOrgAndPosition(String orgId, String positionId) {
    Validate.notBlank(orgId, "组织机构ID不能为空!");
    Validate.notBlank(positionId, "岗位ID不能为空!");
    Optional<OrganizationEntity> optional = organizationRepository.findById(orgId);
    Validate.isTrue(optional.isPresent(), "未找到指定的组织机构!");
    OrganizationEntity organization = optional.get();
    Optional<PositionEntity> op = positionRepository.findById(positionId);
    PositionEntity position = op.orElse(null);
    Validate.notNull(position, "未找到指定的岗位!");
    Validate.notNull(position.getOrganization(), "该岗位没有绑定的组织，请检查！");
    Validate.isTrue(organization.getId().equals(position.getOrganization().getId()), "待解绑的岗位与组织机构没有绑定关系，请检查！");
    position.setOrganization(null);
    positionRepository.saveAndFlush(position);
  }

  /* (non-Javadoc)
   * @see com.bizunited.platform.rbac.server.service.PositionService#bindRoleAndPosition(java.lang.String, java.lang.String)
   */
  @Override
  @Transactional
  public void bindRoleAndPosition(String roleId, String positionId) {
    Validate.notBlank(roleId, "角色ID不能为空!");
    Validate.notBlank(positionId, "岗位ID不能为空!");
    Optional<RoleEntity> optional = roleRepository.findById(roleId);
    Validate.isTrue(optional.isPresent(), "未找到指定的角色!");
    Optional<PositionEntity> op = positionRepository.findById(positionId);
    Validate.isTrue(op.isPresent(), "未找到指定的岗位!");
    Validate.isTrue(0 == positionRepository.countByRoleAndPosition(roleId, positionId), "该角色与该岗位已有绑定关系，请检查！");
    positionRepository.bindRole(roleId, positionId);
  }

  /* (non-Javadoc)
   * @see com.bizunited.platform.rbac.server.service.PositionService#bindRolesAndPosition(java.util.List, java.lang.String)
   */
  @Override
  @Transactional
  public void bindRolesAndPosition(List<String> roleIds, String positionId) {
    Validate.isTrue(!CollectionUtils.isEmpty(roleIds), "角色ID集合不能为空，请检查！");
    roleIds.stream().forEach(o -> bindRoleAndPosition(o, positionId));
  }

  /* (non-Javadoc)
   * @see com.bizunited.platform.rbac.server.service.PositionService#unbindRoleAndPosition(java.lang.String, java.lang.String)
   */
  @Override
  @Transactional
  public void unbindRoleAndPosition(String roleId, String positionId) {
    Validate.notBlank(roleId, "角色ID不能为空!");
    Validate.notBlank(positionId, "岗位ID不能为空!");
    Optional<RoleEntity> optional = roleRepository.findById(roleId);
    Validate.isTrue(optional.isPresent(), "未找到指定的角色!");
    Optional<PositionEntity> op = positionRepository.findById(positionId);
    Validate.isTrue(op.isPresent(), "未找到指定的岗位!");
    positionRepository.unbindRole(roleId, positionId);
  }

  /* (non-Javadoc)
   * @see com.bizunited.platform.rbac.server.service.PositionService#unbindRolesAndPosition(java.util.List, java.lang.String)
   */
  @Override
  @Transactional
  public void unbindRolesAndPosition(List<String> roleIds, String positionId) {
    Validate.isTrue(!CollectionUtils.isEmpty(roleIds), "角色ID集合不能为空，请检查！");
    roleIds.stream().forEach(o -> unbindRoleAndPosition(o, positionId));
  }

  /* (non-Javadoc)
   * @see com.bizunited.platform.rbac.server.service.PositionService#bindUsersAndPosition(java.util.List, java.lang.String)
   */
  @Override
  @Transactional
  public void bindUsersAndPosition(List<String> userIds, String positionId) {
    Validate.isTrue(!CollectionUtils.isEmpty(userIds), "用户ID集合不能为空，请检查！");
    userIds.stream().forEach(o -> bindUserAndPosition(o, positionId));
  }
  
  private void bindUserAndPosition(String userId, String positionId) {
    Validate.notBlank(userId, "用户ID不能为空!");
    Validate.notBlank(positionId, "岗位ID不能为空!");
    Optional<UserEntity> op = userRepository.findById(userId);
    UserEntity userEntity = op.orElse(null);
    Validate.notNull(userEntity, "未找到指定的用户!");

    Optional<PositionEntity> optional = positionRepository.findById(positionId);
    Validate.isTrue(optional.isPresent(), "未找到指定的岗位!");
    // 2019.02.20 逻辑暂时修改为一个用户只能绑定一个岗位
    String positionName = CollectionUtils.isEmpty(userEntity.getPositions())? "" : new ArrayList<>(userEntity.getPositions()).get(0).getName();
    Validate.isTrue(CollectionUtils.isEmpty(userEntity.getPositions()), "该用户[" + userEntity.getUserName() + "]已经绑定了岗位[" + positionName + "]，请先解绑再绑定新岗位!");
    positionRepository.binduser(userId, positionId);
  }

  /* (non-Javadoc)
   * @see com.bizunited.platform.rbac.server.service.PositionService#unbindUsersAndPosition(java.util.List, java.lang.String)
   */
  @Override
  @Transactional
  public void unbindUsersAndPosition(List<String> userIds, String positionId) {
    Validate.isTrue(!CollectionUtils.isEmpty(userIds), "用户ID集合不能为空，请检查！");
    userIds.stream().forEach(o -> unbindUserAndPosition(o, positionId));
  }

  private void unbindUserAndPosition(String userId, String positionId) {
    Validate.notBlank(userId, "用户ID不能为空!");
    Validate.notBlank(positionId, "岗位ID不能为空!");
    Optional<UserEntity> op = userRepository.findById(userId);
    Validate.isTrue(op.isPresent(), "未找到指定的用户!");
    Optional<PositionEntity> optional = positionRepository.findById(positionId);
    Validate.isTrue(optional.isPresent(), "未找到指定的岗位!");
    Validate.isTrue(0 != positionRepository.countByUserAndPosition(userId, positionId), "该用户与该岗位未有绑定关系，请检查！");
    positionRepository.unbinduser(userId, positionId);
  }
  
  /* (non-Javadoc)
   * @see com.bizunited.platform.rbac.server.service.PositionService#findByUserId(java.lang.String)
   */
  @Override
  public PositionVo findByUserId(String userId) {
    Validate.notBlank(userId,"传入的用户ID不能为空！");
    UserEntity user = this.userRepository.findDetailsById(userId);
    if(user==null){
      return null;
    }
    
    if(!CollectionUtils.isEmpty(user.getPositions())){
      PositionEntity onePosition = user.getPositions().iterator().next();
      // 转换后返回
      return this.nebulaToolkitService.copyObjectByWhiteList(onePosition, PositionVo.class, HashSet.class, ArrayList.class, new String[]{});
    }
    return null;
  }

  @Override
  public Set<PositionVo> findByIds(List<String> ids) {
    if(CollectionUtils.isEmpty(ids)){
      return Sets.newHashSet();
    }
    Set<PositionEntity> pos = positionRepository.findByIds(ids);
    if(CollectionUtils.isEmpty(pos)){
      return Sets.newHashSet();
    }
    Collection<PositionVo> pvo = this.nebulaToolkitService.copyCollectionByWhiteList(pos,PositionEntity.class,PositionVo.class,HashSet.class,ArrayList.class,new String[]{});
    return Sets.newHashSet(pvo);
  }

  /* (non-Javadoc)
   * @see com.bizunited.platform.rbac.server.service.PositionService#findByConditions(java.lang.String, java.lang.String, java.lang.Integer, org.springframework.data.domain.Pageable)
   */
  @Override
  public Page<PositionVo> findByConditions(String code, String name, Integer tstatus, Pageable pageable) {
    Map<String, Object> conditions = new HashMap<>();
    if (StringUtils.isNotBlank(code)) {
      conditions.put("code", code);
    }
    if (StringUtils.isNotBlank(name)) {
      conditions.put("name", name);
    }
    if (tstatus != null) {
      conditions.put("tstatus", tstatus);
    }
    // 如果当前没有设定分页信息，则默认第一页，每页50条数据
    if (pageable == null) {
      pageable = PageRequest.of(0, 50);
    }
  
    Page<PositionEntity> positionEntityPage = positionRepository.queryPage(pageable, conditions);
    // 分页信息需要自行转换
    List<PositionEntity> positionEntitys = positionEntityPage.getContent();
    Page<PositionVo> positionVoPage = null;
    if(positionEntitys != null && !positionEntitys.isEmpty()) {
      Collection<PositionVo> positionVos = this.nebulaToolkitService.copyCollectionByWhiteList(positionEntitys, PositionEntity.class, PositionVo.class, LinkedHashSet.class, ArrayList.class, new String[]{"users"});
      positionVoPage = new PageImpl<PositionVo>(new ArrayList<>(positionVos), pageable, positionEntityPage.getTotalElements());
    } else {
      positionVoPage = new PageImpl<PositionVo>(Lists.newArrayList(), pageable, 0l);
    }
    return positionVoPage;
  }
}
