package com.bizunited.platform.core.service.security;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Date;
import java.util.HashSet;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Optional;
import java.util.Set;
import java.util.stream.Collectors;

import javax.transaction.Transactional;

import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.security.access.AccessDeniedException;

import com.bizunited.platform.core.entity.OrganizationEntity;
import com.bizunited.platform.core.entity.RoleEntity;
import com.bizunited.platform.core.entity.UserEntity;
import com.bizunited.platform.core.entity.UserGroupEntity;
import com.bizunited.platform.core.repository.OrganizationRepository;
import com.bizunited.platform.core.repository.RoleRepository;
import com.bizunited.platform.core.repository.UserGroupRepository;
import com.bizunited.platform.core.repository.UserRepository;
import com.bizunited.platform.core.service.NebulaToolkitService;
import com.bizunited.platform.rbac.server.service.RoleService;
import com.bizunited.platform.rbac.server.vo.RoleVo;
import com.google.common.collect.Lists;
import com.google.common.collect.Sets;

import org.springframework.util.CollectionUtils;

/**
 * @author yinwenjie
 */
public class RoleServiceImpl implements RoleService {

  /**
   * 角色repository自动注入.
   */
  @Autowired
  private RoleRepository roleRepository;
  @Autowired
  private UserGroupRepository userGroupRepository;
  @Autowired
  private OrganizationRepository organizationRepository;
  @Autowired
  private UserRepository userService;
  @Autowired
  @Qualifier("nebulaToolkitService")
  private NebulaToolkitService nebulaToolkitService;
  /**
   * 这个参数记录了那些可以忽略“功能”级别权限验证的角色
   */
  @Value("${rbac.ignoreMethodCheckRoles}")
  private String[] ignoreMethodCheckRoles; 
  /**
   * 配置的那些不允许被删除被作废的角色
   */
  @Value("${rbac.roles.deleteDeny}")
  private String[] deleteDenys;

  /* (non-Javadoc)
   * @see com.bizunited.platform.rbac.server.service.RoleService#create(com.bizunited.platform.rbac.server.vo.RoleVo)
   */
  @Override
  @Transactional
  public RoleVo create(RoleVo role) {
    // 进入传入信息的检查
    this.validateRoleBeforeAdd(role);
    // 开始插入
    RoleEntity roleEntity = this.nebulaToolkitService.copyObjectByWhiteList(role, RoleEntity.class, HashSet.class, ArrayList.class, new String[]{});
    this.roleRepository.saveAndFlush(roleEntity);
    role.setId(roleEntity.getId());
    return role;
  }
  /* (non-Javadoc)
   * @see com.bizunited.platform.rbac.server.service.RoleService#update(com.bizunited.platform.rbac.server.vo.RoleVo)
   */
  @Override
  @Transactional
  public RoleVo update(RoleVo role) {
    /*
     * 修改角色的过程如下： 
     * 1、首先确定当前role是可以进行修改的role 
     * 2、只进行comment的修改，其它即使传递了也不进行修改
     */
    Validate.notNull(role, "role mssage not null");
    String roleId = role.getId();
    String updateComment = role.getComment();
    Validate.notEmpty(roleId, "role id not empty!");
    Validate.notEmpty(updateComment, "role comment not empty!");
    Optional<RoleEntity> op = this.roleRepository.findById(roleId);
    Validate.isTrue(op.isPresent(), "role not found");
    RoleEntity currentRole = op.get();
    // 1、========
    String currentName = currentRole.getRoleName();
    // 如果条件成立，说明这个角色信息不能被修改
    for (String deleteDeny : deleteDenys) {
      if (StringUtils.equals(currentName, deleteDeny)) {
        throw new AccessDeniedException("这个角色为系统特定默认角色，不允许修改！");
      }
    }

    // 2、========
    currentRole.setComment(updateComment);
    this.roleRepository.save(currentRole);
    return role;
  }
  /* (non-Javadoc)
   * @see com.bizunited.platform.rbac.server.service.RoleService#disable(java.lang.String)
   */
  @Override
  @Transactional
  public RoleVo disable(String roleId) {
    /*
     * 注意：在配置文件中已经设定的那些不能操作的角色信息，是不允许禁用的
     */
    Validate.notEmpty(roleId, "role id not be found!");
    Optional<RoleEntity> op = this.roleRepository.findById(roleId);
    Validate.isTrue(op.isPresent(), "未找到指定的角色信息，请检查!!");
    RoleEntity currentRole = op.get();

    // 如果条件成立，说明这个角色信息不能被删除（或者说作废）
    for (String deleteDeny : deleteDenys) {
      if (StringUtils.equals(currentRole.getRoleName(), deleteDeny)) {
        throw new AccessDeniedException("the role not allow be disable！");
      }
    }

    // 禁用角色
    currentRole.setTstatus(0);
    this.roleRepository.save(currentRole);
    RoleVo roleVo = this.nebulaToolkitService.copyObjectByWhiteList(currentRole, RoleVo.class, HashSet.class, ArrayList.class, new String[]{});
    return roleVo;
  }
  /* (non-Javadoc)
   * @see com.bizunited.platform.rbac.server.service.RoleService#enable(java.lang.String)
   */
  @Override
  @Transactional
  public RoleVo enable(String roleId) {
    Validate.notEmpty(roleId, "role id not be found!");
    Optional<RoleEntity> op = this.roleRepository.findById(roleId);
    Validate.isTrue(op.isPresent(), "未找到指定的角色信息，请检查!!");
    RoleEntity currentRole = op.get();

    // 启用角色
    currentRole.setTstatus(1);
    this.roleRepository.save(currentRole);
    RoleVo roleVo = this.nebulaToolkitService.copyObjectByWhiteList(currentRole, RoleVo.class, HashSet.class, ArrayList.class, new String[]{});
    return roleVo;
  }
  /**
   * 该私有方法在新增一个role信息前，检查传入信息的证确定   * @param role
   */
  private void validateRoleBeforeAdd(RoleVo role) {
    Validate.notNull(role, "role input object not be null!");
    // 开始验证
    Validate.notBlank(role.getRoleName(), "role name not be null!");
    // 必须是大写
    role.setRoleName(role.getRoleName().toUpperCase());
    RoleEntity oldRole = this.roleRepository.findByRoleName(role.getRoleName());
    Validate.isTrue(oldRole == null, "当前设定的角色名称（role name）已经被使用，请更换!");
    // code必须验证
    oldRole = this.roleRepository.findByRoleCode(role.getRoleCode());
    Validate.isTrue(oldRole == null, "当前设定的角色编号（role code）已经被使用，请更换!");
    // 当前的创建时间和修改时间要一起写入
    Date currentTime = new Date();
    role.setCreateDate(currentTime);
    // 说明性信息
    if(StringUtils.isBlank(role.getComment())) {
      role.setComment("[未填写]");
    }
    // 当前角色必须是状态正常的
    role.setTstatus(1);
  }
  /* (non-Javadoc)
   * @see com.bizunited.platform.rbac.server.service.RoleService#bindRoles(java.lang.String, java.lang.String[])
   */
  @Override
  @Transactional
  public void bindRoles(String userId, String[] roleNames) {
    Validate.notBlank(userId, "绑定权限时，用户id不能为空");
    UserEntity user = userService.findDetailsById(userId);
    Validate.notNull(user, "用户不存在");
    Validate.isTrue(roleNames != null && roleNames.length > 0 , "在进行角色绑定时，请至少传入一个角色名信息");
    
    // 去重
    Set<String> roleNameSet = new HashSet<>();
    roleNameSet.addAll(Arrays.asList(roleNames));
    // 开始添加（不用做存在性判断，没有自然会报错）
    for (String roleName : roleNameSet) {
      RoleEntity currentRole = this.roleRepository.findByRoleName(roleName);
      Validate.notNull(currentRole , "未找到指定的角色!!");
      // 一般来说，这里抛出异常就是重复绑定
      try {
        this.roleRepository.bindUser(userId, currentRole.getId());
        this.roleRepository.flush();
      } catch(Exception e) {
        throw new IllegalArgumentException("错误的绑定信息，请检测是否重复绑定，或者角色信息未传入!");
      } 
    }
  }
  /* (non-Javadoc)
   * @see com.bizunited.platform.rbac.server.service.RoleService#reBindRoles(java.lang.String, java.lang.String[])
   */
  @Override
  @Transactional
  public void reBindRoles(String userId, String[] roleNames) {
    Validate.notBlank(userId, "绑定权限时，用户id不能为空");
    UserEntity user = userService.findDetailsById(userId);
    Validate.notNull(user, "用户不存在");
    Validate.isTrue(roleNames != null && roleNames.length > 0 , "在进行角色绑定时，请至少传入一个角色名信息");
    
    // 去重
    Set<String> roleNameSet = new HashSet<>();
    roleNameSet.addAll(Arrays.asList(roleNames));
    // 首先删除以前的绑定信息
    this.roleRepository.deleteRoleByUserId(userId);
    // 然后重新进行绑定
    for (String roleName : roleNameSet) {
      RoleEntity currentRole = this.roleRepository.findByRoleName(roleName);
      Validate.notNull(currentRole , "未找到指定的角色!!");
      // 一般来说，这里抛出异常就是重复绑定
      try {
        this.roleRepository.bindUser(userId, currentRole.getId());
      } catch(Exception e) {
        throw new IllegalArgumentException("错误的绑定信息，请检测是否重复绑定，或者角色信息未传入!");
      } 
    }
  }
  /* (non-Javadoc)
   * @see com.bizunited.platform.rbac.server.service.RoleService#findAllByUserId(java.lang.String)
   */
  @Override
  public List<RoleVo> findAllByUserId(String userId) {
    /*
     * 在表单引擎V1.2+，按照一个用户数据库编号，查询这个用户对应的角色信息，就没有那么简单了，
     * 因为一个用户的角色可以由三方面决定：
     * a、这个用户直接绑定的角色信息
     * b、这个用户所属的一个或者多个用户组绑定的角色信息
     * c、这个用户所属的一个或者多个组织机构（不包括父级）绑定的角色信息
     * 
     * 注意：数据层并没有使用status进行过滤，这里需要自行进行过滤
     * */
    if(StringUtils.isBlank(userId)) {
      return null;
    }
    
    // a、====
    Set<RoleEntity> resultRoles = new HashSet<>();
    List<RoleEntity> roles = this.roleRepository.findByUserId(userId);
    if(roles != null && !roles.isEmpty()) {
      resultRoles.addAll(roles);
    }
    
    // b、====
    Set<UserGroupEntity> groups = this.userGroupRepository.findByUserId(userId);
    if(groups != null && !groups.isEmpty()) {
      groups.stream().filter(item -> item.getRoles() != null && item.getTstatus() == 1).forEach(item ->  resultRoles.addAll(item.getRoles()));
    }
    
    // c、====
    Set<OrganizationEntity> orgs = this.organizationRepository.findByUserId(userId);
    if(orgs != null && !orgs.isEmpty()) {
      orgs.stream().filter(item -> item.getRoles() != null && item.getTstatus() == 1).forEach(item ->  resultRoles.addAll(item.getRoles()));
    }
    
    // 过滤出唯一信息后返回
    if(resultRoles.isEmpty()) {
      return null;
    }
    List<RoleEntity> roleEntitys = resultRoles.stream().filter(item -> item.getTstatus() == 1).collect(Collectors.toList());
    // 转换vo后进行返回
    Collection<RoleVo> roleVos = this.nebulaToolkitService.copyCollectionByWhiteList(roleEntitys, RoleEntity.class, RoleVo.class, LinkedHashSet.class, ArrayList.class, new String[]{});
    return Lists.newArrayList(roleVos);
  }
  /* (non-Javadoc)
   * @see com.bizunited.platform.rbac.server.service.RoleService#findByCompetenceId(java.lang.String)
   */
  @Override
  public List<RoleVo> findByCompetenceId(String competenceId) {
    if(StringUtils.isBlank(competenceId)) {
      return null;
    }
    List<RoleEntity> roles = this.roleRepository.findByCompetenceId(competenceId);
    if (roles == null || roles.isEmpty()) {
      return null;
    }
    Collection<RoleVo> roleVos = this.nebulaToolkitService.copyCollectionByWhiteList(roles, RoleEntity.class, RoleVo.class, LinkedHashSet.class, ArrayList.class, new String[]{});
    return Lists.newArrayList(roleVos);
  }
  /* (non-Javadoc)
   * @see com.bizunited.platform.rbac.server.service.RoleService#findByStatus(java.lang.Integer)
   */
  @Override
  public List<RoleVo> findByStatus(Integer useStatus) {
    if(useStatus == null) {
      return null;
    }
    List<RoleEntity> rolesList = this.roleRepository.findByTstatus(useStatus);
    if (rolesList == null || rolesList.size() == 0) {
      return null;
    }
    Collection<RoleVo> roleVos = this.nebulaToolkitService.copyCollectionByWhiteList(rolesList, RoleEntity.class, RoleVo.class, LinkedHashSet.class, ArrayList.class, new String[]{});
    return Lists.newArrayList(roleVos);
  }
  /* (non-Javadoc)
   * @see com.bizunited.platform.rbac.server.service.RoleService#findById(java.lang.String)
   */
  @Override
  public RoleVo findById(String roleId) {
    Validate.notNull(roleId, "role Id must not null!");
    Optional<RoleEntity> op = this.roleRepository.findById(roleId);
    if (!op.isPresent()) {
      return null;
    }
    RoleVo roleVo = this.nebulaToolkitService.copyObjectByWhiteList(op.get(), RoleVo.class, HashSet.class, ArrayList.class, new String[]{});
    return roleVo;
  }
  /* (non-Javadoc)
   * @see com.bizunited.platform.rbac.server.service.RoleService#findAll()
   */
  @Override
  public List<RoleVo> findAll() {
    List<RoleEntity> rolesList = this.roleRepository.findAll();
    // 如果条件成立说明系统该数据异常，这是直接抛出错误信息
    if (rolesList == null || rolesList.isEmpty()) {
      return null;
    }
    
    Collection<RoleVo> roleVos = this.nebulaToolkitService.copyCollectionByWhiteList(rolesList, RoleEntity.class, RoleVo.class, LinkedHashSet.class, ArrayList.class, new String[]{});
    return Lists.newArrayList(roleVos);
  }

  @Override
  public Set<RoleVo> findByIds(List<String> ids) {
    if(CollectionUtils.isEmpty(ids)){
      return Sets.newHashSet();
    }
    Set<RoleEntity> roles = this.roleRepository.findByIds(ids);
    if(CollectionUtils.isEmpty(roles)){
      return Sets.newHashSet();
    }
    Collection<RoleVo> rvos = this.nebulaToolkitService.copyCollectionByWhiteList(roles,RoleEntity.class,RoleVo.class,HashSet.class,ArrayList.class,new String[]{});
    return Sets.newHashSet(rvos);
  }
}
