package com.bizunited.platform.core.service.security;

import com.bizunited.platform.core.entity.RoleEntity;
import com.bizunited.platform.core.entity.UserEntity;
import com.bizunited.platform.core.entity.UserGroupEntity;
import com.bizunited.platform.core.repository.RoleRepository;
import com.bizunited.platform.core.repository.UserGroupRepository;
import com.bizunited.platform.core.repository.UserRepository;
import com.bizunited.platform.core.service.NebulaToolkitService;
import com.bizunited.platform.rbac.server.service.UserGroupService;
import com.bizunited.platform.rbac.server.vo.UserGroupVo;
import com.google.common.collect.Lists;
import com.google.common.collect.Sets;

import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.PageImpl;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.domain.Pageable;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.CollectionUtils;

import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.HashSet;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

public class UserGroupServiceImpl implements UserGroupService {
  @Autowired
  private UserGroupRepository userGroupRepository;
  @Autowired
  private UserRepository userRepository;
  @Autowired
  private RoleRepository roleRepository;
  @Autowired
  @Qualifier("nebulaToolkitService")
  private NebulaToolkitService nebulaToolkitService;
  private static final String ERROR_GROUP_ID = "用户组id不能为空，请检查";
  private static final String ERROR_NOT_FOUND = "没有该工作组，请检查!!";
  /* (non-Javadoc)
   * @see com.bizunited.platform.rbac.server.service.UserGroupService#create(com.bizunited.platform.rbac.server.vo.UserGroupVo)
   */
  @Transactional
  @Override
  public UserGroupVo create(UserGroupVo userGroup) {
    // 验证传入用户组信息的正确性
    Validate.notNull(userGroup, "用户组信息不能为空，请检查");
    Validate.isTrue(StringUtils.isBlank(userGroup.getId()), "添加用户组时，不能传入用户组id,请重新添加!!");
    String groupName = userGroup.getGroupName();
    Validate.notBlank(groupName, "用户组名称不能为空，请检查!!");
    Validate.isTrue(!isChinese(groupName), "用户组名称不能包含中文，请重新输入!!");
    Validate.notBlank(userGroup.getGroupDescription(), "用户组描述不能为空，请检查!!");
    Validate.notNull(userGroup.getTstatus(), "用户组状态不能为空，请检查!!");
    UserGroupEntity oldUserGroup = userGroupRepository.findByGroupName(groupName);
    Validate.isTrue(null == oldUserGroup, "该用户组已经存在，请重新输入用户组名");
    
    // 转换后保存
    UserGroupEntity userGroupEntity = this.nebulaToolkitService.copyObjectByWhiteList(userGroup, UserGroupEntity.class, HashSet.class, ArrayList.class, new String[]{});
    userGroupRepository.save(userGroupEntity);
    userGroup.setId(userGroupEntity.getId());
    return userGroup;
  }
  /* (non-Javadoc)
   * @see com.bizunited.platform.rbac.server.service.UserGroupService#update(com.bizunited.platform.rbac.server.vo.UserGroupVo)
   */
  @Transactional
  @Override
  public UserGroupVo update(UserGroupVo userGroup) {
    // 验证传入的用户信息
    Validate.notNull(userGroup, "用户组信息不能为空，请检查");
    String groupId = userGroup.getId();
    Validate.notNull(groupId, ERROR_GROUP_ID);
    Optional<UserGroupEntity> op = userGroupRepository.findById(groupId);
    Validate.isTrue(op.isPresent(), "该用户组不存在，请重新输入用户组名");
    UserGroupEntity currentGroup = op.get();

    // 只有用户组描述信息能够修改
    if(!StringUtils.isBlank(userGroup.getGroupDescription())) {
      currentGroup.setGroupDescription(userGroup.getGroupDescription());
      userGroupRepository.save(currentGroup);
    }
    
    // 转换后输出
    UserGroupVo userGroupVo = this.nebulaToolkitService.copyObjectByWhiteList(currentGroup, UserGroupVo.class, HashSet.class, ArrayList.class, new String[]{});
    return userGroupVo;
  }
  /* (non-Javadoc)
   * @see com.bizunited.platform.rbac.server.service.UserGroupService#updateStatus(java.lang.String)
   */
  @Transactional
  @Override
  public UserGroupVo updateStatus(String groupId) {
    Validate.notNull(groupId, ERROR_GROUP_ID);
    Optional<UserGroupEntity> op = userGroupRepository.findById(groupId);
    Validate.isTrue(op.isPresent(), ERROR_NOT_FOUND);
    UserGroupEntity userGroup = op.get();
    Integer status = userGroup.getTstatus();
    status = (status == 1 ? 0 : 1); //0:禁用,1:启用
    userGroup.setTstatus(status);
    userGroupRepository.save(userGroup);
    
    // 转换后输出
    UserGroupVo userGroupVo = this.nebulaToolkitService.copyObjectByWhiteList(userGroup, UserGroupVo.class, HashSet.class, ArrayList.class, new String[]{});
    return userGroupVo;
  }
  /* (non-Javadoc)
   * @see com.bizunited.platform.rbac.server.service.UserGroupService#bindUser(java.lang.String, java.lang.String[])
   */
  @Override
  @Transactional
  public void bindUser(String groupId, String userIds[]) {
    Validate.notNull(groupId, ERROR_GROUP_ID);
    Validate.isTrue(userIds != null && userIds.length > 0, "用户id不能为空（至少需要传入一个），请检查");
    Optional<UserGroupEntity> op = userGroupRepository.findById(groupId);
    Validate.isTrue(op.isPresent(), ERROR_NOT_FOUND);

    for (String userId : userIds) {
      Optional<UserEntity> optional = userRepository.findById(userId);
      Validate.isTrue(optional.isPresent(), "没有该用户[%s],请检查!!" , userId);
      int count = userGroupRepository.countUserGroup(groupId, userId);
      Validate.isTrue(count == 0, "在批量绑定是，指定用户[%s]已经绑定了，不能重复绑定，请检查!!" , userId);
      userGroupRepository.bindUser(groupId, userId);
    }
  }
  /* (non-Javadoc)
   * @see com.bizunited.platform.rbac.server.service.UserGroupService#unBindUser(java.lang.String, java.lang.String[])
   */
  @Override
  @Transactional
  public void unBindUser(String groupId, String userIds[]) {
    Validate.notNull(groupId, ERROR_GROUP_ID);
    Validate.isTrue(userIds != null && userIds.length > 0, "用户id不能为空（至少需要传入一个），请检查");
    Optional<UserGroupEntity> op = userGroupRepository.findById(groupId);
    Validate.isTrue(op.isPresent(), ERROR_NOT_FOUND);
    
    for (String userId : userIds) {
      Optional<UserEntity> optional = userRepository.findById(userId);
      Validate.isTrue(optional.isPresent(), "没有该用户,请检查!!");
      userGroupRepository.unBindUser(groupId, userId);
    }
  }
  /* (non-Javadoc)
   * @see com.bizunited.platform.kuiper.starter.service.UserGroupService#bindRole(java.lang.String, java.lang.String[])
   */
  @Override
  @Transactional
  public void bindRole(String groupId, String roleIds[]) {
    Validate.notNull(groupId, ERROR_GROUP_ID);
    Validate.isTrue(roleIds != null && roleIds.length > 0, "角色id不能为空（至少需要传入一个），请检查");
    Optional<UserGroupEntity> op = userGroupRepository.findById(groupId);
    Validate.isTrue(op.isPresent(), ERROR_NOT_FOUND);

    for (String roleId : roleIds) {
      Optional<RoleEntity> optional = roleRepository.findById(roleId);
      Validate.isTrue(optional.isPresent(), "没有该角色[%s],请检查!!" , roleId);
      int count = userGroupRepository.countRoleGroup(groupId, roleId);
      Validate.isTrue(count == 0, "在批量绑定时，指定角色[%s]已经绑定了，不能重复绑定，请检查!!" , roleId);
      userGroupRepository.bindRole(groupId, roleId);
    }
  }
  /* (non-Javadoc)
   * @see com.bizunited.platform.rbac.server.service.UserGroupService#unBindRole(java.lang.String, java.lang.String[])
   */
  @Override
  @Transactional
  public void unBindRole(String groupId, String roleIds[]) {
    Validate.notNull(groupId, ERROR_GROUP_ID);
    Validate.isTrue(roleIds != null && roleIds.length > 0, "角色id不能为空（至少需要传入一个），请检查");
    Optional<UserGroupEntity> op = userGroupRepository.findById(groupId);
    Validate.isTrue(op.isPresent(), ERROR_NOT_FOUND);
    
    for (String roleId : roleIds) {
      Optional<RoleEntity> optional = roleRepository.findById(roleId);
      Validate.isTrue(optional.isPresent(), "没有该角色[%s],请检查!!" , roleId);
      userGroupRepository.unBindRole(groupId, roleId);
    }
  }
  /* (non-Javadoc)
   * @see com.bizunited.platform.rbac.server.service.UserGroupService#findById(java.lang.String)
   */
  @Override
  public Set<UserGroupVo> findDetailsById(String id) {
    if (StringUtils.isBlank(id)) {
      return Sets.newHashSet();
    }
    Set<UserGroupEntity> userGroupEntitys = userGroupRepository.findDetailsById(id);
    if(userGroupEntitys == null || userGroupEntitys.isEmpty()) {
      return Sets.newHashSet();
    }
    
    // 转换后输出
    Collection<UserGroupVo> userGroupVos = this.nebulaToolkitService.copyCollectionByWhiteList(userGroupEntitys, UserGroupEntity.class, UserGroupVo.class, HashSet.class, ArrayList.class, new String[]{"roles", "users"});
    return Sets.newHashSet(userGroupVos);
  }

  @Override
  public Set<UserGroupVo> findByIds(List<String> ids) {
    if(CollectionUtils.isEmpty(ids)){
      return Sets.newHashSet();
    }
    Set<UserGroupEntity> groups = userGroupRepository.findByIds(ids);
    if(CollectionUtils.isEmpty(groups)){
      return Sets.newHashSet();
    }
    Collection<UserGroupVo> ugroups = this.nebulaToolkitService.copyCollectionByWhiteList(groups,UserGroupEntity.class,UserGroupVo.class,HashSet.class,ArrayList.class,new String[]{});
    return Sets.newHashSet(ugroups);
  }

  /**
   * 判断是否包含中文
   * @param str
   * @return
   */
  private static boolean isChinese(String str) {
    String regEx = "[\\u4e00-\\u9fa5]+";
    Pattern p = Pattern.compile(regEx);
    Matcher m = p.matcher(str);
    if (m.find())
      return true;
    else
      return false;
  }
  /* (non-Javadoc)
   * @see com.bizunited.platform.kuiper.starter.service.UserGroupService#findByUserId(java.lang.String)
   */
  @Override
  public Set<UserGroupVo> findByUserId(String userId) {
    if (StringUtils.isBlank(userId)) {
      return Sets.newHashSet();
    }
    return this.findByUserId(userId);
  }
  /* (non-Javadoc)
   * @see com.bizunited.platform.rbac.server.service.UserGroupService#findByCondition(java.lang.String, java.lang.Integer, org.springframework.data.domain.Pageable)
   */
  @Override
  public Page<UserGroupVo> findByCondition(String groupName, Integer status, Pageable pageable) {
    Map<String, Object> conditions = new HashMap<>();
    if (StringUtils.isNotBlank(groupName)) {
      conditions.put("groupName", groupName);
    }
    if (status != null) {
      conditions.put("status", status);
    }
    // 如果当前没有设定分页信息，则默认第一页，每页50条数据
    if (pageable == null) {
      pageable = PageRequest.of(0, 50);
    }
    Page<UserGroupEntity> userGroupPage = userGroupRepository.queryPage(pageable, conditions);
    
    // 分页信息需要自行转换
    List<UserGroupEntity> userGroupEntitys = userGroupPage.getContent();
    Page<UserGroupVo> userGroupVoPage = null;
    if(userGroupEntitys != null && !userGroupEntitys.isEmpty()) {
      Collection<UserGroupVo> userGroupVos = this.nebulaToolkitService.copyCollectionByWhiteList(userGroupEntitys, UserGroupEntity.class, UserGroupVo.class, LinkedHashSet.class, ArrayList.class, new String[]{});
      userGroupVoPage = new PageImpl<UserGroupVo>(new ArrayList<>(userGroupVos), pageable, userGroupPage.getTotalElements());
    } else {
      userGroupVoPage = new PageImpl<UserGroupVo>(Lists.newArrayList(), pageable, 0l);
    }
    return userGroupVoPage;
  }
}
