package com.bizunited.platform.core.service.security;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Date;
import java.util.HashMap;
import java.util.HashSet;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;

import javax.transaction.Transactional;

import com.bizunited.platform.core.entity.RoleEntity;
import com.bizunited.platform.core.entity.UserEntity;
import com.bizunited.platform.core.repository.RoleRepository;
import com.bizunited.platform.core.repository.UserRepository;
import com.bizunited.platform.core.service.NebulaToolkitService;

import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.context.annotation.Lazy;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.PageImpl;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.domain.Pageable;
import org.springframework.security.crypto.password.PasswordEncoder;

import com.bizunited.platform.rbac.server.service.RoleService;
import com.bizunited.platform.rbac.server.service.UserService;
import com.bizunited.platform.rbac.server.vo.RoleVo;
import com.bizunited.platform.rbac.server.vo.UserVo;
import com.google.common.collect.Lists;
import com.google.common.collect.Sets;

import org.springframework.util.CollectionUtils;

/**
 * 基本用户信息服务的默认实现
 * @author yinwenjie
 */
public class UserServiceImpl implements UserService {
  @Autowired
  private UserRepository userRepository;
  @Autowired
  private RoleRepository roleRepository;
  @Autowired
  private RoleService roleService;
  @Autowired
  @Qualifier("passwordEncoder")
  @Lazy
  private PasswordEncoder passwordEncoder;
  @Autowired
  @Qualifier("nebulaToolkitService")
  private NebulaToolkitService nebulaToolkitService;
  
  private static String BASE_ROLE_NAME = "BASEROLE";

  /* (non-Javadoc)
   * @see com.bizunited.platform.rbac.server.service.UserService#create(com.bizunited.platform.rbac.server.vo.UserVo)
   */
  @Override
  @Transactional
  public UserVo create(UserVo userVo) {
    Validate.notNull(userVo , "必须传入要添加的用户信息!");
    String userName = userVo.getUserName();
    String account = userVo.getAccount();
    Validate.notBlank(userName , "用户真实姓名必须传入!");
    Validate.notBlank(account , "用户账号信息（一般是手机号）必须传入!");
    // 验证唯一性
    UserEntity currentUser = this.userRepository.findByAccount(account);
    Validate.isTrue(currentUser == null , "该账号信息已经被使用，请重新填写!");
    // 检测可能填写的电话信息
    String phone = userVo.getPhone();
    if(!StringUtils.isBlank(phone)) {
      currentUser = this.userRepository.findByPhone(phone);
      Validate.isTrue(currentUser == null , "该联系方式已经被使用，请重新填写!");
    } else {
      userVo.setPhone(null);
    }
    userVo.setCreateTime(new Date());
    userVo.setUseStatus(1);
    // 验证初始填入的密码信息
    String password = userVo.getPassword();
    Validate.notBlank(password , "密码信息必须传入!");
    userVo.setPassword(passwordEncoder.encode(password));
    // 开始添加
    currentUser = this.nebulaToolkitService.copyObjectByWhiteList(userVo, UserEntity.class, HashSet.class, ArrayList.class, new String[]{});
    this.userRepository.saveAndFlush(currentUser);
    //默认添加一个用户后，为该用户绑定基础角色
    RoleEntity baseRole = roleRepository.findByRoleName(BASE_ROLE_NAME);
    Validate.notNull(baseRole,"新增用户绑定基础角色时，未找到基础角色");
    roleService.bindRoles(currentUser.getId(),new String[]{baseRole.getRoleName()});
    
    userVo.setId(currentUser.getId());
    return userVo;
  }
  
  /* (non-Javadoc)
   * @see com.bizunited.platform.rbac.server.service.UserService#update(com.bizunited.platform.rbac.server.vo.UserVo)
   */
  @Override
  @Transactional
  public UserVo update(UserVo user) {
    Validate.notNull(user , "必须传入要修改的用户信息!");
    String userId = user.getId();
    String userName = user.getUserName();
    String account = user.getAccount();
    Validate.notBlank(userId , "修改时，用户数据的编号信息必须传入!");
    Validate.notBlank(userName , "用户真实姓名必须传入!");
    Validate.notBlank(account , "用户账号信息（一般是手机号）必须传入!");
    // 验证新的账户的唯一性
    // 验证唯一性
    UserEntity currentUser = this.userRepository.findByAccount(account);
    Validate.isTrue(currentUser == null || StringUtils.equals(currentUser.getId(), userId), "该账号信息已经被使用，请重新填写!");
    // 检测可能填写的电话信息
    String phone = user.getPhone();
    if(!StringUtils.isBlank(phone)) {
      currentUser = this.userRepository.findByPhone(phone);
      Validate.isTrue(currentUser == null || StringUtils.equals(currentUser.getId(), userId), "指定的电话信息已被使用，请检查!!");
    }

    // 开始更新(这个方法只能更新这些信息，其他信息即使传入也不更新)
    Optional<UserEntity> op = this.userRepository.findById(userId);
    UserEntity oldUser = op.orElse(null);
    Validate.notNull(oldUser , "未找到原始的用户信息，请检查!!");
    oldUser.setAccount(account);
    oldUser.setEntryTime(user.getEntryTime());
    oldUser.setPhone(phone);
    oldUser.setUserHead(user.getUserHead());
    oldUser.setUserName(userName);

    this.userRepository.save(oldUser);
    return user;
  }
  
  /* (non-Javadoc)
   * @see com.bizunited.platform.rbac.server.service.UserService#updatePassword(java.lang.String, java.lang.String)
   */
  @Override
  @Transactional
  public UserVo updatePassword(String userId, String newPassword) {
    Validate.notBlank(userId, "修改密码时 ,用户编号不能为空!!");
    Validate.notBlank(newPassword, "修改密码时 ,新的密码不能为空!!");

    Optional<UserEntity> op = this.userRepository.findById(userId);
    UserEntity user = op.orElse(null);
    Validate.notNull(user , "指定的用户没有被找到，请检查!!");

    user.setPassword(passwordEncoder.encode(newPassword));
    userRepository.saveAndFlush(user);
    
    // 转换成VO
    UserVo userVo = this.nebulaToolkitService.copyObjectByWhiteList(user, UserVo.class, HashSet.class, ArrayList.class, new String[]{});
    return userVo;
  }
  /* (non-Javadoc)
   * @see com.bizunited.platform.rbac.server.service.UserService#updatePassword(java.lang.String, java.lang.String, java.lang.String)
   */
  @Override
  @Transactional
  public UserVo updatePassword(String userId, String newPassword, String oldPassword) {
    Validate.notBlank(userId, "修改密码时 ,用户编号不能为空!!");
    Validate.notBlank(newPassword, "修改密码时 ,新的密码不能为空!!");
    Validate.notBlank(oldPassword, "修改密码时 ,原有的密码不能为空!!");

    // 对原来的密码进行加密后判断
    Optional<UserEntity> op = this.userRepository.findById(userId);
    UserEntity user = op.orElse(null);
    Validate.notNull(user , "指定的用户没有被找到，请检查!!");
    Validate.isTrue(passwordEncoder.matches(oldPassword, user.getPassword()), "原始密码输入错误，请检查!!");

    // 开始更新
    user.setPassword(passwordEncoder.encode(newPassword));
    // 转换成VO
    UserVo userVo = this.nebulaToolkitService.copyObjectByWhiteList(user, UserVo.class, HashSet.class, ArrayList.class, new String[]{});
    return userVo;
  }
  
  @Override
  @Transactional
  public void updateLastloginTime(String account) {
    Validate.notBlank(account, "修改最后登录时间时 ,用户账号不能为空!!");
    Date lastloginTime = new Date();

    this.userRepository.updateLastloginTime(account, lastloginTime);
    this.userRepository.flush();
  }

  /* (non-Javadoc)
   * @see com.bizunited.platform.rbac.server.service.UserService#enabled(java.lang.String)
   */
  @Override
  @Transactional
  public UserVo enabled(String account) {
    Validate.notBlank(account, "用户账号必须传入!!");
    UserEntity user = userRepository.findByAccount(account);
    Validate.notNull(user , "未找到需要修改的用户信息，请检查!!");

    user.setUseStatus(1);
    userRepository.save(user);
    return this.nebulaToolkitService.copyObjectByWhiteList(user, UserVo.class, HashSet.class, ArrayList.class, new String[]{});
  }

  /* (non-Javadoc)
   * @see com.bizunited.platform.rbac.server.service.UserService#disable(java.lang.String)
   */
  @Override
  @Transactional
  public UserVo disable(String account) {
    Validate.notBlank(account, "用户账号必须传入!!");
    UserEntity user = userRepository.findByAccount(account);
    Validate.notNull(user , "未找到需要修改的用户信息，请检查!!");

    user.setUseStatus(0);
    userRepository.save(user);
    return this.nebulaToolkitService.copyObjectByWhiteList(user, UserVo.class, HashSet.class, ArrayList.class, new String[]{});
  }
  
  /* (non-Javadoc)
   * @see com.bizunited.platform.rbac.server.service.UserService#findById(java.lang.String)
   */
  @Override
  public UserVo findDetailsById(String id) {
    if (StringUtils.isBlank(id)) {
      return null;
    }
    // 用户详细信息，但是不包括用户角色，因为用户角色需要单独查
    UserEntity currentUser = userRepository.findDetailsById(id);
    if(currentUser == null) {
      return null;
    }
    UserVo userVo = this.nebulaToolkitService.copyObjectByWhiteList(currentUser, UserVo.class, HashSet.class, ArrayList.class, new String[]{"positions","orgs","groups"});
    
    // 查询所有符合要求的角色进行绑定
    List<RoleVo> roles = this.roleService.findAllByUserId(id);
    userVo.setRoles(CollectionUtils.isEmpty(roles)?new HashSet<>():new HashSet<>(roles));
    return userVo;
  }
  
  /* (non-Javadoc)
   * @see com.bizunited.platform.rbac.server.service.UserService#findByAccount(java.lang.String)
   */
  @Override
  public UserVo findByAccount(String account) {
    if (StringUtils.isBlank(account)) {
      return null;
    }
    // 用户详细信息，但是不包括用户角色，等其它关联信息
    UserEntity currentUser = this.userRepository.findByAccount(account);
    if(currentUser == null) {
      return null;
    }
    return this.nebulaToolkitService.copyObjectByWhiteList(currentUser, UserVo.class, HashSet.class, ArrayList.class, new String[]{}); 
  }
  
  /* (non-Javadoc)
   * @see com.bizunited.platform.rbac.server.service.UserService#findByConditions(java.lang.String, java.lang.String, java.lang.Integer, org.springframework.data.domain.Pageable)
   */
  @Override
  public Page<UserVo> findByConditions(String userName, String account, Integer useStatus, Pageable pageable) {
    Map<String, Object> conditions = new HashMap<>();
    if (StringUtils.isNotBlank(userName)) {
      conditions.put("userName", userName);
    }
    if (StringUtils.isNotBlank(account)) {
      conditions.put("account", account);
    }
    if (useStatus != null) {
      conditions.put("useStatus", useStatus);
    }
    // 如果当前没有设定分页信息，则默认第一页，每页50条数据
    if(pageable == null) {
      pageable = PageRequest.of(0, 50);
    }
    
    // Page需要自己转
    Page<UserEntity> userEntityPage = this.userRepository.queryPage(pageable, conditions);
    List<UserEntity> userEntitys = userEntityPage.getContent();
    Page<UserVo> userVoPage = null;
    if(userEntitys != null && !userEntitys.isEmpty()) {
      Collection<UserVo> userVos = this.nebulaToolkitService.copyCollectionByWhiteList(userEntitys, UserEntity.class, UserVo.class, LinkedHashSet.class, ArrayList.class, new String[]{"roles"});
      userVoPage = new PageImpl<UserVo>(new ArrayList<>(userVos), pageable, userEntityPage.getTotalElements());
    } else {
      userVoPage = new PageImpl<UserVo>(Lists.newArrayList(), pageable, 0l);
    }
    return userVoPage;
  }

  /* (non-Javadoc)
   * @see com.bizunited.platform.rbac.server.service.UserService#findByUserId(java.lang.String)
   */
  @Override
  public UserVo findByUserId(String userId) {
    Validate.notBlank(userId,"传入的用户ID不能为空！");
    UserEntity user= userRepository.findDetailsById(userId);
    if(user==null){
      return null;
    }
    return this.nebulaToolkitService.copyObjectByWhiteList(user, UserVo.class, HashSet.class, ArrayList.class, new String[]{});
  }

  @Override
  public Set<UserVo> findByIds(List<String> ids) {
    if(CollectionUtils.isEmpty(ids)){
      return Sets.newHashSet();
    }
    Set<UserEntity> users = userRepository.findByIds(ids);
    if(CollectionUtils.isEmpty(users)){
      return Sets.newHashSet();
    }
    return (Set<UserVo>)this.nebulaToolkitService.copyCollectionByWhiteList(users,UserEntity.class,UserVo.class,HashSet.class,ArrayList.class,new String[]{});
  }

  @Override
  public UserVo findByPhone(String phone) {
    if(StringUtils.isBlank(phone)) {
      return null;
    }
    
    UserEntity user= this.userRepository.findByPhone(phone);
    if(user==null){
      return null;
    }
    return this.nebulaToolkitService.copyObjectByWhiteList(user, UserVo.class, HashSet.class, ArrayList.class, new String[]{});
  }
}
