package com.bizunited.platform.core.configuration;

import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.Arrays;
import java.util.Date;
import java.util.List;
import java.util.concurrent.ThreadFactory;
import java.util.concurrent.atomic.AtomicInteger;

import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.apache.commons.lang3.time.DateUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.boot.CommandLineRunner;
import org.springframework.context.ApplicationContext;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.scheduling.annotation.EnableAsync;
import org.springframework.scheduling.concurrent.ThreadPoolTaskScheduler;
import org.springframework.scheduling.support.CronTrigger;
import org.springframework.stereotype.Component;

import com.bizunited.platform.core.annotations.DynamicTaskService;
import com.bizunited.platform.core.entity.DynamicTaskSchedulerEntity;
import com.bizunited.platform.core.service.DynamicTaskSchedulerService;
import com.bizunited.platform.core.service.invoke.InvokeProxyContext;
import com.bizunited.platform.core.service.scheduler.DynamicTaskNotifyMaster;
import com.google.common.collect.Lists;

import javassist.ClassPool;
import javassist.CtClass;
import javassist.CtMethod;
import javassist.NotFoundException;

/**
 * 动态任务管理器的配置，动态任务管理器所管理的任务全部通过groovy进行驱动，并且可以实现动态启停、错误日志记录等操作
 * @author yinwenjie
 */
@Configuration
@EnableAsync
@Component("_schedulerConfig")
public class SchedulerConfig implements CommandLineRunner  {
  @Autowired
  private ApplicationContext applicationContext;
  @Autowired
  private DynamicTaskNotifyMaster dynamicTaskNotifyMaster;
  private ClassPool pool = ClassPool.getDefault();
  @Autowired
  private DynamicTaskSchedulerService dynamicTaskSchedulerService;
  /**
   * 当前应用程序名
   */
  @Value("${spring.application.name:default}")
  private String applicationName;
  
  @Bean("platform_dynamicTaskScheduler")
  public ThreadPoolTaskScheduler getThreadPoolTaskScheduler() {
    ThreadPoolTaskScheduler scheduler = new ThreadPoolTaskScheduler();
    // 配置参数暂时不开放给开发人员
    scheduler.setBeanName("platform_dynamicTaskScheduler");
    scheduler.setDaemon(false);
    scheduler.setPoolSize(50);
    scheduler.setThreadFactory(getThreadFactory());
    scheduler.setThreadNamePrefix("platform_");
    return scheduler;
  }
  
  /**
   * 自定义ThreadFactory，主要是为了指定classLoader和一个可以很好做区别的threadName
   * @return
   */
  private ThreadFactory getThreadFactory() {
    ThreadFactory dynamicTaskSchedulerFactory = new ThreadFactory() {
      private AtomicInteger count = new AtomicInteger();
      
      @Override
      public Thread newThread(Runnable r) {
        Thread currentThread = new Thread(r);
        ClassLoader classLoader = applicationContext.getClassLoader();
        currentThread.setDaemon(false);
        currentThread.setName("dynamicTask_" + count.getAndIncrement());
        currentThread.setContextClassLoader(classLoader);
        return currentThread;
      }
    };
    return dynamicTaskSchedulerFactory;
  }

  @Override
  public void run(String... args) throws Exception {
    /*
     * 由于2019-11月28日，对业务构建平台提供的定时器功能进行了调整，所以整个处理过程调整如下：
     * 
     * 在系统初始化启动时，该涉及业务构建平台动态定时任务的启动器会做以下动作：
     * 1、检查当前所有已在数据库中持久化存储的所有Groovy执行性质的，拥有corn表达式的所有定时任务，其属性信息是满足合规性的
     * 2、分析当前系统中当前基于DynamicTaskService注解的java method执行任务，并同步更新相关数据库中记录的信息
     * 3、检测成功后，开始对动态调度任务的执行权的抢占
     * 并在其中的dynamicTaskSchedulerService.loading()方法中，完成基于Groovy脚本或者java method注解的定时任务的加载过程
     * */
    
    // 1、TODO 第一步骤有时间再补上
    
    // 2、=======
    String[] definitionNames = applicationContext.getBeanDefinitionNames();
    List<DynamicTaskSchedulerEntity> currentScanTasks = Lists.newArrayList();
    for (String definitionName : definitionNames) {
      List<CtMethod> nebulaServiceMethods = Lists.newArrayList();
      Class<?> beanClass = applicationContext.getType(definitionName);
      String beanClassName = beanClass.getName();
      // 如果这个类被Spring代理，或者是一个内隐类，则判定其主类
      if(beanClassName.indexOf("$") != -1) {
        beanClassName = beanClassName.substring(0, beanClassName.indexOf("$"));
      }
      if(StringUtils.isBlank(beanClassName)) {
        continue;
      }
      CtClass beanCtClass = null;
      try {
        beanCtClass = pool.get(beanClassName);
      } catch(NotFoundException e) {
        continue;
      }
      // 试图获取这个class中带有DynamicTaskService注解的方法
      Arrays.stream(beanCtClass.getDeclaredMethods()).filter(item -> item.hasAnnotation(DynamicTaskService.class)).forEach(methodItem -> {
        nebulaServiceMethods.add(methodItem);
      });
      // 开始对这些method进行验证-并进行数据库同步
      if(!nebulaServiceMethods.isEmpty()) {
        currentScanTasks.addAll(this.analysisMethods(nebulaServiceMethods, definitionName));
      }
    }
    this.dynamicTaskSchedulerService.updateExsitReflexTasks(currentScanTasks);
    
    // 3、=======
    dynamicTaskNotifyMaster.notifyMaster();
  }
  
  /**
   * 该死有方法，分析指定spring实现类中的所有被标注了DynamicTaskService注解的服务，并形成对应的DynamicTaskSchedulerEntity集合
   * @param nebulaServiceMethods 指定的java method信息
   * @param definitionName 所属的spring接口的定义名
   * @return 
   * @throws ClassNotFoundException
   * @throws NotFoundException
   */
  private List<DynamicTaskSchedulerEntity> analysisMethods(List<CtMethod> nebulaServiceMethods , String definitionName) throws ClassNotFoundException , NotFoundException {
    // currentTasks变量记录当前初始化扫描的所有基于DynamicTaskService注解的任务信息
    List<DynamicTaskSchedulerEntity> currentScanTasks = Lists.newArrayList();
    SimpleDateFormat dateFormat = new SimpleDateFormat("yyyy-MM-dd HH:mm:ss");
    for (CtMethod ctMethod : nebulaServiceMethods) {
      DynamicTaskService dynamicTaskService  = (DynamicTaskService)ctMethod.getAnnotation(DynamicTaskService.class);
      if(dynamicTaskService == null) {
        continue;
      }
      String cornExpression = dynamicTaskService.cornExpression();
      String executePoint = dynamicTaskService.executePoint();
      String taskDesc = dynamicTaskService.taskDesc();
      String validityPoint = dynamicTaskService.validityPoint();
      boolean onlyOnce = dynamicTaskService.onlyOnce();
      String methodName = ctMethod.getName();
      // 将会当成task code使用
      String methodLongName = ctMethod.getLongName();
      CtClass[] parameterTypes = ctMethod.getParameterTypes();
      
      // 开始进行有效性验证
      DynamicTaskSchedulerEntity currentTask = new DynamicTaskSchedulerEntity();
      Validate.isTrue(parameterTypes == null || parameterTypes.length == 0 || (parameterTypes.length == 1 && parameterTypes[0] == pool.getCtClass(InvokeProxyContext.class.getName()))
          , "使用DynamicTaskService注解的方法，必须托管给Spring，并且目前只允许没有入参或者入参为com.bizunited.platform.core.service.invoke.InvokeProxyContext的方法，请检查%s" , methodLongName);
      Validate.notBlank(taskDesc , "基于DynamicTaskService注解的定时器任务并没有说明其业务意义（expressionDesc），请检查%s!", methodLongName);
      
      // 如果条件成立，就是一次性任务，否则就是周期性任务
      if(onlyOnce) {
        // 判定并设置执行时间信息（一次性执行任务有两种设定方式）
        if(!StringUtils.isBlank(executePoint)) {
          Date executePointTime = null;
          try {
            executePointTime = dateFormat.parse(executePoint);
          } catch(ParseException e) {
            throw new IllegalArgumentException(String.format("一次性定时任务%s设定的日期格式存在问题，请检查", methodLongName));
          }
          currentTask.setExecutePoint(executePointTime);
          // 默认设定当前一次性任务会在执行时间点（executePoint）之后的1小时内自动到期
          Date validityTime = DateUtils.addMinutes(executePointTime, 60);
          currentTask.setValidityTime(validityTime);
        } else if(!StringUtils.isBlank(cornExpression)) {
          try {
            new CronTrigger(cornExpression);
          } catch(RuntimeException e) {
            throw new IllegalArgumentException(String.format("一次性任务%s设定的corn表达是存在格式问题，请检查", methodLongName));
          }
          
          currentTask.setExecuteExpression(cornExpression);
          currentTask.setExpressionDesc(cornExpression);
        } else {
          throw new IllegalArgumentException(String.format("当任务是一次性执行任务时，要么设定corn表达式（cornExpression），要么设定确定的执行时间点（executePoint），请检查任务%s", methodLongName));
        }
        currentTask.setTaskType(1);
      } else {
        if(!StringUtils.isBlank(cornExpression)) {
          try {
            new CronTrigger(cornExpression);
          } catch(RuntimeException e) {
            throw new IllegalArgumentException(String.format("周期性执行任务%s设定的corn表达是存在格式问题，请检查", methodLongName));
          }
          currentTask.setExecuteExpression(cornExpression);
          currentTask.setExpressionDesc(cornExpression);
        } else {
          throw new IllegalArgumentException(String.format("当任务是周期性执行任务时，要设定corn表达式（cornExpression），请检查任务%s", methodLongName));
        }
        currentTask.setTaskType(2);
      }
      
      // 如果用户自行设定了失效时间
      if(!StringUtils.isBlank(validityPoint)) {
        Date validityPointTime = null;
        try {
          validityPointTime = dateFormat.parse(validityPoint);
        } catch(ParseException e) {
          throw new IllegalArgumentException(String.format("定时任务%s设定的失效时间日期格式存在问题，请检查", methodLongName));
        }
        currentTask.setValidityTime(validityPointTime);
      }
      // 当前定时器的业务意义
      currentTask.setTaskDesc(taskDesc);
      currentTask.setInvokeType(2);
      // 将完整的方法名作为任务code
      currentTask.setTaskCode(methodLongName);
      currentTask.setScriptName("");
      // 其它所需要的反射信息记录
      currentTask.setInvokeBeanName(definitionName);
      currentTask.setInvokeMethod(methodName);
      // 是否需要携带InvokeProxyContext入参
      currentTask.setProxyContextParam(parameterTypes != null && parameterTypes.length == 1);
      // 当前应用程序名
      currentTask.setApplicationName(this.applicationName);
      currentScanTasks.add(currentTask);
    }
    return currentScanTasks;
  }
}