package com.bizunited.platform.core.controller;

import com.bizunited.platform.common.controller.BaseController;
import com.bizunited.platform.common.controller.model.ResponseModel;
import com.bizunited.platform.core.entity.MigrateExportEntity;
import com.bizunited.platform.core.service.migrate.MigrateExportService;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import io.swagger.annotations.ApiParam;
import org.apache.commons.lang3.ArrayUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.RestController;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import java.text.SimpleDateFormat;
import java.util.Date;
import java.util.List;

import static com.bizunited.platform.common.constant.MigrateDataConstants.EXPORT_FILENAME_DATE_FORMAT;
import static com.bizunited.platform.common.constant.MigrateDataConstants.ZIP_FILE_SUBFIX;

@Api(value = "MigrateExportController")
@RestController
@RequestMapping("/v1/nebula/migrateExports")
public class MigrateExportController extends BaseController {
  
  private static final Logger LOGGER = LoggerFactory.getLogger(MigrateExportController.class);

  @Autowired
  private MigrateExportService migrateExportService;

  /**
   * 根据数据类型查询导出日志
   * @param dataType
   * @return
   */
  @GetMapping("findDetailsByDataType")
  @ApiOperation("根据数据类型查询导出日志")
  public ResponseModel findDetailsByDataType(Integer dataType) {
    try {
      List<MigrateExportEntity> exports = migrateExportService.findDetailsByDataType(dataType);
      return buildHttpResultW(exports, "createUser");
    } catch (RuntimeException e) {
      LOGGER.error(e.getMessage(), e);
      return buildHttpResultForException(e);
    }
  }

  /**
   * 导出数据视图的设计模型
   */
  @GetMapping("exportDataView")
  @ApiOperation(value = "导出数据视图的设计模型", notes = "传入导出的数据ID则按照传入的数据ID来导出，否则导出全部")
  public void exportDataView(@RequestParam(required = false) @ApiParam("筛选的数据ID") String[] ids,
                             HttpServletRequest request, HttpServletResponse response) {
    try {
      byte[] bytes = migrateExportService.exportDataView(ids);
      Validate.isTrue(ArrayUtils.isNotEmpty(bytes), "没有任何下载信息，很大几率原因是导出逻辑出现错误!!");
      String time = new SimpleDateFormat(EXPORT_FILENAME_DATE_FORMAT).format(new Date());
      String fileName = StringUtils.join("数据视图-", time, ".", ZIP_FILE_SUBFIX);
      writeResponseFile(request, response, bytes, fileName);
    } catch (RuntimeException e) {
      LOGGER.error(e.getMessage(), e);
      printResponseModel(buildHttpResultForException(e), response);
    }
  }

  /**
   * 导出远端调用的设计模型
   */
  @GetMapping("exportRemoteService")
  @ApiOperation(value = "导出远端调用的设计模型", notes = "传入导出的数据ID则按照传入的数据ID来导出，否则导出全部")
  public void exportRemoteService(@RequestParam(required = false) @ApiParam("筛选的数据ID") String[] ids,
                                  HttpServletRequest request, HttpServletResponse response) {
    try {
      byte[] bytes = migrateExportService.exportRemoteService(ids);
      Validate.isTrue(ArrayUtils.isNotEmpty(bytes), "没有任何下载信息，很大几率原因是导出逻辑出现错误!!");
      String time = new SimpleDateFormat(EXPORT_FILENAME_DATE_FORMAT).format(new Date());
      String fileName = StringUtils.join("远端地址-", time, ".", ZIP_FILE_SUBFIX);
      writeResponseFile(request, response, bytes, fileName);
    } catch (RuntimeException e) {
      LOGGER.error(e.getMessage(), e);
      printResponseModel(buildHttpResultForException(e), response);
    }
  }

  /**
   * 导出指定编号的业务编码
   * @param codeRuleIds
   * @param request
   * @param response
   */
  @ApiOperation(value = "导出指定编号的业务编码", notes = "传入导出的数据ID则按照传入的数据ID来导出，否则导出全部")
  @GetMapping("/exportCodeRule")
  public void exportCodeRule(@RequestParam(required = false) @ApiParam("筛选的数据ID") String[] codeRuleIds, HttpServletRequest request, HttpServletResponse response) {
    try {
      byte[] bytes = this.migrateExportService.exportCodeRule(codeRuleIds);
      Validate.isTrue(bytes != null && bytes.length != 0, "没有任何下载信息，很大几率原因是导出逻辑出现错误!!");
      String time = new SimpleDateFormat("yyyyMMddHHmm").format(new Date());
      String fileName = StringUtils.join("编码规则-", time, ".zip");
      writeResponseFile(request, response, bytes, fileName);
    } catch(Exception e) {
      LOGGER.error(e.getMessage() , e);
      this.printResponseModel(this.buildHttpResultForException(e), response);
    }
  }

  @ApiOperation(value = "导出指定编号的全局变量", notes = "传入导出的数据ID则按照传入的数据ID来导出，否则导出全部")
  @GetMapping("/exportEnv")
  public void exportEnv(@RequestParam(required = false) @ApiParam("筛选的数据ID") String[] envIds, HttpServletRequest request, HttpServletResponse response) {
    try {
      byte[] bytes = this.migrateExportService.exportEnv(envIds);
      Validate.isTrue(bytes != null && bytes.length != 0, "没有任何下载信息，很大几率原因是导出逻辑出现错误!!");
      String time = new SimpleDateFormat("yyyyMMddHHmm").format(new Date());
      String fileName = StringUtils.join("全局参数-", time, ".zip");
      writeResponseFile(request, response, bytes, fileName);
    } catch(Exception e) {
      LOGGER.error(e.getMessage() , e);
      this.printResponseModel(this.buildHttpResultForException(e), response);
    }
  }
}