package com.bizunited.platform.core.controller;

import com.alibaba.fastjson.JSON;
import com.bizunited.platform.common.controller.BaseController;
import com.bizunited.platform.common.controller.model.ResponseModel;
import com.bizunited.platform.core.entity.ScriptEntity;
import com.bizunited.platform.core.service.ScriptService;
import com.bizunited.platform.core.service.script.model.ScriptInputParamsModel;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.Pageable;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestMethod;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.RestController;

/**
 * ScriptController
 *
 * @description:
 * @author: yanwe
 * @date: 27/May/2019 15:45
 */
@Api(value = "ScriptController", tags = "脚本链调用")
@RestController
@RequestMapping("/v1/nebula/scripts")
public class ScriptController extends BaseController {
  @Autowired 
  private ScriptService scriptService;

  @ApiOperation(value = "后端脚本执行")
  @RequestMapping(value = "/invoke", method = RequestMethod.POST)
  public ResponseModel invoke(@RequestBody ScriptInputParamsModel scriptInputParamsModel) {
    try {
      Validate.notNull(scriptInputParamsModel, "入参不能为空！");
      Validate.notEmpty(scriptInputParamsModel.getScriptIds(), "脚本ID集合不能为空");
      Object result = scriptService.invoke(scriptInputParamsModel.getScriptIds().toArray(new String[] {}) , scriptInputParamsModel.getParams());
      return buildHttpResultW(result);
    } catch (Exception e) {
      return buildHttpResultForException(e);
    }
  }

  @ApiOperation(value = "分页查询")
  @RequestMapping(value = "/findByConditions", method = RequestMethod.GET)
  public ResponseModel findByConditions(
      @RequestParam(name = "name") String name,
      @RequestParam(name = "language") String language, Pageable pageable) {
    try {
      Page<ScriptEntity> page = scriptService.findByConditions(pageable,name,language);
      return buildHttpResultW(page);
    } catch (Exception e) {
      return buildHttpResultForException(e);
    }
  }
  /**
   * 获取脚本内容根据脚本唯一识别名进行查询
   * @param scriptId
   * @return
   */
  @ApiOperation(value = "获取脚本内容根据脚本唯一识别名进行查询")
  @RequestMapping(value = "/findContentByName", method = RequestMethod.GET)
  public ResponseModel findContentByName(@RequestParam(name = "scriptName")  String scriptName) {
    try {
      String content = scriptService.findContentByName(scriptName);
      return buildHttpResultW(content);
    } catch (Exception e) {
      return buildHttpResultForException(e);
    }
  }
  
  @ApiOperation(value = "根据ID查询脚本内容")
  @RequestMapping(value = "/findContentById", method = RequestMethod.GET)
  public ResponseModel findContentById(
      @RequestParam(name = "scriptId") String scriptId) {
    try {
      String content = scriptService.findContentById(scriptId);
      return buildHttpResultW(content);
    } catch (Exception e) {
      return buildHttpResultForException(e);
    }
  }

  @ApiOperation(value = "根据ID查询脚本基础信息")
  @RequestMapping(value = "/findById", method = RequestMethod.GET)
  public ResponseModel findById(
      @RequestParam(name = "scriptId") String scriptId) {
    try {
      ScriptEntity entity = scriptService.findById(scriptId);
      return buildHttpResultW(entity);
    } catch (Exception e) {
      return buildHttpResultForException(e);
    }
  }

  @ApiOperation(value = "保存脚本致文件系统")
  @RequestMapping(value = "", method = RequestMethod.POST)
  public ResponseModel create(
      @RequestParam(name = "scriptEntity") String scriptEntityStr,
      @RequestParam(name = "scriptContent") String scriptContent) {
    try {
      ScriptEntity scriptEntity = JSON.parseObject(scriptEntityStr, ScriptEntity.class);
      ScriptEntity result = scriptService.create(scriptEntity, scriptContent);
      return buildHttpResultW(result);
    } catch (Exception e) {
      return buildHttpResultForException(e);
    }
  }

  @ApiOperation(value = "修改脚本致文件系统")
  @RequestMapping(value = "/update", method = RequestMethod.PATCH)
  public ResponseModel update(
      @RequestParam(name = "scriptEntity") String scriptEntityStr,
      @RequestParam(name = "scriptContent") String scriptContent) {
    try {
      ScriptEntity scriptEntity = JSON.parseObject(scriptEntityStr, ScriptEntity.class);
      ScriptEntity result = scriptService.update(scriptEntity, scriptContent);
      return buildHttpResultW(result);
    } catch (Exception e) {
      return buildHttpResultForException(e);
    }
  }
}