package com.bizunited.platform.core.controller;

import com.alibaba.fastjson.JSONObject;
import com.bizunited.platform.common.controller.BaseController;
import com.bizunited.platform.common.controller.model.ResponseModel;
import com.bizunited.platform.core.service.invoke.InvokeParams;
import com.bizunited.platform.core.service.serviceable.ServicableMethodService;
import com.bizunited.platform.core.service.serviceable.model.ServicableMethodInfo;

import io.swagger.annotations.Api;
import io.swagger.annotations.ApiImplicitParam;
import io.swagger.annotations.ApiImplicitParams;
import io.swagger.annotations.ApiOperation;
import io.swagger.annotations.ApiParam;

import java.util.Map;
import java.util.Map.Entry;
import java.util.Set;

import javax.servlet.http.HttpServletRequest;

import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.Pageable;
import org.springframework.data.web.PageableDefault;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.RestController;


@Api(value = "ServicableMethodController",tags="向外界暴露的服务源接口信息")
@RestController
@RequestMapping("/v1/nebula/servicableMethods")
public class ServicableMethodController extends BaseController {
  /**
   * 日志
   */
  private static final Logger LOGGER = LoggerFactory.getLogger(ServicableMethodController.class);
  private static final String[] EMPTY_STRING_ARR = new String[] {};
  @Autowired
  private ServicableMethodService servicableMethodService;
  
  @ApiOperation(value = "按照条件搜索可用服务源列表，支持分页但不支持模糊查询")
  @GetMapping("/findByConditions")
  public ResponseModel findByConditions(@ApiParam(name = "name", value = "可能的服务源标识名称", required = false) @RequestParam(required=false , name="name") String name,
                                        @ApiParam(name = "description", value = "方法中文描述（支持模糊查询）", required = false) @RequestParam(required=false , name="description") String description,
                                        @ApiParam(name = "interfaceName", value = "可能的服务源完整接口类名查询条件", required = false) @RequestParam(required=false , name="interfaceName") String interfaceName,
                                        @ApiParam(name = "simpleMethodName", value = "可能的服务源方法名简称查询条件", required = false) @RequestParam(required=false , name="simpleMethodName") String simpleMethodName,
                                        @ApiParam(name = "returnClassName", value = "可能的返回classname（全名）作为查询条件", required = false) @RequestParam(required=false , name="returnClassName") String returnClassName ,
                                        @ApiParam(name = "usedScope", value = "服务源方法的应用范围：目前支持：读操作(READ)，写操作(WRITE)", required = false) @RequestParam(required=false , value="usedScope") String usedScope,
                                        @PageableDefault(value = 50) Pageable pageable) {
    try {
      Page<JSONObject> page = this.servicableMethodService.findByConditions(pageable, name , description , interfaceName, simpleMethodName , returnClassName ,usedScope);
      return this.buildHttpResultW(page, new String[] {});
    } catch (RuntimeException e) {
      LOGGER.error(e.getMessage(), e);
      return this.buildHttpResultForException(e);
    }
  }
  
  @ApiOperation(value = "根据所暴露的服务源，用动态代理的方式调用接口，该功能会使用缓存机制", 
      notes = "该方法必须包含serviceName的值。model参数包含4部分：serviceName服务名(必传)，params是k-v结构的，customData也是k-v结构，formData是表单数据。")
  @ApiImplicitParams({@ApiImplicitParam(name = "model", value = "封装的传入参数（serviceName必传）", required = true)})
  @PostMapping("/servicableMethodInvoke")
  public ResponseModel invoke(HttpServletRequest request ,@RequestParam("serviceName") String serviceName , @RequestBody InvokeParams invokeParams) {
    // 对服务源信息进行校验
    InvokeParams currentInvokeParams = invokeParams;
    ServicableMethodInfo servicableMethodInfo = null;
    try {
      Validate.notBlank(serviceName, "服务源名称serviceName不能为空，请检查!!");
      servicableMethodInfo = servicableMethodService.findDetailsByName(serviceName);
      Validate.notNull(servicableMethodInfo,"根据服务名%s，没有查询到服务的相关信息，请检查!!" , serviceName);
      if(currentInvokeParams == null) {
        currentInvokeParams = new InvokeParams();
      }
    } catch (RuntimeException e) {
      LOGGER.error(e.getMessage(),e);
      return buildHttpResultForException(e);
    }
    
    /*
     * 在完成边界校验后，入参的汇聚方式如下所示：
     * 1、汇聚request中parameters中的参数，注意数组和非数组的区别
     * 写入invokeParams，如果没有则新建invokeParams
     * 2、进行服务层调用
     * */
    Object result = null;
    try {
      // 1、======
      Map<String , String[]> parameterMaps = request.getParameterMap();
      if(parameterMaps != null) {
        Set<Entry<String, String[]>> parameterSets =  parameterMaps.entrySet();
        for (Entry<String, String[]> parameterItem : parameterSets) {
          String key = parameterItem.getKey();
          String[] values = parameterItem.getValue();
          currentInvokeParams.putInvokeParam(key, values.length == 1?values[0]:values);
        }
      }
      // 2、=======
      result = this.servicableMethodService.invoke(serviceName, currentInvokeParams);
      // 对结果进行白名单过滤处理并返回统一数据结构结果
      if(result == null) {
        return buildHttpResult();
      } else if(Page.class.isAssignableFrom(result.getClass())) {
        Page<?> yourPage = (Page<?>)result;
        return buildHttpResultW(yourPage, StringUtils.isBlank(servicableMethodInfo.getReturnPropertiesFilter()) ? EMPTY_STRING_ARR : servicableMethodInfo.getReturnPropertiesFilter().split(","));
      } else if(Iterable.class.isAssignableFrom(result.getClass())) {
        Iterable<?> iterable = (Iterable<?>) result;
        return buildHttpResultW(iterable , StringUtils.isBlank(servicableMethodInfo.getReturnPropertiesFilter()) ? EMPTY_STRING_ARR : servicableMethodInfo.getReturnPropertiesFilter().split(","));
      } else {
        return buildHttpResultW(result , StringUtils.isBlank(servicableMethodInfo.getReturnPropertiesFilter()) ? EMPTY_STRING_ARR : servicableMethodInfo.getReturnPropertiesFilter().split(","));
      }
    } catch (Exception e) {
      LOGGER.error(e.getMessage(),e);
      return buildHttpResultForException(e);
    }
  }
  @ApiOperation(value = "根据名称查询服务源详情",
          notes = "该方法必须包含serviceName的值。")
  @GetMapping("/findDetailsByName")
  public ResponseModel findDetailsByName(@RequestParam("serviceName") String serviceName) {
    JSONObject jsonObject = null;
    try {
      jsonObject = servicableMethodService.findDetailJsonByName(serviceName);
      return buildHttpResultW(jsonObject);
    } catch (RuntimeException e) {
      LOGGER.error(e.getMessage(), e);
      return buildHttpResultForException(e);
    }
  }
}