package com.bizunited.platform.core.controller;

import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONObject;
import com.bizunited.platform.common.controller.BaseController;
import com.bizunited.platform.common.controller.model.ResponseModel;
import com.bizunited.platform.common.service.redis.RedisMutexService;
import com.bizunited.platform.core.entity.SystemThemeEntity;
import com.bizunited.platform.core.service.SystemThemeService;
import com.bizunited.platform.venus.common.vo.OrdinaryFileVo;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import io.swagger.annotations.ApiParam;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestMethod;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.RestController;
import org.springframework.web.multipart.MultipartFile;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import java.io.IOException;
import java.io.OutputStream;
import java.nio.charset.StandardCharsets;
import java.security.Principal;
import java.util.UUID;

@Api(value = "SystemThemeController")
@RestController
@RequestMapping("/v1/nebula/theme")
public class SystemThemeController extends BaseController {
  /**
   * 日志
   * */
  private static final Logger LOGGER = LoggerFactory.getLogger(SystemThemeController.class);

  private static final String LOCK_THEME = "theme";

  @Autowired
  private SystemThemeService systemThemeService;
  @Autowired
  private RedisMutexService redisMutexService;



  @ApiOperation(value = "系统主题管理时，上传logo并保存(更新)主题")
  @RequestMapping(value = "/uploadLogo",method = RequestMethod.POST)
  public ResponseModel uploadLogo(@RequestParam("file") @ApiParam(name="file" , value="文件内容")MultipartFile file){
    try {
      Principal opUser = this.getPrincipal();
      OrdinaryFileVo theme = systemThemeService.uploadLogo(file,opUser);
      return this.buildHttpResultW(theme);
    } catch (Exception e) {
      LOGGER.error(e.getMessage(), e);
      return this.buildHttpResultForException(e);
    }
  }


  @ApiOperation(value = "系统主题管理时，保存(更新)主题")
  @RequestMapping(value = "",method = RequestMethod.POST)
  public ResponseModel save(@RequestBody @ApiParam(name="jsonTheme" , value="系统主题json格式数据")JSONObject jsonTheme){
    try {
      redisMutexService.lock(LOCK_THEME);
      Principal opUser = this.getPrincipal();
      SystemThemeEntity theme = systemThemeService.save(jsonTheme,opUser);
      return this.buildHttpResultW(theme);
    } catch (Exception e) {
      LOGGER.error(e.getMessage(), e);
      return this.buildHttpResultForException(e);
    }finally {
      redisMutexService.unlock(LOCK_THEME);
    }
  }

  @ApiOperation(value = "系统主题管理时，根据用户信息，加载主题")
  @RequestMapping(value = "/findTheme",method = RequestMethod.GET)
  public ResponseModel findTheme(){
    try {
      JSONObject theme = systemThemeService.findTheme();
      return this.buildHttpResultW(theme);
    } catch (Exception e) {
      LOGGER.error(e.getMessage(), e);
      return this.buildHttpResultForException(e);
    }
  }

  /**
   * 导出主题信息
   * @return
   */
  @ApiOperation(value = "导出主题信息")
  @GetMapping(value = "/export")
  public void exportTheme(HttpServletRequest request, HttpServletResponse response){
    try {
      byte[] result = this.systemThemeService.export();
      if(result == null || result.length == 0) {
        throw new IllegalArgumentException("没有任何下载信息，很大几率原因是导出逻辑出现错误!!");
      }
      // 随机生成下载内容的文件名（zip文件）
      String fileName = StringUtils.join(UUID.randomUUID().toString() , ".zip");
      this.writeResponseFile(request, response, result, fileName);
    } catch(Exception e) {
      LOGGER.error(e.getMessage() , e);
      this.printErrorMessage(this.buildHttpResultForException(e), response);
    }
  }

  /**
   * 输出错误信息
   * @param errorMessage
   * @param response
   */
  private void printErrorMessage(ResponseModel errorMessage , HttpServletResponse response) {
    response.setContentType("application/json;charset=UTF-8");
    OutputStream out = null;
    try {
      out = response.getOutputStream();
      out.write(JSON.toJSONString(errorMessage).getBytes(StandardCharsets.UTF_8));
    } catch (IOException e) {
      throw new IllegalArgumentException(e.getMessage() , e);
    }
  }

  /**
   * 导入主题信息
   * @param file
   * @param currentUser
   * @return
   */
  @ApiOperation(value = "导入主题信息")
  @PostMapping(value = "/import")
  public ResponseModel importTheme(@RequestParam(name = "file", required = true) @ApiParam(required = true, name = "file", value = "导入主题信息") MultipartFile file, Principal currentUser) {
    try {
      SystemThemeEntity entity = this.systemThemeService.importTheme(currentUser, file);
      return this.buildHttpResultW(entity);
    } catch (Exception e) {
      LOGGER.error(e.getMessage(), e);
      return this.buildHttpResultForException(e);
    }
  }
}
