package com.bizunited.platform.core.controller.dataview;

import com.bizunited.platform.common.controller.BaseController;
import com.bizunited.platform.common.controller.model.ResponseModel;
import com.bizunited.platform.core.entity.DataViewAuthEntity;
import com.bizunited.platform.core.service.dataview.DataViewAuthService;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import io.swagger.annotations.ApiParam;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.web.bind.annotation.DeleteMapping;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.PatchMapping;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.RestController;

import java.util.Set;

/**
 * 数据权限的控制层，实现restful风格的接口
 *
 * @Author: Paul Chan
 * @Date: 2020-04-23 11:21
 */
@RestController
@Api("数据权限")
@RequestMapping("/v1/nebula/dataview/auths")
public class DataViewAuthController extends BaseController {

  private static final Logger LOGGER = LoggerFactory.getLogger(DataViewAuthController.class);

  private static final String[] WHITE_DETAILS = new String[]{"dataView", "horizontalAuths", "horizontalAuths.authRelations",
      "verticalAuths.authRelations.relation", "verticalAuths", "verticalAuths.authRelations", "horizontalAuths.authRelations.relation",
      "interceptors"};

  @Autowired
  private DataViewAuthService dataViewAuthService;

  /**
   * 创建数据权限
   *
   * @param dataViewAuth
   * @return
   */
  @PostMapping("")
  @ApiOperation("创建数据权限")
  public ResponseModel create(@RequestBody DataViewAuthEntity dataViewAuth) {
    try {
      DataViewAuthEntity result = dataViewAuthService.create(dataViewAuth);
      return buildHttpResultW(result, WHITE_DETAILS);
    } catch (RuntimeException e) {
      LOGGER.error(e.getMessage(), e);
      return buildHttpResultForException(e);
    }
  }

  /**
   * 更新数据权限
   *
   * @param dataViewAuth
   * @return
   */
  @PatchMapping("")
  @ApiOperation("更新数据权限")
  public ResponseModel update(@RequestBody DataViewAuthEntity dataViewAuth) {
    try {
      DataViewAuthEntity result = dataViewAuthService.update(dataViewAuth);
      return buildHttpResultW(result);
    } catch (RuntimeException e) {
      LOGGER.error(e.getMessage(), e);
      return buildHttpResultForException(e);
    }
  }

  /**
   * 根据权限编码获取权限详情
   * @param code
   * @return
   */
  @GetMapping("findDetailsByCode")
  @ApiOperation("根据权限编码获取权限详情")
  public ResponseModel findDetailsByCode(@RequestParam @ApiParam("权限编码") String code) {
    try {
      DataViewAuthEntity auth = dataViewAuthService.findDetailsByCode(code);
      return buildHttpResultW(auth, WHITE_DETAILS);
    } catch (RuntimeException e) {
      LOGGER.error(e.getMessage(), e);
      return buildHttpResultForException(e);
    }
  }

  /**
   * 根据数据视图编码获取权限
   * @param dataViewCode
   * @return
   */
  @GetMapping("findByDataView")
  @ApiOperation("根据数据视图编码获取权限")
  public ResponseModel findByDataView(@RequestParam @ApiParam("数据视图编码")String dataViewCode) {
    try {
      Set<DataViewAuthEntity> auths = dataViewAuthService.findByDataView(dataViewCode);
      return buildHttpResultW(auths);
    } catch (RuntimeException e) {
      LOGGER.error(e.getMessage(), e);
      return buildHttpResultForException(e);
    }
  }

  /**
   * 根据ID删除
   * @param id
   * @return
   */
  @ApiOperation("根据ID删除")
  @DeleteMapping("deleteById")
  public ResponseModel deleteById(@RequestParam @ApiParam("主键ID") String id) {
    try {
      dataViewAuthService.deleteById(id);
      return buildHttpResult();
    } catch (RuntimeException e) {
      return buildHttpResultForException(e);
    }
  }

}
