package com.bizunited.platform.core.repository;

import com.bizunited.platform.core.entity.DynamicTaskSchedulerEntity;
import com.bizunited.platform.core.repository.internal.DynamicTaskSchedulerRepositoryCustom;
import org.springframework.data.jpa.repository.JpaRepository;
import org.springframework.data.jpa.repository.JpaSpecificationExecutor;
import org.springframework.data.jpa.repository.Modifying;
import org.springframework.data.jpa.repository.Query;
import org.springframework.data.repository.query.Param;
import org.springframework.stereotype.Repository;

import java.util.Set;

@Repository("DynamicTaskSchedulerRepository")
public interface DynamicTaskSchedulerRepository
    extends
      JpaRepository<DynamicTaskSchedulerEntity, String> ,
      JpaSpecificationExecutor<DynamicTaskSchedulerEntity> ,
      DynamicTaskSchedulerRepositoryCustom {
  
  /**
   * 按照指定的任务状态和工作状态查询符合条件的动态任务信息，并且按照任务code进行排序
   * @param tstatus 指定的动态任务状态
   * @param workingStatuses 指定的动态任务工作状态（可以是多个）
   * @return
   */
  @Query("from DynamicTaskSchedulerEntity dts where dts.tstatus = :tstatus AND dts.applicationName = :applicationName AND dts.workingStatus in (:workingStatuses) order by dts.taskCode ")
  public Set<DynamicTaskSchedulerEntity> findByTstatusAndWorkingStatusAndApplicationName(@Param("tstatus") int tstatus ,@Param("workingStatuses") int[] workingStatuses , @Param("applicationName") String applicationName);
  /**
   * 按照指定的定时器任务内容的执行方式和当前应用服务名，对任务进行分类查找：1：groovy脚本执行器；2：基于注解的Java method
   * @param invokeType 1：groovy脚本执行器；2：基于注解的Java method
   * @param applicationName 当前应用服务名
   * @return 
   */
  public Set<DynamicTaskSchedulerEntity> findByInvokeTypeAndApplicationName(@Param("invokeType") int invokeType ,@Param("applicationName") String applicationName);
  /**
   * 按照指定的任务状态查询符合要求的动态任务信息，并按照任务code进行排序
   * @param tstatus 指定的动态任务状态
   * @param applicationName 当前应用服务名 
   * @return
   */
  @Query("from DynamicTaskSchedulerEntity dts where dts.tstatus = :tstatus AND dts.applicationName = :applicationName order by dts.taskCode ")
  public Set<DynamicTaskSchedulerEntity> findByTstatusAndApplicationName(@Param("tstatus") int tstatus , @Param("applicationName") String applicationName);
  /**
   * 按照指定的任务运行状态查询符合要求的动态任务信息，并按照任务code进行排序
   * @param workingStatuses 指定的动态任务工作状态（可以是多个）
   * @param applicationName 当前应用服务名 
   * @return
   */
  @Query("from DynamicTaskSchedulerEntity dts where dts.applicationName = :applicationName AND dts.workingStatus in (:workingStatuses) order by dts.taskCode ")
  public Set<DynamicTaskSchedulerEntity> findByWorkingStatusAndApplicationName(@Param("workingStatuses") int[] workingStatuses, @Param("applicationName") String applicationName);
  /**
   * 查询任务编号对应的动态任务信息，也包括可能的创建者、修改者信息
   * @param taskCode 指定的任务编号
   * @return
   */
  @Query("from DynamicTaskSchedulerEntity dts where dts.taskCode = :taskCode and dts.applicationName = :applicationName ")
  public DynamicTaskSchedulerEntity findByTaskCodeAndApplicationName(@Param("taskCode") String taskCode, @Param("applicationName") String applicationName);  
  /**
   *查询已存在的所有任务，按照创建时间进行倒序排列
   */
  @Query("from DynamicTaskSchedulerEntity dts where dts.projectName = '' OR dts.projectName IS NULL order by dts.createTime desc ")
  public Set<DynamicTaskSchedulerEntity> findByBlankProjectName();
  /**
   * 将指定的一个或者多个动态任务的状态变更为“失效”
   * @param taskCodes
   */
  @Modifying
  @Query(value = "update engine_dynamic_task set working_status = 1 , t_status = 0 where application_name = :applicationName and task_code in (:taskCodes)" , nativeQuery = true)
  public void updateInvalidByTaskCode(@Param("taskCodes") String[] taskCodes , @Param("applicationName") String applicationName);
  /**
   * 将指定的一个或者多个动态任务的状态变更为“运行中”
   * @param taskCodes
   */
  @Modifying
  @Query(value = "update engine_dynamic_task set working_status = 2 , t_status = 1 where  application_name = :applicationName and task_code in (:taskCodes)" , nativeQuery = true)
  public void updateRunningByTaskCode(@Param("taskCodes") String[] taskCodes , @Param("applicationName") String applicationName);

  /**
   * 根据项目名查询数据
   * @param projectName
   * @return
   */
  @Query("from DynamicTaskSchedulerEntity dts where dts.projectName = :projectName order by dts.createTime desc ")
  Set<DynamicTaskSchedulerEntity> findByProjectName(@Param("projectName") String projectName);

  /**
   * 根据任务编码查询数据
   * @param taskcode
   * @return
   */
  DynamicTaskSchedulerEntity findByTaskCode(String taskcode);
}