package com.bizunited.platform.core.repository.dataview.analysis;

import com.bizunited.platform.core.common.enums.SqlOperatorEnum;
import com.bizunited.platform.core.entity.DataViewAuthEntity;
import com.bizunited.platform.core.entity.DataViewAuthHorizontalEntity;
import com.bizunited.platform.core.entity.DataViewAuthHorizontalRelationEntity;
import com.bizunited.platform.core.entity.DataViewFieldEntity;
import com.bizunited.platform.core.service.dataview.model.ExecuteContextModel;
import org.apache.commons.lang3.Validate;
import org.springframework.util.CollectionUtils;

import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;

import static com.bizunited.platform.core.repository.dataview.analysis.Constants.PARAM_SOURCE_TYPE_FIXED;
import static com.bizunited.platform.core.repository.dataview.analysis.Constants.PARAM_SOURCE_TYPE_PRESET;

/**
 * 用于数据视图之横向数据权限条件的分析以及构造SQL工具类
 */
class SQLAuthHorizontalAnalysis {
  /**
   * 通过私有构造函数，让该类暂时不对外开发
   * 因为后期可能会对该类进行重新设计
   */
  private SQLAuthHorizontalAnalysis() {
  }

  /**
   * 构建横向权限的sql条件
   * @param executeContext
   */
  public static void buildSQLAuthHorizoncal(ExecuteContextModel executeContext) {
    DataViewAuthEntity auth = executeContext.getDataViewAuth();
    if(auth == null) {
      return;
    }
    Set<DataViewAuthHorizontalEntity> horizontals = auth.getHorizontalAuths();
    if(CollectionUtils.isEmpty(horizontals)) {
      return;
    }
    Set<DataViewFieldEntity> fields = executeContext.getDataViewFields();
    Map<String, DataViewFieldEntity> fieldsMap = fields.stream().collect(Collectors.toMap(DataViewFieldEntity::getFieldName, f -> f));
    for (DataViewAuthHorizontalEntity horizontal : horizontals) {
      DataViewFieldEntity field = fieldsMap.get(horizontal.getFieldName());
      Validate.notNull("未找到数据视图字段：%s", horizontal.getFieldName());
      Object value = getConditionValue(executeContext, horizontal);
      SqlParamAnalysis.handleParamCondition(executeContext, horizontal.getFieldName(), field.getFieldType(), horizontal.getOprtType(), value);
    }
  }

  /**
   * 获取条件的值
   * @param executeContext
   * @param horizontal
   * @return
   */
  private static Object getConditionValue(ExecuteContextModel executeContext, DataViewAuthHorizontalEntity horizontal) {
    Integer sourceType = horizontal.getParamSourceType();
    switch (sourceType) {
      case PARAM_SOURCE_TYPE_FIXED:
        return getRelationValue(horizontal);
      case PARAM_SOURCE_TYPE_PRESET:
        SQLPresetValueAnalysis sqlPresetValueAnalysis = executeContext.getSqlPresetValueAnalysis();
        return sqlPresetValueAnalysis.getPresetValue(horizontal.getParamKey(), horizontal.getLevel());
      default:
        break;
    }
    throw new IllegalArgumentException(String.format("不能识别参数来源：%s", sourceType));
  }

  /**
   * 获取横向权限的固定值
   * @param horizontal
   * @return
   */
  private static Object getRelationValue(DataViewAuthHorizontalEntity horizontal) {
    SqlOperatorEnum operator = SqlOperatorEnum.valueOf(horizontal.getOprtType());
    Set<DataViewAuthHorizontalRelationEntity> relations = horizontal.getAuthRelations();
    if(operator == SqlOperatorEnum.EQ || operator == SqlOperatorEnum.NEQ) {
      Validate.isTrue(relations != null && relations.size() == 1, "横向权限【%s】的固定值错误，请检查", horizontal.getFieldName());
      DataViewAuthHorizontalRelationEntity relation = relations.iterator().next();
      return relation.getAuthRelationIds();
    } else if(operator == SqlOperatorEnum.IN || operator == SqlOperatorEnum.NIN) {
      Validate.notEmpty(relations, "横向权限【%s】的固定值错误，请检查", horizontal.getFieldName());
      return relations.stream().map(DataViewAuthHorizontalRelationEntity::getAuthRelationIds).collect(Collectors.toList());
    }
    throw new IllegalArgumentException(String.format("不能识别的横向权限操作符：%s", horizontal.getOprtType()));
  }

}
