package com.bizunited.platform.core.repository.dataview.analysis;

import com.bizunited.platform.common.util.ApplicationContextUtils;
import com.bizunited.platform.core.common.enums.AuthOperatorEnum;
import com.bizunited.platform.core.entity.DataViewAuthEntity;
import com.bizunited.platform.core.entity.DataViewAuthVerticalEntity;
import com.bizunited.platform.core.entity.DataViewAuthVerticalRelationEntity;
import com.bizunited.platform.core.entity.DataViewFieldEntity;
import com.bizunited.platform.core.service.dataview.model.ExecuteContextModel;
import com.bizunited.platform.rbac.server.service.RoleService;
import com.bizunited.platform.rbac.server.util.SecurityUtils;
import com.bizunited.platform.rbac.server.vo.RoleVo;
import com.bizunited.platform.user.common.service.organization.OrganizationService;
import com.bizunited.platform.user.common.service.position.PositionService;
import com.bizunited.platform.user.common.service.userGroup.UserGroupService;
import com.bizunited.platform.user.common.vo.OrganizationVo;
import com.bizunited.platform.user.common.vo.PositionVo;
import com.bizunited.platform.user.common.vo.UserGroupVo;
import com.bizunited.platform.user.common.vo.UserVo;
import com.google.common.collect.Sets;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.context.ApplicationContext;
import org.springframework.util.CollectionUtils;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;

/**
 * 用于数据视图之纵向数据权限条件的分析以及构造SQL工具类
 */
public class SQLAuthVerticalAnalysis {
  /**
   * 通过私有构造函数，让该类暂时不对外开发
   * 因为后期可能会对该类进行重新设计
   */
  private SQLAuthVerticalAnalysis() {}

  /**
   * 构建纵向权限
   * @param executeContext
   */
  public static void buildSQLAuthVertical(ExecuteContextModel executeContext) {
    String sqlAlias = executeContext.getWrapperSqlAlias();
    String executeSql = executeContext.getExecuteSql();
    DataViewAuthEntity auth = executeContext.getDataViewAuth();
    Set<DataViewFieldEntity> fields = executeContext.getDataViewFields();
    Validate.notEmpty(fields, "数据视图不存在查询字段：%s", executeContext.getExecuteParam().getDataViewCode());
    Set<String> fieldNames = fields.stream().map(DataViewFieldEntity::getFieldName).collect(Collectors.toSet());
    if(auth != null && !CollectionUtils.isEmpty(auth.getVerticalAuths())) {
      fieldNames = getVerticalField(fields, auth.getVerticalAuths());
    }
    String fieldSql = buildFieldSql(sqlAlias, fieldNames);
    String aliasSql = String.format("select %s from (%s) %s where 1=1 ", fieldSql, executeSql, sqlAlias);
    executeContext.setExecuteSql(aliasSql);
  }

  /**
   * 构建数据视图查询的字段集
   * @param fields
   * @param verticals
   * @return
   */
  public static Set<String> getVerticalField(Set<DataViewFieldEntity> fields, Set<DataViewAuthVerticalEntity> verticals) {
    Validate.notEmpty(fields, "数据视图的字段为空，请检查！！");
    Set<String> fieldNames = fields.stream().map(DataViewFieldEntity::getFieldName).collect(Collectors.toSet());
    if(CollectionUtils.isEmpty(verticals)) {
      return fieldNames;
    }
    Set<String> displayFields = new HashSet<>();
    Map<Integer, Set<String>> cache = new HashMap<>();
    for (DataViewAuthVerticalEntity vertical : verticals) {
      displayFields.addAll(getAuthFields(vertical, cache));
    }
    return Sets.intersection(fieldNames, displayFields);
  }

  /**
   * 获取纵向权限的有权限的字段
   * @param vertical
   * @param cache
   * @return
   */
  private static Set<String> getAuthFields(DataViewAuthVerticalEntity vertical, Map<Integer, Set<String>> cache) {
    Set<String> verticalFields = Arrays.stream(vertical.getDisplayFields().split(",")).collect(Collectors.toSet());
    Set<DataViewAuthVerticalRelationEntity> relations = vertical.getAuthRelations();
    Set<String> presetValues = getPresetValues(vertical.getAuthType(), cache);
    Set<String> relationValues = Sets.newHashSet();
    if(!CollectionUtils.isEmpty(relations)) {
      relationValues = relations.stream().map(DataViewAuthVerticalRelationEntity::getAuthRelationIds).collect(Collectors.toSet());
    }
    Sets.SetView<String> intersection = Sets.intersection(presetValues, relationValues);
    AuthOperatorEnum verticalAuthOperator = AuthOperatorEnum.valueOf(vertical.getOprtType());
    boolean isAuth = false;
    switch (verticalAuthOperator) {
      case EQ:
        isAuth = !CollectionUtils.isEmpty(intersection);
        break;
      case NEQ:
        isAuth = CollectionUtils.isEmpty(intersection);
        break;
      case IN:
        isAuth = !CollectionUtils.isEmpty(intersection);
        break;
      case NIN:
        for (String presetValue : presetValues) {
          if(!relationValues.contains(presetValue)) {
            isAuth = true;
            break;
          }
        }
        break;
      default:
        break;
    }
    if(isAuth) {
      return verticalFields;
    }
    return Sets.newHashSet();
  }

  /**
   * 根据权限类型获取预制值，这里做了缓存，多次相同的权限类型可以从缓存中取
   * @param authType
   * @param cache
   * @return
   */
  private static Set<String> getPresetValues(Integer authType, Map<Integer, Set<String>> cache) {
    if(cache.containsKey(authType)) {
      return cache.get(authType);
    }
    Set<String> presetValues = new HashSet<>();
    ApplicationContext applicationContext = ApplicationContextUtils.getApplicationContext();
    UserVo user = SecurityUtils.getCurrentUser();
    switch (authType) {
      case 1:
        // 用户
        presetValues.add(user.getAccount());
        break;
      case 2:
        // 角色
        RoleService roleService = applicationContext.getBean(RoleService.class);
        List<RoleVo> roles = roleService.findAllByUserId(user.getId(), 0);
        if(!CollectionUtils.isEmpty(roles)) {
          presetValues = roles.stream().map(RoleVo::getRoleCode).collect(Collectors.toSet());
        }
        break;
      case 3:
        // 岗位
        PositionService positionService = applicationContext.getBean(PositionService.class);
        PositionVo position = positionService.findMainPositionByUserId(user.getId());
        if(position != null) {
          presetValues.add(position.getCode());
        }
        break;
      case 4:
        // 用户组
        UserGroupService userGroupService = applicationContext.getBean(UserGroupService.class);
        Set<UserGroupVo> groups = userGroupService.findByUserId(user.getId());
        if(!CollectionUtils.isEmpty(groups)) {
          presetValues = groups.stream().map(UserGroupVo::getGroupCode).collect(Collectors.toSet());
        }
        break;
      case 5:
        // 组织
        OrganizationService organizationService = applicationContext.getBean(OrganizationService.class);
        Set<OrganizationVo> orgs = organizationService.findOrgByUserId(user.getId());
        if(!CollectionUtils.isEmpty(orgs)) {
          presetValues = orgs.stream().map(OrganizationVo::getCode).collect(Collectors.toSet());
        }
        break;
      default:
        break;
    }
    cache.put(authType, presetValues);
    return presetValues;
  }


  /**
   * 构建sql字段
   * @param sqlAlias
   * @param fieldNames
   * @return
   */
  private static String buildFieldSql(String sqlAlias, Set<String> fieldNames) {
    Validate.notEmpty(fieldNames, "沒有对应权限（查询字段为空，请检查sql或纵向权限配置）");
    List<String> sqls = new ArrayList<>();
    for (String fieldName : fieldNames) {
      sqls.add(String.format(" %s.%s as %s", sqlAlias, fieldName, fieldName));
    }
    return StringUtils.join(sqls, ",");
  }

}
