package com.bizunited.platform.core.repository.dataview.analysis;

import com.bizunited.platform.core.entity.DataViewSystemEntity;
import com.bizunited.platform.core.service.dataview.model.ExecuteContextModel;
import com.bizunited.platform.core.service.dataview.model.ExecuteParamModel;
import com.google.common.collect.Sets;
import org.apache.commons.lang3.Validate;
import org.springframework.util.CollectionUtils;

import java.util.HashMap;
import java.util.Map;
import java.util.Set;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import java.util.stream.Collectors;

import static com.bizunited.platform.core.repository.dataview.analysis.Constants.PARAM_SOURCE_TYPE_FIXED;
import static com.bizunited.platform.core.repository.dataview.analysis.Constants.PARAM_SOURCE_TYPE_INPUT;
import static com.bizunited.platform.core.repository.dataview.analysis.Constants.PARAM_SOURCE_TYPE_PRESET;
import static com.bizunited.platform.core.repository.dataview.analysis.Constants.SYSTEM_PARAM_PATTERN;

/**
 * 用于数据视图系统参数部分的参数处理工具类
 * 系统参数只涉及参数值的处理，没有SQL的拼接之类
 * 系统参数传参方式原有：1.外部传入 2.固定值 3.预制值
 * 系统参数书写格式为 {:xxx}
 */
class SQLSystemParamAnalysis {
  /**
   * 通过私有构造函数，让该类暂时不对外开发
   * 因为后期可能会对该类进行重新设计
   */
  private SQLSystemParamAnalysis() {
  }

  /**
   * 构建系统参数的值和分析sql
   * @param executeContext
   */
  static void buildSystemParamAnalysis(ExecuteContextModel executeContext) {
    Set<DataViewSystemEntity> systemFilters = executeContext.getSystemFilters();
    if(CollectionUtils.isEmpty(systemFilters)) {
      systemFilters = Sets.newHashSet();
    }
    Map<String, DataViewSystemEntity> filterMap = systemFilters.stream().collect(Collectors.toMap(DataViewSystemEntity::getParamName, s -> s));
    String executeSql = executeContext.getExecuteSql();
    Pattern p = Pattern.compile(SYSTEM_PARAM_PATTERN);
    Matcher matcher = p.matcher(executeSql);
    Map<String, String> groupMap = new HashMap<>();
    while (matcher.find()) {
      String group = matcher.group();
      String paramName = group.substring(2, group.length() - 1);
      DataViewSystemEntity filter = filterMap.get(paramName);
      Validate.notNull(filter, "未找到参数的配置：%s", paramName);
      Object value = getParamValue(executeContext, filter, paramName);
      String replace = SqlParamAnalysis.handleAndGetParamReplaceStr(executeContext, paramName, filter.getParamType(), value);
      groupMap.put(group, replace);
    }
    for (Map.Entry<String, String> entry : groupMap.entrySet()) {
      executeSql = executeSql.replace(entry.getKey(), entry.getValue());
    }
    executeContext.setExecuteSql(executeSql);
  }

  /**
   * 获取参数的值
   * @param executeContext
   * @param filterMap
   * @param paramName
   * @return
   */
  private static Object getParamValue(ExecuteContextModel executeContext, DataViewSystemEntity filter, String paramName) {
    ExecuteParamModel executeParam = executeContext.getExecuteParam();
    switch (filter.getParamSourceType()) {
      case PARAM_SOURCE_TYPE_INPUT:
        Object value = executeParam.getSystemParam(paramName);
        Validate.notNull(value, "需要传入系统参数：%s", paramName);
        return value;
      case PARAM_SOURCE_TYPE_FIXED:
        return filter.getParamValue();
      case PARAM_SOURCE_TYPE_PRESET:
        SQLPresetValueAnalysis sqlPresetValueAnalysis = executeContext.getSqlPresetValueAnalysis();
        return sqlPresetValueAnalysis.getPresetValue(filter.getParamValue(), null);
      default:
        break;
    }
    throw new IllegalArgumentException(String.format("不能识别参数来源：%s", filter.getParamSourceType()));
  }

}
