package com.bizunited.platform.core.repository.dataview.analysis;

import com.bizunited.platform.common.enums.ClassTypeNameEnum;
import org.apache.commons.lang3.StringUtils;

import java.math.BigDecimal;
import java.sql.Timestamp;
import java.util.Date;

import static com.bizunited.platform.core.repository.dataview.analysis.Constants.ALL_SUPPORT_CLASS;

/**
 * sql参数的格式化，对object类型的参数根据配置的数据类型来进行转换，包括时间类型的转换
 * @Author: Paul Chan
 * @Date: 2020-04-18 15:47
 */
public class SqlParamFormater {

  private SqlParamFormater() {
    throw new IllegalStateException("静态类不能实例化");
  }

  /**
   * 对object类型的参数根据配置的数据类型来进行转换，包括时间类型的转换
   * @param paramType
   * @param value
   * @return
   */
  public static Object format(String paramType, Object value) {
    if(value == null) {
      return value;
    }
    if(StringUtils.isBlank(paramType)) {
      return defaulFormat(value);
    }
    ClassTypeNameEnum classType = ClassTypeNameEnum.valueOfClassName(paramType);
    if(classType == null) {
      return defaulFormat(value);
    }
    try {
      switch (classType) {
        case STRING:
          return value.toString();
        case INT:
        case INTEGER:
          if (!(value instanceof Integer)) {
            return Integer.valueOf(value.toString());
          }
          break;
        case DOUBLE:
        case DOUBLE_PACK:
          if (!(value instanceof Double)) {
            return Double.valueOf(value.toString());
          }
          break;
        case FLOAT:
        case FLOAT_PACK:
          if (!(value instanceof Float)) {
            return Float.valueOf(value.toString());
          }
          break;
        case LONG:
        case LONG_PACK:
          if (!(value instanceof Long)) {
            return Long.valueOf(value.toString());
          }
          break;
        case SHORT:
        case SHORT_PACK:
          if (!(value instanceof Short)) {
            return Short.valueOf(value.toString());
          }
          break;
        case BYTE:
        case BYTE_PACK:
          if (!(value instanceof Byte)) {
            return Byte.valueOf(value.toString());
          }
          break;
        case BOOLEAN:
        case BOOLEAN_PACK:
          if (!(value instanceof Boolean)) {
            return Boolean.valueOf(value.toString());
          }
          break;
        case CHAR:
        case CHAR_PACK:
          if (!(value instanceof Character)) {
            return value.toString();
          }
          break;
        case DATE:
          if (!(Date.class.isAssignableFrom(value.getClass()))) {
            return formatDate(value);
          }
          break;
        case TIMESTAMP:
          if (!(value instanceof Timestamp)) {
            return formatDate(value);
          }
          break;
        case BIG_DECIMAL:
          if (!(BigDecimal.class.isAssignableFrom(value.getClass()))) {
            return new BigDecimal(value.toString());
          }
          break;
        default:
          break;
      }
      return value;
    }catch (NumberFormatException | ClassCastException e){
      throw new NumberFormatException(String.format("传入参数错误：【%s】", value));
    }
  }


  /**
   * 默认的sql参数格式化
   * @param value
   * @return
   */
  public static Object defaulFormat(Object value) {
    if(value == null) {
      return value;
    }
    String className = value.getClass().getName();
    if(!ALL_SUPPORT_CLASS.contains(className)) {
      return value.toString();
    }
    if(ClassTypeNameEnum.DATE.getClassName().equalsIgnoreCase(className)) {
      return formatDate(value);
    }
    return value;
  }

  /**
   * 格式化时间
   * @param value
   * @param addDay
   * @return
   */
  public static Object formatDate(Object value) {
    if(value == null) {
      return value;
    }
    if(Date.class.isAssignableFrom(value.getClass())) {
      return value;
    }
    if(value instanceof Long) {
      return new Date((Long) value);
    }
    return value.toString();
  }

}
