package com.bizunited.platform.core.service.dataview.executor;

import com.alibaba.fastjson.JSONObject;
import com.bizunited.platform.common.enums.ClassTypeNameEnum;
import com.bizunited.platform.common.util.JsonUtils;
import com.bizunited.platform.core.common.enums.DataSourceTypeEnum;
import com.bizunited.platform.core.entity.DataSourceEntity;
import com.bizunited.platform.core.entity.DataViewAuthEntity;
import com.bizunited.platform.core.entity.DataViewEntity;
import com.bizunited.platform.core.repository.dataview.analysis.Constants;
import com.bizunited.platform.core.repository.dataview.analysis.MysqlAnalysis;
import com.bizunited.platform.core.repository.dataview.analysis.OracleSqlAnalysis;
import com.bizunited.platform.core.repository.dataview.analysis.SQLPresetValueAnalysis;
import com.bizunited.platform.core.repository.dataview.analysis.SqlAnalysis;
import com.bizunited.platform.core.repository.dynamic.DynamicDataSourceManager;
import com.bizunited.platform.core.service.dataview.DataViewAuthService;
import com.bizunited.platform.core.service.dataview.DataViewService;
import com.bizunited.platform.core.service.dataview.model.ExecuteContextModel;
import com.bizunited.platform.core.service.dataview.model.ExecuteParamModel;
import com.bizunited.platform.core.service.dataview.model.PageableModel;
import com.fasterxml.jackson.core.JsonProcessingException;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.hibernate.Session;
import org.hibernate.SessionFactory;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.boot.autoconfigure.jdbc.DataSourceProperties;
import org.springframework.context.ApplicationContext;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.PageImpl;
import org.springframework.data.domain.PageRequest;
import org.springframework.orm.hibernate5.SessionFactoryUtils;
import org.springframework.util.CollectionUtils;

import javax.persistence.EntityManager;
import javax.sql.DataSource;
import java.sql.Connection;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.ResultSetMetaData;
import java.sql.SQLException;
import java.sql.Timestamp;
import java.util.ArrayList;
import java.util.Date;
import java.util.Iterator;
import java.util.List;

/**
 * 数据视图执行器
 * @Author: Paul Chan
 * @Date: 2020-04-16 21:08
 */
public class DataViewExecutor {

  private static final Logger LOGGER = LoggerFactory.getLogger(DataViewExecutor.class);

  /**
   * 数据视图执行的上下文
   */
  private ExecuteContextModel executeContext;

  /**
   * sql生成器
   */
  private SqlAnalysis sqlAnalysis;

  public DataViewExecutor(ApplicationContext applicationContext, ExecuteParamModel executeParam) {
    this.init(applicationContext, executeParam);
  }

  /**
   * 初始化信息，为执行做准备
   * @param applicationContext
   * @param executeParam
   */
  private void init(ApplicationContext applicationContext, ExecuteParamModel executeParam) {
    Validate.notNull(executeParam, "执行参数不能为空");
    Validate.notNull(applicationContext, "spring上下文不能为空");
    Validate.notBlank(executeParam.getDataViewCode(), "数据视图编码不能为空");
    DataViewService dataViewService = applicationContext.getBean(DataViewService.class);
    DataViewEntity dataView = dataViewService.findByCode(executeParam.getDataViewCode());
    Validate.notNull(dataView, "未找到数据视图：%s", executeParam.getDataViewCode());
    DataSourceEntity dataSource = dataView.getDataSource();
    DataSourceProperties dataSourceProperties = applicationContext.getBean(DataSourceProperties.class);
    DataSourceTypeEnum dataSourceType = this.getDataSourceType(dataSource, dataSourceProperties);
    SQLPresetValueAnalysis sqlPresetValueAnalysis = new SQLPresetValueAnalysis(applicationContext);
    DataViewAuthEntity auth = null;
    if(StringUtils.isNotBlank(executeParam.getAuthCode())) {
      DataViewAuthService dataViewAuthService = applicationContext.getBean(DataViewAuthService.class);
      auth = dataViewAuthService.findDetailsByDataViewCodeAndCode(dataView.getCode(), executeParam.getAuthCode());
      Validate.notNull(auth, "未找到指定的数据权限:%s", executeParam.getAuthCode());
    }
    executeContext = new ExecuteContextModel();
    executeContext.setApplicationContext(applicationContext);
    executeContext.setExecuteParam(executeParam);
    executeContext.setDataView(dataView);
    executeContext.setDataSourceType(dataSourceType);
    executeContext.setDataSource(dataSource);
    executeContext.setDataViewFields(dataView.getFields());
    executeContext.setSqlPresetValueAnalysis(sqlPresetValueAnalysis);
    executeContext.setWrapperSqlAlias(Constants.ALIAS_STRING);
    executeContext.setDataViewFilters(dataView.getFilters());
    executeContext.setSystemFilters(dataView.getSystemFilters());
    executeContext.setDataViewAuth(auth);

    if(dataSourceType.equals(DataSourceTypeEnum.MYSQL)) {
      sqlAnalysis = new MysqlAnalysis(executeContext);
    } else if(dataSourceType.equals(DataSourceTypeEnum.ORACLE)) {
      sqlAnalysis = new OracleSqlAnalysis(executeContext);
    }
    executeContext.setSqlAnalysis(sqlAnalysis);
    sqlAnalysis.analysis();
  }

  /**
   * 获取数据库类型
   * @param dataSource
   * @param dataSourceProperties
   * @return
   */
  private DataSourceTypeEnum getDataSourceType(DataSourceEntity dataSource, DataSourceProperties dataSourceProperties) {
    if(dataSource == null) {
      DataSourceTypeEnum dataSourceTypeEnum = DataSourceTypeEnum.valueOfDriver(dataSourceProperties.getDriverClassName());
      Validate.notNull(dataSourceTypeEnum, String.format("不支持的数据库驱动类：%s", dataSourceProperties.getDriverClassName()));
      return dataSourceTypeEnum;
    }
    DataSourceTypeEnum dataSourceTypeEnum = DataSourceTypeEnum.valueOfDatabase(dataSource.getType());
    Validate.notNull(dataSourceTypeEnum, "不支持的数据库类型：%s", dataSource.getType());
    return dataSourceTypeEnum;
  }

  /**
   * 执行数据视图并且返回sql查询到的数据，此处查询返回的List对象
   * @return
   */
  public List<JSONObject> executeQuery() {
    SessionFactory sessionFactory = this.getSessionFactory();
    DataSource dataSource = SessionFactoryUtils.getDataSource(sessionFactory);
    try(Connection conn = dataSource.getConnection()) {
      return this.executeQuery(conn, executeContext.getExecuteSql(), executeContext.getExecuteSqlParameters());
    } catch (SQLException e) {
      LOGGER.error(e.getMessage(), e);
      throw new IllegalArgumentException(String.format("执行sql出错：%s", e.getMessage()));
    }
  }

  /**
   * 执行数据视图并且返回sql查询到的数据，分页查询返回的page对象
   * @return
   */
  public Page<JSONObject> executePageQuery() {
    SessionFactory sessionFactory = this.getSessionFactory();
    DataSource dataSource = SessionFactoryUtils.getDataSource(sessionFactory);
    PageableModel pageable = executeContext.getExecuteParam().getPageable();
    Validate.notNull(pageable, "未找到分页信息，请检查参数");
    try(Connection conn = dataSource.getConnection()) {
      List<JSONObject> resultList = this.executeQuery(conn, executeContext.getExecuteSql(), executeContext.getExecuteSqlParameters());
      List<JSONObject> countResult = this.executeQuery(conn, executeContext.getExecuteCountSql(), executeContext.getExecuteCountSqlParameters());
      long count = this.getCount(countResult);
      return new PageImpl<>(resultList, PageRequest.of(pageable.getPage(), pageable.getSize()), count);
    } catch (SQLException e) {
      LOGGER.error(e.getMessage(), e);
      throw new IllegalArgumentException(String.format("执行sql出错：%s", e.getMessage()));
    }
  }

  /**
   * 根据sql查询的结果获取统计数
   * @param countResultSet
   * @return
   */
  private long getCount(List<JSONObject> countResult) {
    if(CollectionUtils.isEmpty(countResult)) {
      return 0;
    }
    JSONObject jsonObject = countResult.get(0);
    Iterator<String> iterator = jsonObject.keySet().iterator();
    return iterator.hasNext() ? jsonObject.getLongValue(iterator.next()) : 0;
  }

  /**
   * 执行sql查询
   * @param conn
   * @param sql
   * @return
   * @throws SQLException
   */
  private List<JSONObject> executeQuery(Connection conn, String sql, List<Object> parameters) throws SQLException {
    LOGGER.info("executeSql:{}", sql);
    try {
      LOGGER.info("parameters:{}", JsonUtils.obj2JsonString(parameters));
    } catch (JsonProcessingException e) {
      LOGGER.warn(e.getMessage(), e);
    }
    try (PreparedStatement preparedStatement = conn.prepareStatement(sql)) {
      this.setStatementParameters(preparedStatement, parameters);
      ResultSet resultSet = preparedStatement.executeQuery();
      return this.getResultList(resultSet);
    }
  }

  /**
   * 根据sql查询的结果转换未JSONObject对象的list
   * @param resultSet
   * @return
   * @throws SQLException
   */
  private List<JSONObject> getResultList(ResultSet resultSet) throws SQLException {
    List<JSONObject> list = new ArrayList<>();
    while (resultSet.next()) {
      ResultSetMetaData metaData = resultSet.getMetaData();
      int columnCount = metaData.getColumnCount();
      JSONObject jsonObject = new JSONObject();
      for (int i = 1; i <= columnCount; i++) {
        jsonObject.put(metaData.getColumnLabel(i), resultSet.getObject(i));
      }
      list.add(jsonObject);
    }
    return list;
  }

  /**
   * 获取hibernate的会话工厂类
   * @return
   */
  private SessionFactory getSessionFactory() {
    ApplicationContext applicationContext = executeContext.getApplicationContext();
    DataSourceEntity dataSource = executeContext.getDataSource();
    if(dataSource == null) {
      EntityManager entityManager = applicationContext.getBean(EntityManager.class);
      Session session = (Session) entityManager.getDelegate();
      return session.getSessionFactory();
    }
    DynamicDataSourceManager dataSourceManager = applicationContext.getBean(DynamicDataSourceManager.class);
    SessionFactory sessionFactory = dataSourceManager.getCurrentSessionFactory(dataSource.getCode());
    if(sessionFactory == null){
      throw new NullPointerException(String.format("未找到数据源：%s", dataSource.getCode()));
    }
    return sessionFactory;
  }

  /**
   * 设置查询参数
   * @param preparedStatement
   * @param parameters
   * @throws SQLException
   */
  private void setStatementParameters(PreparedStatement preparedStatement, List<Object> parameters) throws SQLException {
    if(CollectionUtils.isEmpty(parameters)) {
      return;
    }
    for (int i = 0; i < parameters.size(); i++) {
      int index = i + 1;
      Object value = parameters.get(i);
      if(value == null) {
        preparedStatement.setObject(index, value);
        continue;
      }
      ClassTypeNameEnum classTypeName = ClassTypeNameEnum.valueOfClassName(value.getClass().getName());
      if(classTypeName == null) {
        preparedStatement.setObject(index, value);
        continue;
      }
      switch (classTypeName) {
        case STRING:
          preparedStatement.setString(index, value.toString());
          break;
        case INT:
        case INTEGER:
          preparedStatement.setInt(index, (Integer) value);
          break;
        case DOUBLE:
        case DOUBLE_PACK:
          preparedStatement.setDouble(index, (Double) value);
          break;
        case FLOAT:
        case FLOAT_PACK:
          preparedStatement.setFloat(index, (Float) value);
          break;
        case LONG:
        case LONG_PACK:
          preparedStatement.setLong(index, (Long) value);
          break;
        case SHORT:
        case SHORT_PACK:
          preparedStatement.setShort(index, (Short) value);
          break;
        case BYTE:
        case BYTE_PACK:
          preparedStatement.setByte(index, (Byte) value);
          break;
        case BOOLEAN:
        case BOOLEAN_PACK:
          preparedStatement.setBoolean(index, (Boolean) value);
          break;
        case CHAR:
        case CHAR_PACK:
          preparedStatement.setString(index, value.toString());
          break;
        case DATE:
          Date date = (Date) value;
          Timestamp timestamp = new Timestamp(date.getTime());
          preparedStatement.setTimestamp(index, timestamp);
          break;
        case BIG_DECIMAL:
          preparedStatement.setString(index, value.toString());
          break;
        default:
          preparedStatement.setObject(index, value);
          break;
      }
    }
  }

}
