package com.bizunited.platform.core.service.dataview.executor;

import com.bizunited.platform.common.util.ApplicationContextUtils;
import com.bizunited.platform.core.common.enums.ClassTypeEnum;
import com.bizunited.platform.core.entity.DataViewAuthInterceptorEntity;
import com.bizunited.platform.core.interceptor.DataViewAuthInterceptor;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.context.ApplicationContext;
import org.springframework.util.CollectionUtils;

import java.util.Set;

/**
 * 拦截器执行器
 * @Author: Paul Chan
 * @Date: 2020-04-24 13:59
 */
public class InterceptorExecutor {

  private static final Logger LOGGER = LoggerFactory.getLogger(InterceptorExecutor.class);

  protected static final String ERROR_INTERCEPTOR_IMPLEMENTS = "权限拦截器类必须实现接口：com.bizunited.platform.core.interceptor.DataViewAuthInterceptor";

  /**
   * 执行的上下文，如数据视图在执行时的上下文ExecuteContextModel
   */
  private Object context;

  /**
   * 拦截器对象
   */
  private Set<DataViewAuthInterceptorEntity> authInterceptors;

  public InterceptorExecutor(Object context, Set<DataViewAuthInterceptorEntity> authInterceptors) {
    this.context = context;
    this.authInterceptors = authInterceptors;
  }

  /**
   * 执行拦截器
   */
  public void execute() {
    if(CollectionUtils.isEmpty(authInterceptors)) {
      return;
    }
    for (DataViewAuthInterceptorEntity authInterceptor : authInterceptors) {
      DataViewAuthInterceptor interceptor = this.getInterceptor(authInterceptor);
      if(interceptor != null) {
        interceptor.intercept(context);
      }
    }
  }

  /**
   * 获取拦截器
   * @return
   */
  protected DataViewAuthInterceptor getInterceptor(DataViewAuthInterceptorEntity interceptor) {
    if(interceptor == null) {
      return null;
    }
    String interceptorClass = interceptor.getInterceptor();
    if(StringUtils.isBlank(interceptorClass)) {
      return null;
    }
    Integer classType = interceptor.getClassType();
    if(ClassTypeEnum.CLASS_PATH.getType().equals(classType)) {
      // 根据类路径获取拦截器
      try {
        Class<?> clazz = Class.forName(interceptorClass);
        Validate.isTrue(DataViewAuthInterceptor.class.isAssignableFrom(clazz), ERROR_INTERCEPTOR_IMPLEMENTS);
        return (DataViewAuthInterceptor) clazz.newInstance();
      } catch (ClassNotFoundException e) {
        LOGGER.error(e.getMessage(), e);
        throw new IllegalArgumentException(String.format("未找到java类：%s", interceptorClass));
      } catch (IllegalAccessException|InstantiationException e) {
        LOGGER.error(e.getMessage(), e);
        throw new IllegalArgumentException(String.format("实例化类错误：%s", interceptorClass));
      }
    } else if (ClassTypeEnum.BEAN_NAME.getType().equals(classType)) {
      // 从ioc容器获取拦截器
      ApplicationContext applicationContext = ApplicationContextUtils.getApplicationContext();
      Validate.isTrue(applicationContext.containsBean(interceptorClass), "ioc容器中未发现bean：%s", interceptorClass);
      Object bean = applicationContext.getBean(interceptorClass);
      Validate.isTrue(DataViewAuthInterceptor.class.isAssignableFrom(bean.getClass()), ERROR_INTERCEPTOR_IMPLEMENTS);
      return (DataViewAuthInterceptor) bean;
    } else {
      throw new IllegalArgumentException(String.format("不支持的拦截器类型:%s", classType));
    }
  }

}
