package com.bizunited.platform.core.service.dataview.internal;

import com.bizunited.platform.common.service.NebulaToolkitService;
import com.bizunited.platform.core.common.enums.AuthOperatorEnum;
import com.bizunited.platform.core.entity.DataViewAuthEntity;
import com.bizunited.platform.core.entity.DataViewAuthHorizontalEntity;
import com.bizunited.platform.core.entity.DataViewAuthHorizontalRelationEntity;
import com.bizunited.platform.core.repository.dataview.DataViewAuthHorizontalRepository;
import com.bizunited.platform.core.service.dataview.DataViewAuthHorizontalRelationService;
import com.bizunited.platform.core.service.dataview.DataViewAuthHorizontalService;
import com.bizunited.platform.core.service.dataview.DataViewAuthService;
import com.google.common.collect.Sets;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;

import javax.transaction.Transactional;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;

import static com.bizunited.platform.core.repository.dataview.analysis.Constants.PARAM_SOURCE_TYPE_FIXED;
import static com.bizunited.platform.core.repository.dataview.analysis.Constants.PARAM_SOURCE_TYPE_PRESET;

/**
 * DataViewAuthHorizontalEntity业务模型的服务层接口实现
 *
 * @author saturn
 */
@Service("DataViewAuthHorizontalServiceImpl")
public class DataViewAuthHorizontalServiceImpl implements DataViewAuthHorizontalService {
  @Autowired
  private DataViewAuthHorizontalRepository dataViewAuthHorizontalRepository;
  @Autowired
  private DataViewAuthService dataViewAuthService;
  @Autowired
  private DataViewAuthHorizontalRelationService dataViewAuthHorizontalRelationService;
  /**
   * private DataViewAuthHorizontalRelationRepository dataViewAuthHorizontalRelationRepository;
   * <p>
   * /**
   * Kuiper表单引擎用于减少编码工作量的工具包
   */
  @Autowired
  private NebulaToolkitService nebulaToolkitService;


  /**
   * 批量更新横向数据权限
   *
   * @param authHorizontals
   * @return
   */
  @Override
  @Transactional
  public Set<DataViewAuthHorizontalEntity> save(String authId, Set<DataViewAuthHorizontalEntity> authHorizontals) {
    this.saveValidation(authId, authHorizontals);
    DataViewAuthEntity auth = dataViewAuthService.findById(authId);
    Validate.notNull(auth, "未找到数据权限：%s", authId);
    if (authHorizontals == null) {
      authHorizontals = Sets.newHashSet();
    }
    Map<String, DataViewAuthHorizontalEntity> authHorizontalsMap = authHorizontals.stream().collect(Collectors.toMap(DataViewAuthHorizontalEntity::getId, ah -> ah, (ah1, ah2) -> ah1));
    Set<DataViewAuthHorizontalEntity> dbAuthHorizontals = dataViewAuthHorizontalRepository.findByAuth(auth);
    Set<DataViewAuthHorizontalEntity> deletes = Sets.newHashSet();
    Set<DataViewAuthHorizontalEntity> updates = Sets.newHashSet();
    Set<DataViewAuthHorizontalEntity> creates = Sets.newHashSet();
    nebulaToolkitService.collectionDiscrepancy(authHorizontals, dbAuthHorizontals, DataViewAuthHorizontalEntity::getId, deletes, updates, creates);
    for (DataViewAuthHorizontalEntity delete : deletes) {
      dataViewAuthHorizontalRelationService.deleteByAuthHorizontal(delete.getId());
      dataViewAuthHorizontalRepository.delete(delete);
    }
    for (DataViewAuthHorizontalEntity update : updates) {
      DataViewAuthHorizontalEntity authHorizontal = authHorizontalsMap.get(update.getId());
      update.setAuthType(authHorizontal.getAuthType());
      update.setOprtType(authHorizontal.getOprtType());
      update.setParamSourceType(authHorizontal.getParamSourceType());
      update.setParamKey(authHorizontal.getParamKey());
      update.setFieldName(authHorizontal.getFieldName());
      update.setSortIndex(authHorizontal.getSortIndex());
      update.setLevel(authHorizontal.getLevel());
      dataViewAuthHorizontalRepository.save(update);
      if (PARAM_SOURCE_TYPE_FIXED == update.getParamSourceType()) {
        dataViewAuthHorizontalRelationService.save(update, authHorizontal.getAuthRelations());
      }
    }
    for (DataViewAuthHorizontalEntity create : creates) {
      create.setDataViewAuth(auth);
      create.setId(null);
      dataViewAuthHorizontalRepository.save(create);
      if (PARAM_SOURCE_TYPE_FIXED == create.getParamSourceType()) {
        dataViewAuthHorizontalRelationService.save(create, create.getAuthRelations());
      }
    }
    updates.addAll(creates);
    return updates;
  }

  /**
   * 在更新一个已有的DataViewAuthHorizontalEntity模型对象之前，该私有方法检查对象各属性的正确性，其id属性必须有值
   */
  private void saveValidation(String authId, Set<DataViewAuthHorizontalEntity> authHorizontals) {
    Validate.notBlank(authId, "数据权限对象ID不能为空");
    if (CollectionUtils.isEmpty(authHorizontals)) {
      return;
    }
    for (DataViewAuthHorizontalEntity authHorizontal : authHorizontals) {
      Validate.notNull(authHorizontal, "更新时，横向权限信息必须传入！！");
      Validate.notNull(authHorizontal.getSortIndex(), "排序字段不能为空");
      Validate.notNull(authHorizontal.getParamSourceType(), "参数来源类型不能为空！");
      Validate.notNull(authHorizontal.getAuthType(), "权限类型不能为空！");
      Validate.notBlank(authHorizontal.getOprtType(), "操作符类型不能为空！");
      AuthOperatorEnum authOperator = AuthOperatorEnum.valueOf(authHorizontal.getOprtType());
      Validate.notNull(authHorizontal, "不支持的操作符：%s", authHorizontal.getOprtType());
      if(authHorizontal.getParamSourceType() == PARAM_SOURCE_TYPE_FIXED) {
        Validate.notEmpty(authHorizontal.getAuthRelations(), "字段权限【%s】的固定参数没有值，请检查！！", authHorizontal.getFieldName());
        if(authOperator == AuthOperatorEnum.EQ || authOperator == AuthOperatorEnum.NEQ) {
          Validate.isTrue(authHorizontal.getAuthRelations().size() == 1, "字段权限【%s】的固定参数值只能有一个，请检查！！");
        }
      } else if(authHorizontal.getParamSourceType() == PARAM_SOURCE_TYPE_PRESET) {
        Validate.notBlank(authHorizontal.getParamKey(), "预制值的值不能为空");
      }

      // 验证长度，被验证的这些字段符合特征: 字段类型为String，且不为PK，且canupdate = true
      Validate.isTrue(authHorizontal.getOprtType() == null || authHorizontal.getOprtType().length() < 32, "操作符类型,填入值超过了限定长度(32)，请检查!");
      Validate.isTrue(authHorizontal.getParamKey() == null || authHorizontal.getParamKey().length() < 64, "预制值标识,填入值超过了限定长度(64)，请检查!");
    }
  }

  @Override
  public DataViewAuthHorizontalEntity findDetailsById(String id) {
    if (StringUtils.isBlank(id)) {
      return null;
    }
    return this.dataViewAuthHorizontalRepository.findDetailsById(id);
  }

  @Override
  public DataViewAuthHorizontalEntity findById(String id) {
    if (StringUtils.isBlank(id)) {
      return null;
    }
    return dataViewAuthHorizontalRepository.findById(id).orElse(null);
  }

  @Override
  @Transactional
  public void deleteById(String id) {
    // 只有存在才进行删除
    Validate.notBlank(id, "进行删除时，必须给定主键信息!!");
    DataViewAuthHorizontalEntity authHorizontal = dataViewAuthHorizontalRepository.findById(id).orElse(null);
    if (authHorizontal == null) {
      return;
    }
    dataViewAuthHorizontalRelationService.deleteByAuthHorizontal(id);
    dataViewAuthHorizontalRepository.delete(authHorizontal);
  }

  @Override
  public Set<DataViewAuthHorizontalEntity> findDetailsByAuthId(String authId) {
    if(StringUtils.isBlank(authId)) {
      return Sets.newHashSet();
    }
    DataViewAuthEntity auth = new DataViewAuthEntity();
    auth.setId(authId);
    Set<DataViewAuthHorizontalEntity> horizontals = dataViewAuthHorizontalRepository.findByAuth(auth);
    if(CollectionUtils.isEmpty(horizontals)) {
      return Sets.newHashSet();
    }
    for (DataViewAuthHorizontalEntity horizontal : horizontals) {
      Set<DataViewAuthHorizontalRelationEntity> relations = dataViewAuthHorizontalRelationService.findDetailsByAuthHorizontal(horizontal.getId());
      horizontal.setAuthRelations(relations);
    }
    return horizontals;
  }

  @Override
  @Transactional
  public void deleteByField(String fieldName) {
    if(StringUtils.isBlank(fieldName)) {
      return;
    }
    DataViewAuthHorizontalEntity horizontal = dataViewAuthHorizontalRepository.findByFieldName(fieldName);
    if(horizontal == null) {
      return;
    }
    dataViewAuthHorizontalRelationService.deleteByAuthHorizontal(horizontal.getId());
    dataViewAuthHorizontalRepository.delete(horizontal);
  }

  @Override
  @Transactional
  public void deleteByAuthId(String authId) {
    Validate.notBlank(authId, "权限ID不能为空");
    DataViewAuthEntity auth = new DataViewAuthEntity();
    auth.setId(authId);
    Set<DataViewAuthHorizontalEntity> horizontals = dataViewAuthHorizontalRepository.findByAuth(auth);
    if(CollectionUtils.isEmpty(horizontals)) {
      return;
    }
    for (DataViewAuthHorizontalEntity horizontal : horizontals) {
      dataViewAuthHorizontalRelationService.deleteByAuthHorizontal(horizontal.getId());
      dataViewAuthHorizontalRepository.delete(horizontal);
    }
  }

} 