package com.bizunited.platform.core.service.dataview.internal;

import com.bizunited.platform.common.service.NebulaToolkitService;
import com.bizunited.platform.core.entity.DataViewAuthEntity;
import com.bizunited.platform.core.entity.DataViewAuthInterceptorEntity;
import com.bizunited.platform.core.interceptor.DataViewAuthInterceptor;
import com.bizunited.platform.core.repository.dataview.DataViewAuthInterceptorRepository;
import com.bizunited.platform.core.service.dataview.DataViewAuthInterceptorService;
import com.bizunited.platform.core.service.dataview.DataViewAuthService;
import com.google.common.collect.Lists;
import com.google.common.collect.Sets;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.ApplicationContext;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;

import javax.transaction.Transactional;
import java.util.Arrays;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;

/**
 * 数据视图权限拦截器服务实现
 *
 * @Author: Paul Chan
 * @Date: 2019-12-23 10:24
 */
@Service("DataViewAuthInterceptorServiceImpl")
public class DataViewAuthInterceptorServiceImpl implements DataViewAuthInterceptorService {

  @Autowired
  private DataViewAuthInterceptorRepository dataViewAuthInterceptorRepository;
  @Autowired
  private DataViewAuthService dataViewAuthService;
  @Autowired
  private NebulaToolkitService nebulaToolkitService;
  @Autowired
  private ApplicationContext applicationContext;


  @Override
  public Set<DataViewAuthInterceptorEntity> save(String authId, Set<DataViewAuthInterceptorEntity> interceptors) {
    this.saveValidation(authId, interceptors);
    DataViewAuthEntity auth = dataViewAuthService.findById(authId);
    Validate.notNull(auth, "未找到数据权限：%s", authId);
    if(interceptors == null) {
      interceptors = Sets.newHashSet();
    }
    Map<String, DataViewAuthInterceptorEntity> interceptorsMap = interceptors.stream().collect(Collectors.toMap(DataViewAuthInterceptorEntity::getId, i -> i, (i1, i2) -> i1));
    Set<DataViewAuthInterceptorEntity> dbInterceptors = dataViewAuthInterceptorRepository.findByAuth(auth);
    Set<DataViewAuthInterceptorEntity> deletes = Sets.newHashSet();
    Set<DataViewAuthInterceptorEntity> updates = Sets.newHashSet();
    Set<DataViewAuthInterceptorEntity> creates = Sets.newHashSet();
    nebulaToolkitService.collectionDiscrepancy(interceptors, dbInterceptors, DataViewAuthInterceptorEntity::getId, deletes, updates, creates);
    for (DataViewAuthInterceptorEntity delete : deletes) {
      dataViewAuthInterceptorRepository.delete(delete);
    }
    for (DataViewAuthInterceptorEntity update : updates) {
      DataViewAuthInterceptorEntity interceptor = interceptorsMap.get(update.getId());
      update.setClassType(interceptor.getClassType());
      update.setInterceptor(interceptor.getInterceptor());
      update.setSort(interceptor.getSort());
      dataViewAuthInterceptorRepository.save(update);
    }
    for (DataViewAuthInterceptorEntity create : creates) {
      create.setId(null);
      create.setDataViewAuth(auth);
      dataViewAuthInterceptorRepository.save(create);
    }
    updates.addAll(creates);
    return updates;
  }

  @Override
  public Set<DataViewAuthInterceptorEntity> findByAuthId(String authId) {
    if(StringUtils.isBlank(authId)) {
      return Sets.newHashSet();
    }
    DataViewAuthEntity auth = new DataViewAuthEntity();
    auth.setId(authId);
    return dataViewAuthInterceptorRepository.findByAuth(auth);
  }

  @Override
  @Transactional
  public void deleteByAuthId(String authId) {
    Validate.notBlank(authId, "权限ID不能为空");
    Set<DataViewAuthInterceptorEntity> interceptors = this.findByAuthId(authId);
    if(CollectionUtils.isEmpty(interceptors)) {
      return;
    }
    dataViewAuthInterceptorRepository.deleteAll(interceptors);
  }

  /**
   * 保存前数据验证
   * @param authId
   * @param interceptors
   */
  private void saveValidation(String authId, Set<DataViewAuthInterceptorEntity> interceptors) {
    Validate.notBlank(authId, "权限对象的ID不能为空");
    if(CollectionUtils.isEmpty(interceptors)) {
      return;
    }
    Set<String> interceptorSet = Sets.newHashSet();
    for (DataViewAuthInterceptorEntity interceptor : interceptors) {
      Validate.notNull(interceptor, "保存对象不能为空！！");
      Validate.notBlank(interceptor.getInterceptor(), "拦截器不能为空！！");
      Validate.notNull(interceptor.getClassType(), "拦截器类类型不能为空");
      Validate.notNull(interceptor.getSort(), "排序字段不能为空");
      Validate.isTrue(interceptorSet.add(interceptor.getInterceptor()), "重复的拦截器：%s", interceptor.getInterceptor());
    }
  }

  /**
   * 查询数据视图权限拦截器
   * @return
   */
  @Override
  public List<String> findAll() {
    String[] result = applicationContext.getBeanNamesForType(DataViewAuthInterceptor.class);
    if(result.length <= 0){
      return Lists.newArrayList();
    }
    return Arrays.asList(result);
  }

}
