package com.bizunited.platform.core.service.dataview.internal;

import com.bizunited.platform.common.service.NebulaToolkitService;
import com.bizunited.platform.core.common.enums.AuthOperatorEnum;
import com.bizunited.platform.core.entity.DataViewAuthEntity;
import com.bizunited.platform.core.entity.DataViewAuthVerticalEntity;
import com.bizunited.platform.core.entity.DataViewAuthVerticalRelationEntity;
import com.bizunited.platform.core.repository.dataview.DataViewAuthVerticalRepository;
import com.bizunited.platform.core.service.dataview.DataViewAuthService;
import com.bizunited.platform.core.service.dataview.DataViewAuthVerticalRelationService;
import com.bizunited.platform.core.service.dataview.DataViewAuthVerticalService;
import com.google.common.collect.Sets;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;

import javax.transaction.Transactional;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.stream.Collectors;

/**
 * 数据视图纵向权限，DataViewAuthVerticalEntity业务模型的服务层接口实现
 *
 * @author saturn
 */
@Service("DataViewAuthVerticalServiceImpl")
public class DataViewAuthVerticalServiceImpl implements DataViewAuthVerticalService {
  @Autowired
  private DataViewAuthVerticalRepository dataViewAuthVerticalRepository;
  @Autowired
  private DataViewAuthService dataViewAuthService;
  @Autowired
  private DataViewAuthVerticalRelationService dataViewAuthVerticalRelationService;

  /**
   * Kuiper表单引擎用于减少编码工作量的工具包
   */
  @Autowired
  private NebulaToolkitService nebulaToolkitService;

  @Override
  @Transactional
  public Set<DataViewAuthVerticalEntity> save(String authId, Set<DataViewAuthVerticalEntity> authVerticals) {
    this.saveValidation(authId, authVerticals);
    DataViewAuthEntity auth = dataViewAuthService.findById(authId);
    Validate.notNull(auth, "未找到数据权限：%s", authId);
    if (authVerticals == null) {
      authVerticals = Sets.newHashSet();
    }
    Map<String, DataViewAuthVerticalEntity> authVerticalsMap = authVerticals.stream().collect(Collectors.toMap(DataViewAuthVerticalEntity::getId, v -> v, (v1, v2) -> v2));
    Set<DataViewAuthVerticalEntity> dbAuthVerticals = dataViewAuthVerticalRepository.findByAuth(auth);
    Set<DataViewAuthVerticalEntity> creates = Sets.newHashSet();
    Set<DataViewAuthVerticalEntity> updates = Sets.newHashSet();
    Set<DataViewAuthVerticalEntity> deletes = Sets.newHashSet();
    nebulaToolkitService.collectionDiscrepancy(authVerticals, dbAuthVerticals, DataViewAuthVerticalEntity::getId, deletes, updates, creates);
    // 删除数据
    for (DataViewAuthVerticalEntity delete : deletes) {
      dataViewAuthVerticalRelationService.deleteByAuthVertical(delete.getId());
      dataViewAuthVerticalRepository.delete(delete);
    }
    // 更新数据
    for (DataViewAuthVerticalEntity update : updates) {
      DataViewAuthVerticalEntity authVertical = authVerticalsMap.get(update.getId());
      update.setAuthType(authVertical.getAuthType());
      update.setDisplayFields(authVertical.getDisplayFields());
      update.setOprtType(authVertical.getOprtType());
      if (authVertical.getSortIndex() != null) {
        update.setSortIndex(authVertical.getSortIndex());
      }
      Set<DataViewAuthVerticalRelationEntity> relations = dataViewAuthVerticalRelationService.save(update, update.getAuthRelations());
      update.setAuthRelations(relations);
    }
    // 新增数据
    for (DataViewAuthVerticalEntity create : creates) {
      if (create.getSortIndex() == null) {
        create.setSortIndex(0);
      }
      create.setId(null);
      create.setDataViewAuth(auth);
      dataViewAuthVerticalRepository.save(create);
      Set<DataViewAuthVerticalRelationEntity> relations = dataViewAuthVerticalRelationService.save(create, create.getAuthRelations());
      create.setAuthRelations(relations);
    }
    updates.addAll(creates);
    return updates;
  }

  /**
   * 验证权限信息
   *
   * @param authVertical
   */
  private void validation(DataViewAuthVerticalEntity authVertical) {
    Validate.notBlank(authVertical.getDisplayFields(), "显示的字段不能为空");
    Validate.notBlank(authVertical.getOprtType(), "操作符不能为空");
    Validate.notNull(AuthOperatorEnum.valueOf(authVertical.getOprtType()), "不支持的操作符：%s", authVertical.getOprtType());
    Validate.notNull(authVertical.getAuthType(), "权限类型不能为空");
    Validate.inclusiveBetween(0, 6, authVertical.getAuthType(), "不支持的权限类型：%s", authVertical.getAuthType());
    Validate.notEmpty(authVertical.getAuthRelations(), "权限值不能为空");
  }

  /**
   * 更新数据验证
   *
   * @param dataViewAuth
   * @param authVerticals
   */
  private void saveValidation(String authId, Set<DataViewAuthVerticalEntity> authVerticals) {
    Validate.notBlank(authId, "权限对象的ID不能为空");
    if (CollectionUtils.isEmpty(authVerticals)) {
      return;
    }
    for (DataViewAuthVerticalEntity authVertical : authVerticals) {
      this.validation(authVertical);
    }
  }

  @Override
  public DataViewAuthVerticalEntity findDetailsById(String id) {
    if (StringUtils.isBlank(id)) {
      return null;
    }
    return this.dataViewAuthVerticalRepository.findDetailsById(id);
  }

  @Override
  public DataViewAuthVerticalEntity findById(String id) {
    if (StringUtils.isBlank(id)) {
      return null;
    }
    Optional<DataViewAuthVerticalEntity> op = dataViewAuthVerticalRepository.findById(id);
    return op.orElse(null);
  }

  @Override
  @Transactional
  public void deleteById(String id) {
    Validate.notBlank(id, "进行删除时，必须给定主键信息!!");
    DataViewAuthVerticalEntity av = this.dataViewAuthVerticalRepository.findById(id).orElse(null);
    if (av == null) {
      return;
    }
    dataViewAuthVerticalRelationService.deleteByAuthVertical(id);
    dataViewAuthVerticalRepository.delete(av);
  }

  @Override
  public Set<DataViewAuthVerticalEntity> findDetailsByAuthId(String authId) {
    if(StringUtils.isBlank(authId)) {
      return Sets.newHashSet();
    }
    DataViewAuthEntity auth = new DataViewAuthEntity();
    auth.setId(authId);
    Set<DataViewAuthVerticalEntity> verticals = dataViewAuthVerticalRepository.findByAuth(auth);
    if(CollectionUtils.isEmpty(verticals)) {
      return Sets.newHashSet();
    }
    for (DataViewAuthVerticalEntity vertical : verticals) {
      Set<DataViewAuthVerticalRelationEntity> relations = dataViewAuthVerticalRelationService.findDetailsByAuthVertical(vertical.getId());
      vertical.setAuthRelations(relations);
    }
    return verticals;
  }

  @Override
  @Transactional
  public void deleteByAuthId(String authId) {
    Validate.notBlank(authId, "权限ID不能为空");
    DataViewAuthEntity auth = new DataViewAuthEntity();
    auth.setId(authId);
    Set<DataViewAuthVerticalEntity> verticals = dataViewAuthVerticalRepository.findByAuth(auth);
    if(CollectionUtils.isEmpty(verticals)) {
      return;
    }
    for (DataViewAuthVerticalEntity vertical : verticals) {
      dataViewAuthVerticalRelationService.deleteByAuthVertical(vertical.getId());
    }
    dataViewAuthVerticalRepository.deleteAll(verticals);
  }

} 
