package com.bizunited.platform.core.service.internal;

import com.bizunited.platform.common.enums.ImportExecuteModeEnum;
import com.bizunited.platform.common.model.MigrateImportModel;
import com.bizunited.platform.common.service.redis.RedisMutexService;
import com.bizunited.platform.common.util.ChineseCharUtils;
import com.bizunited.platform.common.util.ZipFileUtils;
import com.bizunited.platform.core.common.PlatformContext;
import com.bizunited.platform.core.entity.CodeRuleEntity;
import com.bizunited.platform.core.entity.ScriptEntity;
import com.bizunited.platform.core.repository.CodeRuleRepository;
import com.bizunited.platform.core.service.CodeRuleService;
import com.bizunited.platform.core.service.ScriptService;
import com.bizunited.platform.core.service.invoke.InvokeProxyException;
import com.bizunited.platform.user.common.service.user.UserService;
import com.bizunited.platform.user.common.vo.UserVo;
import com.google.common.collect.Lists;
import com.google.common.collect.Sets;

import org.apache.commons.io.IOUtils;
import org.apache.commons.lang3.ObjectUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.domain.Pageable;
import org.springframework.security.core.Authentication;
import org.springframework.security.core.context.SecurityContext;
import org.springframework.security.core.context.SecurityContextHolder;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;

import javax.transaction.Transactional;
import java.io.IOException;
import java.io.InputStream;
import java.io.ObjectInputStream;
import java.nio.charset.StandardCharsets;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.zip.ZipEntry;
import java.util.zip.ZipFile;

import static com.bizunited.platform.common.constant.Constants.LINE_SEPARATOR;
import static com.bizunited.platform.common.constant.Constants.PROJECT_NAME;
import static com.bizunited.platform.common.constant.MigrateDataConstants.CODE_RULE_FILENAME;
import static javax.transaction.Transactional.TxType.REQUIRES_NEW;

/**
 * 2020年4月16日，对服务源进行了较大调整后，这里的相关使用方法也做了调整
 * @description:
 * @author: yanwe yinwenjie
 * @date: 12/Jun/2019 15:44
 */
@Service("CodeRuleServiceImpl")
public class CodeRuleServiceImpl implements CodeRuleService {

  private static final Logger LOGGER = LoggerFactory.getLogger(CodeRuleServiceImpl.class);
  @Autowired
  private CodeRuleRepository codeRuleRepository;
  @Autowired
  private UserService userService;
  @Autowired
  private RedisMutexService redisService;
  @Autowired
  private ScriptService scriptService;
  @Autowired
  private PlatformContext platformContext;
  @Override
  @Transactional
  public CodeRuleEntity create(CodeRuleEntity entity) {
    this.validEntity(entity, true);
    // 新增脚本内容
    UserVo user = this.getAccount();
    entity.setCreateAccount(user.getAccount());
    entity.setModifyAccount(user.getAccount());
    ScriptEntity script = new ScriptEntity();
    script.setLanguage(entity.getLanguage());
    // 默认保存的脚本实体名称为： "cgr-编码生成编码"
    script.setName(String.format("%s%s", "cgr", entity.getRuleCode()));
    script.setModifyAccount(user.getAccount());
    script = this.scriptService.create(script, entity.getContent());
    entity.setScript(script);
    entity.setCreateTime(new Date());
    entity.setModifyTime(new Date());
    entity.setCreateAccount(user.getAccount());
    entity.setModifyAccount(user.getAccount());
    entity.setProjectName(platformContext.getAppName());
    return codeRuleRepository.save(entity);
  }

  @Override
  @Transactional
  public CodeRuleEntity update(CodeRuleEntity entity) {
    this.validEntity(entity, false);
    UserVo user = this.getAccount();
    Validate.notBlank(entity.getId() , "当前业务编码规则的ID信息必须传入");
    Optional<CodeRuleEntity> exsit = codeRuleRepository.findById(entity.getId());
    Validate.isTrue(exsit.isPresent(), "编码生成规则未查询到！");
    CodeRuleEntity current = exsit.get();
    ScriptEntity scriptEntity = current.getScript();
    // 修改脚本内容
    if(scriptEntity == null){
      ScriptEntity script = new ScriptEntity();
      script.setLanguage(entity.getLanguage());
      // 默认保存的脚本实体名称为： "cgr-编码生成编码"
      script.setName(String.format("%s%s", "cgr", entity.getRuleCode()));
      script.setModifyAccount(user.getAccount());
      scriptEntity = this.scriptService.create(script, entity.getContent());
    }else{
      scriptEntity = this.scriptService.update(scriptEntity, entity.getContent());
    }
    current.setScript(scriptEntity);
    current.setDescription(entity.getDescription());
    current.setLanguage(entity.getLanguage());
    current.setRuleType(entity.getRuleType());
    current.setRuleName(entity.getRuleName());
    current.setModifyTime(new Date());
    current.setModifyAccount(user.getAccount());
    return codeRuleRepository.save(current);
  }

  @Override
  @Transactional
  public CodeRuleEntity updateContent(String ruleId, String content) {
    Validate.notBlank(ruleId, "编码生成规则ID不能为空！");
    Validate.notBlank(content, "脚本内容不能为空！");
    Optional<CodeRuleEntity> exsit = codeRuleRepository.findById(ruleId);
    Validate.isTrue(exsit.isPresent(), "编码生成规则未查询到！");
    CodeRuleEntity entity = exsit.get();
    ScriptEntity scriptEntity = entity.getScript();
    if (null == scriptEntity) {
      // 新增脚本内容
      ScriptEntity script = new ScriptEntity();
      script.setLanguage(entity.getLanguage());
      // 默认保存的脚本实体名称为： "cgr-编码生成编码"
      script.setName(String.format("%s%s", "cgr", entity.getRuleCode()));
      script = this.scriptService.create(script, content);
      entity.setScript(script);
    } else {
      // 修改脚本内容
      scriptEntity = this.scriptService.update(scriptEntity, content);
      entity.setScript(scriptEntity);
    }
    return codeRuleRepository.save(entity);
  }

  @Override
  public String invoke(String ruleCode) {
    Validate.notBlank(ruleCode,"规则编码不能为空！");

    CodeRuleEntity codeRuleEntity = codeRuleRepository.findDetailByCode(ruleCode);
    Validate.notNull(codeRuleEntity,"未根据该编码查询到编码规则！");
    ScriptEntity scriptEntity = codeRuleEntity.getScript();
    Validate.notNull(scriptEntity,"未查询到该规则绑定的脚本信息！");
    Validate.notBlank(scriptEntity.getId(),"脚本id不能为空！");

    try{
      // 调用服务源
      Map<String, Object> params = new HashMap<>();
      params.put("ruleCode", ruleCode);
      params.put("redisService", redisService);
      params.put("userService", userService);
      Map<String,Object> result = scriptService.invoke(new String[] {scriptEntity.getId()} , params);
      return String.valueOf(result.get("resultKey"));
    }catch (InvokeProxyException e){
      throw new IllegalArgumentException("执行生成编码脚本错误！");
    }
  }

  /**
   * 验证新增/修改实体
   *
   * @param entity
   * @param isCreate 是否新增
   */
  private void validEntity(CodeRuleEntity entity, Boolean isCreate) {
    Validate.notNull(entity, "编码生成规则基础信息不能为空！");
    Validate.notBlank(entity.getContent(), "脚本内容不能为空！");
    Validate.notBlank(entity.getRuleName(), "编码生成规则名称不能为空!");
    Validate.notBlank(entity.getRuleCode(), "编码生成规则编码不能为空!");
    Validate.isTrue(entity.getRuleName().length() <= 64, "编码生成规则名称不能超过64个字符");
    Validate.isTrue(entity.getRuleCode().length() <= 64, "编码生成规则编码不能超过64个字符");
    Validate.isTrue(!ChineseCharUtils.hasChinese(entity.getRuleCode()), "编码生成规则编码不能含有中文！");
    // 验证编码是否重复
    CodeRuleEntity existName = codeRuleRepository.findByRuleName(entity.getRuleName());
    CodeRuleEntity existCode = codeRuleRepository.findByRuleCode(entity.getRuleCode());
    if (Boolean.TRUE.equals(isCreate)) {
      Validate.isTrue(StringUtils.isBlank(entity.getId()), "新增时ID必须为空！");
      Validate.isTrue(null == existName, "编码规则名称已有重复，请检查！");
      Validate.isTrue(null == existCode, "编码规则编码已有重复，请检查！");
    } else {
      Validate.notBlank(entity.getId(), "修改时id不能为空！");
      Optional<CodeRuleEntity> op = codeRuleRepository.findById(entity.getId());
      CodeRuleEntity oldEntity = op.orElse(null);
      Validate.notNull(oldEntity, "未根据ID查询到编码生成规则实体！");
      Validate.isTrue(StringUtils.equals(entity.getRuleCode(),oldEntity.getRuleCode()),"规则编码不能修改！");
      long count = codeRuleRepository.countByRuleNameWithoutId(entity.getRuleName(), entity.getId());
      Validate.isTrue(count == 0, "编码规则名称已有重复，请检查！");
    }
  }

  /**
   * 获取当前登录账户ID，注入新的UserEntity
   * @return
   */
  private UserVo getAccount(){
    // 获取当前登录账户
    SecurityContext securityContext = SecurityContextHolder.getContext();
    Validate.notNull(securityContext, "未发现任何用户权限信息!!");
    Authentication authentication = securityContext.getAuthentication();
    Validate.notNull(authentication, "未发现任何用户登录信息!!");
    UserVo userVo = userService.findByAccount(authentication.getName());
    Validate.notNull(userVo, "未获取到当前登录账号！");
    return userVo;
  }

  @Override
  public String findContentByCode(String code) {
  
    Validate.notBlank(code, "编码生成规则的业务编号不能为空！");
    CodeRuleEntity entity = codeRuleRepository.findDetailByCode(code);
    if (entity.getScript() == null) {
      return null;
    }
    String scriptId = entity.getScript().getId();
    return scriptService.findContentById(scriptId);
  }

  @Override
  public Page<CodeRuleEntity> findByConditions(
      Pageable pageable, String ruleName, String ruleCode, String ruleType, String language) {
    Map<String, Object> params = new HashMap<>();
    if (StringUtils.isNotEmpty(ruleName)) {
      params.put("ruleName", ruleName);
    }
    if (StringUtils.isNotEmpty(ruleCode)) {
      params.put("ruleCode", ruleCode);
    }
    if (StringUtils.isNotEmpty(ruleType)) {
      params.put("ruleType", ruleType);
    }
    if (StringUtils.isNotEmpty(language)) {
      params.put("language", language);
    }
    params.put(PROJECT_NAME, platformContext.getAppName());
    if (null == pageable) {
      pageable = PageRequest.of(0, 50);
    }
    return codeRuleRepository.findByConditions(pageable, params);
  }

  @Override
  public Set<CodeRuleEntity> findDetailsByIds(String[] ids) {
    if(ids == null || ids.length <= 0) { 
      return Sets.newHashSet();
    }
    return this.codeRuleRepository.findDetailsByIds(ids);
  }

  @Override
  public int countByIds(String[] ids) {
    if(ids == null || ids.length <= 0) {
      return 0;
    }
    return this.codeRuleRepository.countByIds(ids);
  }

  @Override
  public CodeRuleEntity findByRuleCode(String ruleCode) {
    if(StringUtils.isBlank(ruleCode)){
      return null;
    }
    return codeRuleRepository.findByRuleCode(ruleCode);
  }

  @Override
  public CodeRuleEntity findByRuleName(String ruleName) {
    if(StringUtils.isBlank(ruleName)){
      return null;
    }
    return codeRuleRepository.findByRuleName(ruleName);
  }

  /**
   * 查询所有的编码规则
   * @return
   */
  @Override
  public Set<CodeRuleEntity> findAll() {
    List<CodeRuleEntity> all = codeRuleRepository.findAll();
    return Sets.newHashSet(all);
  }

  /**
   * 导入编码规则
   * @param importModel
   */
  @Override
  @Transactional(REQUIRES_NEW)
  @SuppressWarnings("unchecked")
  public void importData(MigrateImportModel importModel) {
    Validate.notNull(importModel, "导入信息不能为空");
    ZipFile zipFile = importModel.getZipFile();
    Validate.notNull(zipFile, "导入文件不能为空");
    Validate.notNull(importModel.getExecuteMode(), "执行模式不能为空");
    StringBuilder executeLog = new StringBuilder();
    importModel.appendLine("开始导入数据");
    ZipEntry codeRuleEntry = zipFile.getEntry(CODE_RULE_FILENAME);
    if(codeRuleEntry == null) {
      importModel.appendLine("导入压缩包中未发现数据文件，请检查");
    }
    if(codeRuleEntry != null) {
      try (InputStream is = zipFile.getInputStream(codeRuleEntry)) {
        ObjectInputStream ois = new ObjectInputStream(is);
        Set<CodeRuleEntity> codeRulesSet = (Set<CodeRuleEntity>) ois.readObject();
        List<CodeRuleEntity> codeRules = Lists.newArrayList(codeRulesSet);
        for(CodeRuleEntity codeRule : codeRules){
          // 读取出脚本文件内容
          ScriptEntity script = codeRule.getScript();
          Boolean startWith = StringUtils.startsWith(script.getFileCode(), "/") || StringUtils.startsWith(script.getFileCode(), "\\");
          String filePath = Boolean.TRUE.equals(startWith) ? script.getFileCode().substring(1) : script.getFileCode();
          // 将脚本文件从之前保存的ZIP文件特定路径下获取出
          ZipEntry scriptFile = ZipFileUtils.getZipEntry(zipFile, filePath, script.getFileName());
          InputStream inputStream = zipFile.getInputStream(scriptFile);
          byte[] bytes = IOUtils.toByteArray(inputStream);

          String content = new String(bytes, StandardCharsets.UTF_8);
          codeRule.setContent(content);
        }

        if(CollectionUtils.isEmpty(codeRules)) {
          importModel.appendLine("导入的数据为空");
        } else {
          this.importData(codeRules, importModel);
        }
      } catch (IOException | ClassNotFoundException e) {
        LOGGER.error(e.getMessage(), e);
        executeLog.append("读取业务数据失败：").append(e.getMessage()).append(LINE_SEPARATOR);
      }
    }
  }

  /**
   * 按照条件搜索编码生成规则，不支持模糊查询,不分页
   * @param codeRule
   * @return
   */
  @Override
  public List<CodeRuleEntity> findAllByConditions(CodeRuleEntity codeRule) {
    ObjectUtils.defaultIfNull(codeRule, new CodeRuleEntity());
    codeRule.setProjectName(platformContext.getAppName());
    return codeRuleRepository.findAllByConditions(codeRule);
  }

  /**
   * 处理数据
   * @param codeRules
   * @param importModel
   */
  private void importData(List<CodeRuleEntity> codeRules, MigrateImportModel importModel) {
    importModel.setTotalCount(codeRules.size());
    for (int i = 0; i < codeRules.size(); i++) {
      CodeRuleEntity codeRule = codeRules.get(i);
      importModel.appendLine(StringUtils.join("--------[", i + 1, "]----------"));
      this.importData(codeRule, importModel);
    }
  }

  /**
   * 单个处理
   * @param codeRule
   * @param importModel
   */
  private void importData(CodeRuleEntity codeRule, MigrateImportModel importModel) {
    importModel.append("开始处理数据：").appendLine(codeRule.getRuleName());
    ImportExecuteModeEnum executeMode = importModel.getExecuteMode();
    CodeRuleEntity dbCodeRule = codeRuleRepository.findByRuleCode(codeRule.getRuleCode());
    this.handleImportDataId(codeRule);
    if(dbCodeRule != null && ImportExecuteModeEnum.SKIP == executeMode) {
      importModel.appendLine("编码规则已存在，跳过");
      importModel.addSkipCount();
      return;
    }
    if(dbCodeRule != null && ImportExecuteModeEnum.UPDATE == executeMode) {
      importModel.appendLine("编码规则已存在，进行更新覆盖");
      this.handleUpdateData(codeRule, dbCodeRule, importModel);
      return;
    }
    if(dbCodeRule == null) {
      this.handleCreateData(codeRule, importModel);
    }

  }

  /**
   * 导入新增编码规则
   * @param codeRule
   * @param importModel
   */
  private void handleCreateData(CodeRuleEntity codeRule, MigrateImportModel importModel) {
    importModel.appendLine("开始新增编码规则");
    importModel.appendLine("导入编码规则");
    this.create(codeRule);
    importModel.addCreateCount();
  }

  /**
   * 导入更新编码规则
   * @param codeRule
   * @param dbCodeRule
   * @param importModel
   */
  private void handleUpdateData(CodeRuleEntity codeRule, CodeRuleEntity dbCodeRule, MigrateImportModel importModel) {
    codeRule.setId(dbCodeRule.getId());
    this.update(codeRule);
    importModel.appendLine("更新成功");
    importModel.addUpdateCount();
  }

  /**
   * 处理数据id
   * @param codeRule
   */
  private void handleImportDataId(CodeRuleEntity codeRule) {
    codeRule.setId(null);
    ScriptEntity script = codeRule.getScript();
    if(script != null) {
      script.setId(null);
    }
  }
}
