package com.bizunited.platform.core.service.internal;

import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONObject;
import com.bizunited.platform.common.enums.ImportExecuteModeEnum;
import com.bizunited.platform.common.model.MigrateImportModel;
import com.bizunited.platform.common.util.ChineseCharUtils;
import com.bizunited.platform.common.util.ZipFileUtils;
import com.bizunited.platform.core.common.PlatformContext;
import com.bizunited.platform.core.entity.RemoteServiceAddressEntity;
import com.bizunited.platform.core.entity.RemoteServiceEntity;
import com.bizunited.platform.core.repository.RemoteServiceAddressRepository;
import com.bizunited.platform.core.service.RemoteServiceAddressService;
import com.bizunited.platform.core.service.RemoteServiceService;
import com.google.common.collect.Lists;
import com.google.common.collect.Sets;
import feign.Client;
import feign.Request;
import feign.Request.HttpMethod;
import feign.Response;
import org.apache.commons.lang3.ArrayUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.Pageable;
import org.springframework.http.HttpStatus;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;

import javax.transaction.Transactional;
import java.io.IOException;
import java.io.InputStream;
import java.io.ObjectInputStream;
import java.nio.charset.Charset;
import java.nio.charset.StandardCharsets;
import java.util.Collection;
import java.util.Date;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.zip.ZipEntry;
import java.util.zip.ZipFile;

import static com.bizunited.platform.common.constant.Constants.PROJECT_NAME;
import static com.bizunited.platform.common.constant.MigrateDataConstants.REMOTE_SERVICE_FILENAME;
import static javax.transaction.Transactional.TxType.REQUIRES_NEW;

/**
 * RemoteServiceAddressServiceImpl
 *
 * @description:
 * @author: yanwe
 * @date: 06/May/2019 10:23
 */
@Service("RemoteServiceAddressServiceImpl")
public class RemoteServiceAddressServiceImpl implements RemoteServiceAddressService {

  private static final Logger LOGGER = LoggerFactory.getLogger(RemoteServiceAddressServiceImpl.class);
  private static final String REQUEST_METHOD = "RequestMethod";
  private static final String INTERFACE = "Interface";
  private static final String HEAD = "Head";
  private static final String REQUEST_PARAMS = "RequestParams";
  private static final String REQUEST_BODY = "RequestBody";
  private static final String ADDRESS_E_ERROR = "入参远程服务实体JSON不能为空";
  private static final String ADDRESS_P_ERROR = "入参远程服务参数JSON不能为空";
  private static final String ADDRESS_C_ERROR = "参数执行转化时发生错误，请检查！";
  private static final String ADDRESS_R_ERROR = "传入远端调用源不存在，请检查！";

  @Autowired
  private RemoteServiceAddressRepository  remoteServiceAddressRepository;
  @Autowired
  private RemoteServiceService remoteServiceService;
  @Autowired
  private PlatformContext platformContext;

  @Override
  public Page<RemoteServiceAddressEntity> findByConditions(String code, String alias, Pageable pageable) {
    Validate.notNull(pageable, "查询分页信息不能为空!");
    Map<String, Object> conditions = new HashMap<>();
    if (!StringUtils.isBlank(code)) {
      conditions.put("code", code);
    }
    if (!StringUtils.isBlank(alias)) {
      conditions.put("alias", alias);
    }
    conditions.put(PROJECT_NAME, platformContext.getAppName());
    return remoteServiceAddressRepository.findByConditions(pageable, conditions);
  }

  @Override
  public void valid(String entityJson, String testJsonStr) {
    Validate.notBlank(entityJson,ADDRESS_E_ERROR);
    RemoteServiceAddressEntity remoteServiceAddress = this.analysisEntityJson(entityJson);
    JSONObject object = this.analysisJsonStr(testJsonStr);
    boolean isCreate = StringUtils.isBlank(remoteServiceAddress.getId());
    this.validEntity(remoteServiceAddress, isCreate);
    this.validConnection(remoteServiceAddress, object);
  }

  /**
   * 根据entityJson的结构生成一个RemoteServiceAddressEntity类的实例
   * @param entityJson
   * @return
   */
  private RemoteServiceAddressEntity analysisEntityJson(String entityJson) {
    RemoteServiceAddressEntity entity;
    try{
      entity = JSON.parseObject(entityJson,RemoteServiceAddressEntity.class);
    }catch (RuntimeException e){
      throw new IllegalArgumentException(ADDRESS_C_ERROR, e);
    }
    return entity;
  }

  /**
   * 根据jsonStr的结构生成一个JSONObject类的实例
   * @param jsonStr
   * @return
   */
  private JSONObject analysisJsonStr(String jsonStr) {
    JSONObject testJson;
    try{
      testJson = JSON.parseObject(jsonStr);
    }catch (RuntimeException e){
      throw new IllegalArgumentException(ADDRESS_C_ERROR, e);
    }
    return testJson;
  }

  @Override
  @Transactional
  public RemoteServiceAddressEntity create(String entityJson, String testJsonStr) {
    Validate.notBlank(entityJson,ADDRESS_E_ERROR);
    RemoteServiceAddressEntity entity = this.analysisEntityJson(entityJson);
    JSONObject object = this.analysisJsonStr(testJsonStr);
    this.validEntity(entity, true);
    this.validConnection(entity, object);
    entity.setCreateDate(new Date());
    entity.setAddressStatus(2);
    entity.setProjectName(platformContext.getAppName());
    return remoteServiceAddressRepository.save(entity);
  }

  @Override
  @Transactional
  public RemoteServiceAddressEntity create(RemoteServiceAddressEntity remoteServiceAddressEntity) {
    this.validEntity(remoteServiceAddressEntity,true);
    remoteServiceAddressEntity.setProjectName(platformContext.getAppName());
    return this.remoteServiceAddressRepository.save(remoteServiceAddressEntity);
  }

  @Override
  @Transactional
  public RemoteServiceAddressEntity update(String entityJson, String testJsonStr) {
    Validate.notBlank(entityJson,ADDRESS_E_ERROR);
    Validate.notBlank(testJsonStr,ADDRESS_P_ERROR);
    RemoteServiceAddressEntity entity;
    JSONObject testJson;
    try{
      entity = JSON.parseObject(entityJson,RemoteServiceAddressEntity.class);
      testJson = JSON.parseObject(testJsonStr);
    }catch (Exception e){
      throw new IllegalArgumentException(ADDRESS_C_ERROR,e);
    }
    this.validEntity(entity, false);
    this.validConnection(entity, testJson);
    // 修改
    Optional<RemoteServiceAddressEntity> op = remoteServiceAddressRepository.findById(entity.getId());
    RemoteServiceAddressEntity existEntity = op.orElse(null);
    Validate.notNull(existEntity,"未能查询到远程服务信息!!");

    existEntity.setProtocolType(entity.getProtocolType());
    existEntity.setConnectionType(entity.getConnectionType());
    existEntity.setAddress(entity.getAddress());
    existEntity.setAddressDesc(entity.getAddressDesc());
    return remoteServiceAddressRepository.save(existEntity);
  }

  @Override
  @Transactional
  public RemoteServiceAddressEntity enable(String addressId) {
    Validate.notBlank(addressId, "传入远端调用源ID不能为空!");
    Optional<RemoteServiceAddressEntity> op = remoteServiceAddressRepository.findById(addressId);
    RemoteServiceAddressEntity existEntity = op.orElse(null);
    Validate.notNull(existEntity,ADDRESS_R_ERROR);

    Validate.isTrue(existEntity.getAddressStatus() == 2, "启用的远端调用源状态必须为未启用，请检查！");
    existEntity.setAddressStatus(1);
    return remoteServiceAddressRepository.save(existEntity);
  }

  @Override
  @Transactional
  public RemoteServiceAddressEntity disable(String addressId) {
    Validate.notBlank(addressId, "传入远端调用源ID不能为空!");
    Optional<RemoteServiceAddressEntity> op = remoteServiceAddressRepository.findById(addressId);
    RemoteServiceAddressEntity existEntity = op.orElse(null);
    Validate.notNull(existEntity,ADDRESS_R_ERROR);

    Validate.isTrue(existEntity.getAddressStatus() == 1, "禁用的远端调用源状态必须为启用，请检查！");
    existEntity.setAddressStatus(0);
    //禁用远端地址的同时，禁用它下面的所有服务
    remoteServiceService.updateRemoteService(addressId, 0);
    return remoteServiceAddressRepository.save(existEntity);
  }

  @Override
  public RemoteServiceAddressEntity findByCode(String code) {
    return remoteServiceAddressRepository.findByCode(code);
  }

  /**
   * 新增/修改验证远端源管理数据
   *
   * @param entity
   */
  private void validEntity(RemoteServiceAddressEntity entity, Boolean isCreate) {
    Validate.notNull(entity, "传入远端地址数据不能为空！");
    Validate.notBlank(entity.getCode(), "远端调用源编码不能为空，请检查！");
    Validate.notBlank(entity.getAlias(), "远端调用源别名不能为空，请检查！");
    Validate.isTrue(!ChineseCharUtils.hasChinese(entity.getAlias()), "新增远端数据源别名不能含有中文！");
    Validate.isTrue(!ChineseCharUtils.hasChinese(entity.getCode()), "新增远端数据源编码不能含有中文！");
    Validate.notNull(entity.getProtocolType(), "远端调用源协议类型不能为空，请检查！");
    Validate.notNull(entity.getConnectionType(), "远端调用源连接类型不能为空，请检查！");
    Validate.notNull(entity.getAddress(), "远端调用源连接地址不能为空，请检查！");
    Validate.isTrue(!entity.getAddress().endsWith("/"), "连接地址不能以 / 号结尾，请移除！");
    Validate.notNull(entity.getAddressStatus(), "远端调用源状态不能为空，请检查！");
    // 新增
    if (Boolean.TRUE.equals(isCreate)) {
      // 验重
      RemoteServiceAddressEntity existCodes =
          remoteServiceAddressRepository.findByCode(entity.getCode());
      RemoteServiceAddressEntity existAliases =
          remoteServiceAddressRepository.findByAlias(entity.getAlias());
      Validate.isTrue(StringUtils.isBlank(entity.getId()), "新增的远端调用源ID必须为空！");
      Validate.isTrue(null == existAliases, "新增远端数据源别名重复，请更换别名！");
      Validate.isTrue(null == existCodes, "新增远端数据源编码重复，请更换编码！");
    } else {
      // 修改
      Validate.notBlank(entity.getId(), "修改的远端调用源ID不能为空！");
      Optional<RemoteServiceAddressEntity> op = remoteServiceAddressRepository.findById(entity.getId());
      RemoteServiceAddressEntity existEntity = op.orElse(null);
      Validate.notNull(existEntity,ADDRESS_R_ERROR);

      Validate.isTrue(existEntity.getAddressStatus() == 2, "只能对未启用状态的远程调用源做修改！");
      Validate.isTrue(existEntity.getCode().equals(entity.getCode()), "修改时不能修改远端调用源编号！");
      Validate.isTrue(existEntity.getAlias().equals(entity.getAlias()), "修改时不能修改远端调用源别名！");
    }
  }

  /*
   * <code>
   * 测试接口数据格式：
   * {
   *  "RequestMethod":"POST"    ---测试接口方法，POST/GET
   *  "Interface":"/uuuu",      ---测试接口（如果远端源类型为自定义，则必填，如果为spring cloud形式，则无需填写）
   *  "Head": {
   *    "a":"b",
   *    ...
   *  },
   *  "RequestParams": {
   *    "a":"b",
   *    ...
   *  },
   *  "RequestBody": {...}      ---测试请求体，格式为JSON
   *  }
   *  </code>
   **/

  /**
   * 验证新增远端数据源测试数据
   *
   * @param testJson
   */
  private void validConnection(RemoteServiceAddressEntity entity, JSONObject testJson) {
    Validate.notNull(testJson, "测试接口数据不能为空！");
    Validate.notBlank(testJson.getString(REQUEST_METHOD), "测试接口方法不能为空！");
    // 如果有接口地址，必须以'/'开始
    if (StringUtils.isNotEmpty(testJson.getString(INTERFACE))) {
      Validate.isTrue(testJson.getString(INTERFACE).startsWith("/"), "接口地址若不为空，必须以/号开始");
    }
    Client.Default defaultClient = new Client.Default(null, null);
    // 建立Request
    // 方法
    String method = testJson.getString(REQUEST_METHOD);
    // 路径
    String url =
        StringUtils.isEmpty(testJson.getString(INTERFACE))
            ? entity.getAddress()
            : entity.getAddress() + testJson.getString(INTERFACE);
    // HEAD
    Map<String, Collection<String>> headers = new HashMap<>();
    if (null != testJson.getJSONObject(HEAD)) {
      JSONObject heads = testJson.getJSONObject(HEAD);
      heads.keySet().forEach(o -> headers.put(o, Sets.newHashSet(heads.getString(o))));
    }
    // requestParams
    if (null != testJson.getJSONObject(REQUEST_PARAMS)) {
      JSONObject requestParams = testJson.getJSONObject(REQUEST_PARAMS);
      Set<String> paramSet = new HashSet<>();
      for(String key : requestParams.keySet()){
        paramSet.add(String.format("%s=%s",key,requestParams.getString(key)));
      }
      String paraUrl = String.format("?%s",StringUtils.join(paramSet,"&"));
      url = url.concat(paraUrl);
    }
    // requestBody
    byte[] body = new byte[] {};
    JSONObject bodyJson = testJson.getJSONObject(REQUEST_BODY);
    if (null != bodyJson) {
      String bodyStr = bodyJson.toJSONString();
      body = bodyStr.getBytes(StandardCharsets.UTF_8);
    }


    Request.Options options = new Request.Options(18000, 18000);
    HttpMethod httpMethod = HttpMethod.valueOf(method);
    Request request = Request.create(httpMethod, url, headers, body, Charset.defaultCharset());
    try {
      Response response = defaultClient.execute(request, options);
      Validate.isTrue(response.status() == HttpStatus.OK.value(), String.format("连接失败，返回编码：%s,原因：%s", response.status(), response.reason()));
    } catch (IOException e) {
      throw new IllegalArgumentException(e.getMessage() , e);
    }
  }

  /**
   * 根据状态查询远端服务地址
   * @param status
   * @return
   */
  @Override
  public List<RemoteServiceAddressEntity> findByAddressStatus(Integer status) {
    if(status == null){
      return Lists.newArrayList();
    }
    return remoteServiceAddressRepository.findByAddressStatus(status);
  }

  /**
   * 根据id查询远端服务地址数量
   * @param ids
   * @return
   */
  @Override
  public int countByIds(String[] ids) {
    if(ids == null || ids.length == 0) {
      return 0;
    }
    return this.remoteServiceAddressRepository.countByIds(ids);
  }

  /**
   * 根据id查询远端服务地址，包含关联信息
   * @param ids
   * @return
   */
  @Override
  public List<RemoteServiceAddressEntity> findDetailsByIds(String[] ids) {
    if(ids == null || ids.length == 0) {
      return Lists.newArrayList();
    }
    return this.remoteServiceAddressRepository.findDetailsByIds(ids);
  }

  @Override
  @Transactional(REQUIRES_NEW)
  @SuppressWarnings("unchecked")
  public void importData(MigrateImportModel importModel) {
    Validate.notNull(importModel, "导入信息不能为空");
    ZipFile zipFile = importModel.getZipFile();
    Validate.notNull(zipFile, "导入文件不能为空");
    Validate.notNull(importModel.getExecuteMode(), "执行模式不能为空");
    importModel.appendLine("开始导入数据");
    ZipEntry dataViewEntry = zipFile.getEntry(REMOTE_SERVICE_FILENAME);
    if(dataViewEntry == null) {
      importModel.appendLine("导入压缩包中未发现数据文件，请检查");
    }
    if(dataViewEntry != null) {
      try (InputStream is = zipFile.getInputStream(dataViewEntry);
           ObjectInputStream ois = new ObjectInputStream(is)) {

        List<RemoteServiceAddressEntity> importRemoteServiceAddress = (List<RemoteServiceAddressEntity>) ois.readObject();
        if(CollectionUtils.isEmpty(importRemoteServiceAddress)) {
          importModel.appendLine("导入的数据为空，请检查数据");
          return;
        }
        this.importData(importRemoteServiceAddress, importModel);
      } catch (IOException | ClassNotFoundException e) {
        LOGGER.error(e.getMessage(), e);
        importModel.append("读取业务数据失败：").appendLine(e.getMessage());
      }
    }
  }

  /**
   * 批量导入远端调用源数据
   * @param importRemoteServiceAddress
   * @param importModel
   */
  private void importData(List<RemoteServiceAddressEntity> importRemoteServiceAddress, MigrateImportModel importModel) {
    importModel.setTotalCount(importRemoteServiceAddress.size());
    for (int i = 0; i < importRemoteServiceAddress.size(); i++) {
      RemoteServiceAddressEntity address = importRemoteServiceAddress.get(i);
      importModel.appendLine(StringUtils.join("--------[", i + 1, "]----------"));
      this.importData(address, importModel);
    }
  }

  /**
   * 单条导入远端调用源数据
   * @param address
   * @param importModel
   */
  private void importData(RemoteServiceAddressEntity address, MigrateImportModel importModel) {
    importModel.append("开始导入数据：").appendLine(address.getCode());
    ImportExecuteModeEnum executeMode = importModel.getExecuteMode();
    RemoteServiceAddressEntity dbService = remoteServiceAddressRepository.findByCode(address.getCode());
    this.handleImportDataId(address);
    if(dbService != null && ImportExecuteModeEnum.SKIP == executeMode) {
      importModel.appendLine("远端数据源已存在，执行跳过");
      importModel.addSkipCount();
      return;
    }
    if(dbService != null && ImportExecuteModeEnum.UPDATE == executeMode) {
      importModel.appendLine("远端数据源已存在，将执行更新");
      this.handleUpdateData(address, dbService, importModel);
      return;
    }
    if(dbService == null) {
      this.handleCreateData(address, importModel);
      return;
    }
    importModel.appendLine("暂不支持的执行方式");
    importModel.addSkipCount();
  }

  /**
   * 处理新增数据
   * @param service
   * @param importModel
   */
  private void handleCreateData(RemoteServiceAddressEntity service, MigrateImportModel importModel) {
    importModel.appendLine("开始导入新增数据源");
    RemoteServiceAddressEntity dbAddress = remoteServiceAddressRepository.findByCode(service.getCode());
    if(dbAddress == null) {
      importModel.appendLine("导入数据源关联的调用地址");
      dbAddress = this.create(service);
    }
    ZipFile zipFile = importModel.getZipFile();
    try {
      for(RemoteServiceEntity item : service.getRemoteServices()) {
        item.setRemoteServiceAddress(dbAddress);
        importModel.appendLine("读取json内容").appendLine(item.getCode());
        byte[] bytes = ZipFileUtils.readZipFile(zipFile, item.getJsonRelativePath(), item.getJsonName());
        Validate.isTrue(bytes != null && bytes.length > 0, "json内容为空，请检查数据");
        String json = new String(bytes, StandardCharsets.UTF_8);
        importModel.appendLine("导入新增数据源");
        String itemJson = JSON.toJSONString(item);
        remoteServiceService.create(itemJson, json);
      }
      importModel.addCreateCount();
      importModel.appendLine("导入新增成功！！");
    } catch (IOException e) {
      LOGGER.error(e.getMessage(),e);
      throw new IllegalArgumentException("读取json内容出错", e);
    }
  }

  /**
   * 处理更新的数据
   * @param address
   * @param importModel
   */
  private void handleUpdateData(RemoteServiceAddressEntity address, RemoteServiceAddressEntity dbAddress, MigrateImportModel importModel) {
    Validate.isTrue(address.getCode().equals(dbAddress.getCode()), "导入远端数据源的远端调用地址与本系统的远端调用编码不一致");
    for(RemoteServiceEntity service : address.getRemoteServices()){
      service.setRemoteServiceAddress(dbAddress);
    }
    Map<String, RemoteServiceEntity> map = new HashMap<>();
    for(RemoteServiceEntity dbItem : dbAddress.getRemoteServices()){
      map.put(dbItem.getCode(), dbItem);
    }
    address.setId(dbAddress.getId());
    remoteServiceAddressRepository.save(address);

    ZipFile zipFile = importModel.getZipFile();
    try {
      for(RemoteServiceEntity item : address.getRemoteServices()) {
        if(!map.containsKey(item.getCode())){
          continue;
        }
        item.setId(map.get(item.getCode()).getId());
        importModel.appendLine("读取json内容");
        byte[] bytes = ZipFileUtils.readZipFile(zipFile, item.getJsonRelativePath(), item.getJsonName());
        Validate.isTrue(ArrayUtils.isNotEmpty(bytes), "json内容为空，请检查数据");
        String json = new String(bytes, StandardCharsets.UTF_8);
        String itemJson = JSON.toJSONString(item);
        remoteServiceService.update(itemJson, json);
      }
      importModel.addUpdateCount();
      importModel.appendLine("更新成功！！");
    } catch (IOException e) {
      LOGGER.error(e.getMessage(),e);
      throw new IllegalArgumentException("读取json内容出错", e);
    }
  }

  /**
   * 将导入数据对象的ID设置成null
   * @param address
   * @return
   */
  private RemoteServiceAddressEntity handleImportDataId(RemoteServiceAddressEntity address) {
    address.setId(null);
    Set<RemoteServiceEntity> service = address.getRemoteServices();
    for(RemoteServiceEntity item : service) {
      item.setId(null);
    }
    return address;
  }
}
