package com.bizunited.platform.core.service.internal;

import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONObject;
import com.bizunited.platform.core.common.PlatformContext;
import com.bizunited.platform.core.entity.SystemThemeEntity;
import com.bizunited.platform.core.repository.SystemThemeRepository;
import com.bizunited.platform.core.service.SystemThemeService;
import com.bizunited.platform.venus.common.service.file.FileRelativeTemplate;
import com.bizunited.platform.venus.common.service.file.VenusFileService;
import com.bizunited.platform.venus.common.service.image.FileUpdateService;
import com.bizunited.platform.venus.common.vo.OrdinaryFileVo;
import org.apache.commons.io.FileUtils;
import org.apache.commons.lang3.ArrayUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;
import org.springframework.web.multipart.MultipartFile;

import javax.transaction.Transactional;
import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.nio.charset.StandardCharsets;
import java.security.Principal;
import java.text.SimpleDateFormat;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.UUID;
import java.util.zip.ZipEntry;
import java.util.zip.ZipFile;
import java.util.zip.ZipOutputStream;

//TODO 暂时这一版数据始终只有一条，用于全局
@Service("SystemThemeServiceImpl")
public class SystemThemeServiceImpl implements SystemThemeService {
  @Value("${venus.file.fileRoot}")
  private String fileRoot;

  private static final Logger LOGGER = LoggerFactory.getLogger(SystemThemeServiceImpl.class);
  private static final String THEME_FILE_NAME = "theme.json";
  private static final String MESS_FIELD_NAME = "fileName";
  private static final String MESS_RELATIVE_LOCAL = "relativeLocal";
  private static final String MESS_THEME = "theme";
  private static final String UNDEFINED = "undefined/undefined";
  @Autowired
  private FileUpdateService fileUpdateService;
  @Autowired
  private VenusFileService venusFileService;
  @Autowired
  private SystemThemeRepository systemThemeRepository;
  @Autowired
  private FileRelativeTemplate fileRelativeTemplate;
  @Autowired
  private PlatformContext platformContext;

  @Override
  @Transactional
  public OrdinaryFileVo uploadLogo(MultipartFile file, Principal principal) {
    String account = this.validatePrincipal(principal);
    Validate.notNull(file,"logo图片数据不能为空，请检查!!");
    List<OrdinaryFileVo> logos = fileUpdateService.fileUpload("logo",account,null, new MultipartFile[]{file});
    Validate.notEmpty(logos,"logo文件上传失败，请检查!!");
    Validate.isTrue(!logos.isEmpty(),"logo文件上传失败，请检查!!");
    return logos.get(0);
  }

  @Override
  @Transactional
  public SystemThemeEntity save(JSONObject theme, Principal principal) {
    String account = this.validatePrincipal(principal);
    Validate.notEmpty(theme,"系统主题数据不能为空，请检查!!");

 // 解析当前传递的json结构中可能携带的文件名和文件路径信息
    // 如果没有则进行补足
    Map<String,String> themePath = this.resolveThemePath(theme);
    String folderName = new SimpleDateFormat("yyyyMMdd").format(new Date());
    String relativePath = themePath.get(MESS_RELATIVE_LOCAL);
    if(StringUtils.isBlank(relativePath)) {
      relativePath = "/theme/" + folderName;
      JSONObject themePathObject = theme.getJSONObject(MESS_THEME); 
      themePathObject.put(MESS_RELATIVE_LOCAL, relativePath);
    }
    String renameFile = themePath.get(MESS_FIELD_NAME);
    if(StringUtils.isBlank(renameFile)) {
      renameFile = UUID.randomUUID().toString()+".json";
      JSONObject themePathObject = theme.getJSONObject(MESS_THEME); 
      themePathObject.put(MESS_FIELD_NAME, renameFile);
    }
    
    // 查询数据库中可能存在的样式记录信息
    String projectName = platformContext.getAppName();
    List<SystemThemeEntity> themes = findByProjectName(projectName);

    Validate.isTrue(themes == null || themes.size() <= 1,"系统主题信息有多条，请检查!!");
    SystemThemeEntity entity;
    Date now = new Date();
    if(themes == null || themes.isEmpty()){
      //目前包含：logo，色板，布局框架，列表等
      this.saveThemeFile(renameFile, relativePath, theme);

      entity = new SystemThemeEntity();
      entity.setCommitUser(account);
      entity.setUpdateUser(account);
      entity.setCreateTime(now);
      entity.setUpdateTime(now);
      entity.setProjectName(projectName);
      entity.setFileName(renameFile);
      entity.setOriginalName(THEME_FILE_NAME);
      entity.setRelativeLocal(relativePath);
    } else {
      //保证数据库只有一条数据
      entity = themes.get(0);
      Validate.notNull(entity,"根据主题信息，未能查询到主题相关信息，请检查!!");
      entity.setUpdateTime(now);
      entity.setUpdateUser(account);
      entity.setFileName(renameFile);
      entity.setOriginalName(THEME_FILE_NAME);
      entity.setRelativeLocal(relativePath);
      this.saveThemeFile(renameFile, relativePath, theme);
    }
    systemThemeRepository.save(entity);
    return entity;
  }

  /**
   * 根据项目名查询主题
   * @param projectName
   * @return
   */
  private List<SystemThemeEntity> findByProjectName(String projectName) {
    if(StringUtils.isNotBlank(projectName)) {
      return systemThemeRepository.findByProjectName(projectName);
    } else {
      return systemThemeRepository.findByBlankProjectName();
    }
  }

  @Override
  public JSONObject findTheme() {
    //1.======查询主题信息
    String projectName = platformContext.getAppName();
    List<SystemThemeEntity> themes = this.findByProjectName(projectName);
    if(CollectionUtils.isEmpty(themes)){
      return null;
    }
    Validate.isTrue(themes.size() <= 1,"系统主题信息有多条，请检查!!");
    SystemThemeEntity theme = themes.get(0);

    //2.======解析主题数据，返回json格式
    byte[] content = venusFileService.readFileContent(theme.getRelativeLocal(),theme.getFileName());
    if(ArrayUtils.isEmpty(content)) {
      LOGGER.warn("未能发现系统主题信息数据，请检查!!");
      return null;
    }
    String jsonStr = new String(content, StandardCharsets.UTF_8);
    return JSON.parseObject(jsonStr);
  }

  /**
   * 验证用户信息
   * @param principal
   */
  private String validatePrincipal(Principal principal){
    Validate.notNull(principal,"未能获取到用户信息，请检查!!");
    String account = principal.getName();
    Validate.notBlank(account,"未能获取到用户账户信息，请检查!!");
    return account;
  }

  /**
   * 解析前端传入的theme配置路径信息
   * @return
   */
  private Map<String,String> resolveThemePath(JSONObject theme){
    Validate.notNull(theme,"系统主题数据不能为空，请检查!!");
    Map<String,String> themePathMap = new HashMap<>();
    JSONObject themeJson = theme.getJSONObject(MESS_THEME);
    if(themeJson == null || themeJson.size() == 0){
      return themePathMap;
    }
    String relativeLocal = themeJson.getString(MESS_RELATIVE_LOCAL);
    String fileName = themeJson.getString(MESS_FIELD_NAME);
    Validate.notBlank(relativeLocal,"系统主题中，给定的相对路径不能为空，请检查!!");
    Validate.notBlank(fileName,"系统主题中，给定的文件名不能为空，请检查!!");
    themePathMap.put(MESS_FIELD_NAME,fileName);
    themePathMap.put(MESS_RELATIVE_LOCAL,relativeLocal);
    return themePathMap;
  }

  /**
   * 保存主题文件到文件系统中
   * @param renameFile
   * @param relativePath
   * @param theme
   */
  private void saveThemeFile(String renameFile , String relativePath ,JSONObject theme){
    Validate.notEmpty(theme,"传入的主题json数据不能为空，请检查!!");
    //目前包含：logo，色板，布局框架，列表等
    byte[] content;
    try{
      content = theme.toJSONString().getBytes("utf-8");
    }catch (Exception e){
      throw new IllegalArgumentException("系统主题数据转码失败，请重试!");
    }
    venusFileService.saveFile(relativePath,THEME_FILE_NAME, renameFile,content);
  }

  /**
   * 导出主题信息
   * @return
   */
  public byte[] export(){
    List<SystemThemeEntity> themes = this.findByProjectName(platformContext.getAppName());
    Validate.isTrue(themes.size() == 1,"获取系统主题信息出错，请检查!!");
    SystemThemeEntity theme = themes.get(0);
    byte[] fullZipBytes;
    try (ByteArrayOutputStream fullZipOutputStream = new ByteArrayOutputStream();
         ZipOutputStream zipOutputStream = new ZipOutputStream(fullZipOutputStream)) {
      this.handleTheme(zipOutputStream, theme);
      zipOutputStream.finish();
      fullZipBytes = fullZipOutputStream.toByteArray();
    } catch(IOException e) {
      throw new IllegalArgumentException(e.getMessage());
    }
    return fullZipBytes;
  }

  /**
   * 处理主题导出的过程，主要将文件打成一个压缩包
   * @param zipf
   * @param theme
   */
  private void handleTheme(ZipOutputStream zipf, SystemThemeEntity theme) {
    Validate.notNull(theme, "主题信息不能为空，请检查");
    String fileName = theme.getFileName();
    Validate.notBlank(fileName, "未找到主题的文件名，请检查");
    String relativeLocal = theme.getRelativeLocal();
    Validate.notBlank(relativeLocal, "未找到主题的相对路径，请检查");

    byte[] themeByte = venusFileService.readFileContent(relativeLocal, fileName);
    Validate.isTrue(themeByte != null && themeByte.length != 0, "theme.json文件内容不能为空");
    String jsonStr = new String(themeByte, StandardCharsets.UTF_8);
    Object logo = JSON.parseObject(jsonStr).get("logo");
    Object systemIcon = JSON.parseObject(jsonStr).get("system_icon");
    Object smallLogo = JSON.parseObject(jsonStr).get("small_logo");

    try {
      if(logo != null && !(UNDEFINED.equals(logo))){
        String logoRelativeLocal = this.getRelativeLocal(logo.toString());
        String logoFileName = this.getFileName(logo.toString());
        byte[] logoByte = venusFileService.readFileContent(logoRelativeLocal, logoFileName);
        Validate.isTrue(logoByte != null && logoByte.length != 0, "logo文件内容不能为空");
        this.writeZipFile(zipf, "logo.png", logoByte);
      }

      if(systemIcon != null && !(UNDEFINED.equals(systemIcon))){
        String systemIconRelativeLocal = this.getRelativeLocal(systemIcon.toString());
        String systemIconFileName = this.getFileName(systemIcon.toString());
        byte[] systemIconByte = venusFileService.readFileContent(systemIconRelativeLocal, systemIconFileName);
        Validate.isTrue(systemIconByte != null && systemIconByte.length != 0, "icon文件内容不能为空");
        this.writeZipFile(zipf, "system_icon.png", systemIconByte);
      }

      if(smallLogo != null && !(UNDEFINED.equals(smallLogo))){
        String smallLogoRelativeLocal = this.getRelativeLocal(smallLogo.toString());
        String smallLogoFileName = this.getFileName(smallLogo.toString());
        byte[] smallLogoByte = venusFileService.readFileContent(smallLogoRelativeLocal, smallLogoFileName);
        Validate.isTrue(smallLogoByte != null && smallLogoByte.length != 0, "small_logo文件内容不能为空");
        this.writeZipFile(zipf, "small_logo.png", smallLogoByte);
      }

      this.writeZipFile(zipf, "theme.json", themeByte);
    } catch (IOException e) {
      LOGGER.error(e.getMessage() , e);
      throw new IllegalArgumentException(e.getMessage());
    }
  }

  /**
   * 获取文件名称
   * @param str
   * @return
   */
  private String getFileName(String str) {
    Validate.notBlank(str, "文件字符串不能为空");
    return str.substring(str.lastIndexOf("/")+1);
  }

  /**
   * 获取文件相对路径
   * @param str
   * @return
   */
  private String getRelativeLocal(String str) {
    Validate.notBlank(str, "文件字符串不能为空");
    return str.substring(0,str.lastIndexOf("/"));
  }

  /**
   * 将指定的二进制内容，写入zip压缩包中对应的文件路径下
   * @param zipf zip压缩流
   * @param zipFileName zip压缩包中文件的相对路径
   * @param bosBytes 写入的压缩内容
   * @throws IOException
   */
  private void writeZipFile(ZipOutputStream zipf , String zipFileName , byte[] bosBytes) throws IOException {
    ZipEntry zipEntry = new ZipEntry(zipFileName);
    zipf.putNextEntry(zipEntry);
    int maxLen = 9060;
    byte[] fileContents = new byte[maxLen];
    int realLen;
    try (ByteArrayInputStream bis = new ByteArrayInputStream(bosBytes);) {
      while((realLen = bis.read(fileContents, 0, maxLen)) != -1) {
        zipf.write(fileContents, 0, realLen);
      }
    }
  }

  /**
   * 导入主题信息
   * @param currentUser
   * @param file
   * @return
   */
  public SystemThemeEntity importTheme(Principal currentUser, MultipartFile file) {
    Validate.notNull(file, "上传文件不能为空！");
    Validate.notNull(currentUser,"当前登录信息不能为空！");
    SystemThemeEntity themeEntity = null;
    try {
      File tempFile = new File(file.getName());
      FileUtils.copyInputStreamToFile(file.getInputStream(), tempFile);
      ZipFile zipFile = new ZipFile(tempFile);

      ZipEntry theme = zipFile.getEntry("theme.json");
      ZipEntry logo = zipFile.getEntry("logo.png");
      ZipEntry icon = zipFile.getEntry("system_icon.png");
      ZipEntry smallLogo = zipFile.getEntry("small_logo.png");

      String folderName = new SimpleDateFormat("yyyyMMdd").format(new Date());

      String logoStr = "";
      if(logo != null){
        byte[] bytes = this.zipEntryToByte(zipFile, logo);
        String logoRelativePath = fileRelativeTemplate.generatRelativePath("logo", null);
        String logoName = UUID.randomUUID().toString() + ".png";
        logoStr = logoRelativePath + "/" + logoName;
        venusFileService.saveFile(logoRelativePath, "logo.png", logoName, bytes);
      }
      String iconStr = "";
      if(icon != null){
        byte[] bytes = this.zipEntryToByte(zipFile, icon);
        String iconRelativePath = fileRelativeTemplate.generatRelativePath("system_icon", null);
        String iconName = UUID.randomUUID().toString() + ".png";
        iconStr = iconRelativePath + "/" + iconName;
        venusFileService.saveFile(iconRelativePath, "system_icon.png", iconName, bytes);
      }
      String smallLogoStr = "";
      if(smallLogo != null){
        byte[] bytes = this.zipEntryToByte(zipFile, smallLogo);
        String smallLogoRelativePath = fileRelativeTemplate.generatRelativePath("small_logo", null);
        String smallLogoName = UUID.randomUUID().toString() + ".png";
        smallLogoStr = smallLogoRelativePath + "/" + smallLogoName;
        venusFileService.saveFile(smallLogoRelativePath, "small_logo.png", smallLogoName, bytes);
      }
      if (theme != null) {
        byte[] bytes = this.zipEntryToByte(zipFile, theme);
        //修改文件
        String jsonStr = new String(bytes, StandardCharsets.UTF_8);
        JSONObject jsonObject = JSON.parseObject(jsonStr);
        if(StringUtils.isNotBlank(logoStr)){
          jsonObject.put("logo", logoStr);
        }
        if(StringUtils.isNotBlank(iconStr)){
          jsonObject.put("system_icon", iconStr);
        }
        if(StringUtils.isNotBlank(smallLogoStr)){
          jsonObject.put("small_logo", smallLogoStr);
        }

        String themeRelativePath = "/theme/" + folderName;
        String themeName = UUID.randomUUID().toString() + ".json";
        String originalName = "theme.json";

        JSONObject object = (JSONObject) JSON.toJSON(jsonObject.get("theme"));
        object.put("fileName", themeName);
        object.put("relativeLocal", themeRelativePath);
        byte[] rbytes = jsonObject.toString().getBytes();
        venusFileService.saveFile(themeRelativePath, originalName, themeName, rbytes);

        //1.======查询主题信息
        String projectName = platformContext.getAppName();
        List<SystemThemeEntity> themes = this.findByProjectName(projectName);
        Validate.isTrue(themes.size() <= 1,"获取系统主题信息出错，请检查!!");
        if(themes.size() == 1){
          themeEntity = themes.get(0);
        }else {
          themeEntity = new SystemThemeEntity();
          themeEntity.setCommitUser(currentUser.getName());
          themeEntity.setCreateTime(new Date());
          themeEntity.setProjectName(projectName);
        }
        themeEntity.setFileName(themeName);
        themeEntity.setOriginalName(originalName);
        themeEntity.setRelativeLocal(themeRelativePath);
        themeEntity.setUpdateTime(new Date());
        themeEntity.setUpdateUser(currentUser.getName());
        systemThemeRepository.save(themeEntity);
      }
      return themeEntity;
    } catch (IOException e) {
      LOGGER.error(e.getMessage(), e);
      throw new IllegalArgumentException(e.getMessage());
    }
  }

  /**
   * 将ZipEntry转成byte[]
   * @param zipFile
   * @param zipEntry
   * @return
   * @throws IOException
   */
  private byte[] zipEntryToByte(ZipFile zipFile, ZipEntry zipEntry) throws IOException {
    InputStream is = zipFile.getInputStream(zipEntry);
    ByteArrayOutputStream output = new ByteArrayOutputStream();
    byte[] buffer = new byte[1024 * 4];
    int n = 0;
    while (-1 != (n = is.read(buffer))) {
      output.write(buffer, 0, n);
    }
    return output.toByteArray();
  }
}
