package com.bizunited.platform.core.service.invoke;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;

import org.apache.commons.lang3.Validate;

import com.bizunited.platform.core.service.invoke.binding.InvokeRequestHandleBinding;
import com.bizunited.platform.core.service.invoke.binding.InvokeResponseHandleBinding;

/**
 * InvokeProxy的构建器，定义创建InvokeProxy时，那些具体的设定信息
 * @author yinwenjie
 */
public class InvokeProxyBuilder {
  /**
   * 这个集合保证描述一个InvokeRequestHandle和InvokeRequestHandle class被添加到构建器中的顺序
   * 先添加的heandle执行器将先被执行
   */
  private List<? super Object> invokeRequestHandleCredentials = new ArrayList<>(16);
  /**
   * 这个集合保证描述一个InvokeResponseHandle和InvokeResponseHandle class被添加到构建器中的顺序
   * 先添加的heandle执行器将先被执行
   */
  private List<? super Object> invokeResponseHandleCredentials = new ArrayList<>(16);
  /**
   * 在构建器验证过程中使用的类加载器，如果构建时没有传入，则采用当前线程的类加载器
   */
  private ClassLoader classLoader;
  /**
   * 该绑定器在进行InvokeProxy构件时，基于一个特定的InvokeRequestHandle类，进行InvokeRequestHandle对象的初始化
   */
  private InvokeRequestHandleBinding invokeRequestHandleBinding;
  /**
   * 该绑定器在进行InvokeProxy构件时，基于一个特定的InvokeResponseHandle类，进行InvokeResponseHandle对象的初始化
   */
  private InvokeResponseHandleBinding invokeResponseHandleBinding;
  /**
   * 注意：每次调用build方法，都将生成一个独立运作的InvokeProxy对象
   * @return
   */
  @SuppressWarnings("unchecked")
  public InvokeProxy build() {
    ClassLoader currentClassLoader = classLoader;
    if(currentClassLoader == null) {
      currentClassLoader = Thread.currentThread().getContextClassLoader();
    }
    if(invokeRequestHandleCredentials.isEmpty() && invokeResponseHandleCredentials.isEmpty()) {
      throw new IllegalArgumentException("没有发现调用中存在任何的Response Handle和Response Handle，请检查!!");
    }
    
    /*
     * 进行边界校验后，处理过程如下：
     * 1、根据invokeRequestHandleCredentials中的信息完成request handle对象的初始化和填充
     * 2、根据invokeResponseHandleCredentials中的信息完成response handle对象的初始化和填充
     * 3、进行上下文，特别是厨师参数的构建，现在只需要进行DefaultHandleChain的构建
     * */
    // 1、======
    List<InvokeRequestHandle> requestHandles = new ArrayList<>(16);
    for (Object credentialItem : invokeRequestHandleCredentials) {
      if(credentialItem instanceof InvokeRequestHandle) {
        requestHandles.add((InvokeRequestHandle)credentialItem);
      }
      if(credentialItem instanceof Class) {
        Validate.notNull(this.invokeRequestHandleBinding , "根据设定，需要进行InvokeRequestHandle Class的绑定操作，但是未发现任何invokeRequestHandleBinding对象信息，请检查!!");
        requestHandles.add(this.invokeRequestHandleBinding.binding((Class<InvokeRequestHandle>)credentialItem));
      }
    }
    // 2、======
    List<InvokeResponseHandle> responseHandles = new ArrayList<>(16);
    for (Object credentialItem : invokeResponseHandleCredentials) {
      if(credentialItem instanceof InvokeResponseHandle) {
        responseHandles.add((InvokeResponseHandle)credentialItem);
      }
      if(credentialItem instanceof Class) {
        Validate.notNull(this.invokeResponseHandleBinding , "根据设定，需要进行InvokeResponseHandle Class的绑定操作，但是未发现任何invokeResponseHandleBinding对象信息，请检查!!");
        responseHandles.add(this.invokeResponseHandleBinding.binding((Class<InvokeResponseHandle>)credentialItem));
      }
    }
    // 3、=======
    DefaultHandleChain defaultHandleChain = new DefaultHandleChain();
    defaultHandleChain.setInvokeRequestHandles(requestHandles);
    defaultHandleChain.setInvokeResponseHandles(responseHandles);
    InvokeProxy invokeProxy = new InvokeProxy();
    invokeProxy.setTargetHandleChain(defaultHandleChain);
    return invokeProxy;
  }
  
  /**
   * 通过该方法，可将指定构建器中已设置的相关信息复制到当前构建器对象中
   * @param source
   * @return
   */
  public InvokeProxyBuilder copy(InvokeProxyBuilder source) {
    if(source.invokeRequestHandleCredentials != null && source.invokeRequestHandleCredentials.size() > 0) {
      this.invokeRequestHandleCredentials.addAll(source.invokeRequestHandleCredentials);
    }
    if(source.invokeResponseHandleCredentials != null && source.invokeResponseHandleCredentials.size() > 0) {
      this.invokeResponseHandleCredentials.addAll(source.invokeResponseHandleCredentials);
    }
    return this;
  }
  /**
   * 设置类加载器
   * @param classLoader 
   * @return 
   */
  public InvokeProxyBuilder addClassLoader(ClassLoader classLoader) {
    this.classLoader = classLoader;
    return this;
  }
  /**
   * 添加response结果处理链
   * @param responses 
   * @return 
   */
  public InvokeProxyBuilder addInvokeResponseFilter(InvokeResponseHandle ...responses) {
    Arrays.stream(responses).forEach(filter -> invokeResponseHandleCredentials.add(filter));
    return this;
  }
  /**
   * 添加response结果处理链(以处理器类型的方式进行添加)
   * @param responses 
   * @return 
   */
  @SuppressWarnings("unchecked")
  public InvokeProxyBuilder addInvokeResponseTypeFilter(Class<? extends InvokeResponseHandle> ...responseTypes) {
    Arrays.stream(responseTypes).forEach(filter -> invokeResponseHandleCredentials.add(filter));
    return this;
  }
  /**
   * 添加request请求处理链
   * @param requests 
   * @return
   */
  public InvokeProxyBuilder addInvokeRequestFilter(InvokeRequestHandle ...requests) {
    Arrays.stream(requests).forEach(filter -> invokeRequestHandleCredentials.add(filter));
    return this;
  }
  /**
   * 添加request请求处理链
   * @param requests 
   * @return
   */
  @SuppressWarnings("unchecked")
  public InvokeProxyBuilder addInvokeRequestTypeFilter(Class<? extends InvokeRequestHandle> ...requestTypes) {
    Arrays.stream(requestTypes).forEach(filter -> invokeRequestHandleCredentials.add(filter));
    return this;
  }
  /**
   * 如果在构建链式调用代理器时，发现有通过request handle类进行初始化的情况，就会使用InvokeRequestHandleBinding根据类信息进行request handle对象的初始化
   */
  public InvokeProxyBuilder setInvokeRequestHandleBinding(InvokeRequestHandleBinding invokeRequestHandleBinding) {
    this.invokeRequestHandleBinding = invokeRequestHandleBinding;
    return this;
  }
  /**
   * 如果在构建链式调用代理器时，发现有通过reponse handle类进行初始化的情况，就会使用InvokeResponseHandleBinding根据类信息进行reponse handle对象的初始化
   * @param invokeResponseHandleBinding
   */
  public InvokeProxyBuilder setInvokeResponseHandleBinding(InvokeResponseHandleBinding invokeResponseHandleBinding) {
    this.invokeResponseHandleBinding = invokeResponseHandleBinding;
    return this;
  }
}