package com.bizunited.platform.core.service.log;


import com.bizunited.platform.common.service.NebulaToolkitService;
import com.bizunited.platform.core.entity.log.LoggerTemplateEntity;
import com.bizunited.platform.core.repository.log.LoggerTemplateEntityRepository;
import com.bizunited.platform.core.service.LoggerTemplateEntityService;

import com.bizunited.platform.rbac.server.util.SecurityUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.Pageable;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;

import javax.transaction.Transactional;
import java.util.*;
import java.util.Map.Entry;

/**
 * LoggerTemplateEntity业务模型的服务层接口实现
 * @author saturn
 */
@Service
public class LoggerTemplateEntityServiceImpl implements LoggerTemplateEntityService { 
  /**
   * 在本地的日志模板使用的缓存
   */
  public Map<String, LoggerTemplateEntity> localCache = new HashMap<>();

  private Map<String,LoggerTemplateEntity>  codeCache = new HashMap<>();


  private NebulaToolkitService nebulaToolkitService;

  private LoggerTemplateEntityRepository loggerTemplateEntityRepository;

  @Autowired
  public LoggerTemplateEntityServiceImpl(NebulaToolkitService nebulaToolkitService,LoggerTemplateEntityRepository loggerTemplateEntityRepository){
    this.nebulaToolkitService  = nebulaToolkitService;
    this.loggerTemplateEntityRepository = loggerTemplateEntityRepository;
    List<LoggerTemplateEntity>  list = this.loggerTemplateEntityRepository.findAll();
    list.parallelStream().filter(t->Objects.nonNull(t)&&StringUtils.isNotEmpty(t.getMethodName()))
            .forEach(t->{
              this.localCache.put(t.getMethodName(),t);
            });
  }

  @Transactional
  @Override
  public LoggerTemplateEntity create(LoggerTemplateEntity loggerTemplateEntity) {
    LoggerTemplateEntity current = this.createForm(loggerTemplateEntity);
    //==================================================== 
    //    这里可以处理第三方系统调用（或特殊处理过程）
    //====================================================
    this.localCache.put(loggerTemplateEntity.getMethodName(),loggerTemplateEntity);
    return current;
  } 
  @Transactional
  @Override
  public LoggerTemplateEntity createForm(LoggerTemplateEntity loggerTemplateEntity) {
   /* 
    * 针对1.1.3版本的需求，这个对静态模型的保存操作做出调整，新的包裹过程为：
    * 1、如果当前模型对象不是主模型
    * 1.1、那么创建前只会验证基本信息，直接的ManyToOne关联（单选）和ManyToMany关联（多选）
    * 1.2、验证完成后，也只会保存当前对象的基本信息，直接的单选
    * TODO 1.3、ManyToMany的关联（多选），暂时需要开发人员自行处理
    * 2、如果当前模型对象是主业务模型
    *  2.1、创建前会验证当前模型的基本属性，单选和多选属性
    *  2.2、然后还会验证当前模型关联的各个OneToMany明细信息，调用明细对象的服务，明每一条既有明细进行验证
    *  （2.2的步骤还需要注意，如果当前被验证的关联对象是回溯对象，则不需要验证了）
    * 2.3、还会验证当前模型关联的各个OneToOne分组，调用分组对象的服务，对分组中的信息进行验证
    *   2.3.1、包括验证每一个分组项的基本信息、直接的单选、多选信息
    *   2.3.2、以及验证每个分组的OneToMany明细信息
    * */
    //设置创建时间与账号，更新时间与账号
    Date date = new Date();
    String userAccount = SecurityUtils.getUserAccount();
    loggerTemplateEntity.setCreateTime(date);
    loggerTemplateEntity.setCreateAccount(userAccount);
    loggerTemplateEntity.setModifyTime(date);
    loggerTemplateEntity.setModifyAccount(userAccount);
    this.createValidation(loggerTemplateEntity);
    
    // ===============================
    //  和业务有关的验证填写在这个区域    
    // ===============================
    
    this.loggerTemplateEntityRepository.save(loggerTemplateEntity);
    // 返回最终处理的结果，里面带有详细的关联信息
    return loggerTemplateEntity;
  }
  /**
   * 在创建一个新的LoggerTemplateEntity模型对象之前，检查对象各属性的正确性，其主键属性必须没有值
   */
  private void createValidation(LoggerTemplateEntity loggerTemplateEntity) {
    Validate.notNull(loggerTemplateEntity , "进行当前操作时，信息对象必须传入!!");
    // 判定那些不能为null的输入值：条件为 caninsert = true，且nullable = false
    Validate.isTrue(StringUtils.isBlank(loggerTemplateEntity.getId()), "添加信息时，当期信息的数据编号（主键）不能有值！");
    loggerTemplateEntity.setId(null);
    Validate.notNull(loggerTemplateEntity.getName(), "添加信息时，模板名称不能为空！");
    Validate.notBlank(loggerTemplateEntity.getCode(), "添加信息时，模板编号，唯一的编号，必须有不能为空！");
    Validate.notBlank(loggerTemplateEntity.getType(), "添加信息时，操作类型，可以自行添加，但是只能是英文不能为空！");
    Validate.notBlank(loggerTemplateEntity.getModule(), "添加信息时，业务模块，可以随便编写，但必须是英文不能为空！");
    Validate.notBlank(loggerTemplateEntity.getMethodName(), "添加信息时，完整的被拦截的方法名不能为空！");
    Validate.notNull(loggerTemplateEntity.getMethodParamIndex(), "添加信息时，查询数据详情时，使用该方法的第几个参数作为入参不能为空！");
    Validate.notBlank(loggerTemplateEntity.getExpression(), "添加信息时，日志信息表达式不能为空！");
    Validate.notNull(loggerTemplateEntity.getCompare(), "添加信息时，该日志模板是否需要进行明细比较不能为空！");
    Validate.notNull(loggerTemplateEntity.getState(), "添加信息时，状态不能为空！");
    // 验证长度，被验证的这些字段符合特征: 字段类型为String，且不为PK （注意连续空字符串的情况）
    Validate.isTrue(loggerTemplateEntity.getName() == null || loggerTemplateEntity.getName().length() < 255 , "模板名称,在进行添加时填入值超过了限定长度(255)，请检查!");
    Validate.isTrue(loggerTemplateEntity.getCode() == null || loggerTemplateEntity.getCode().length() < 255 , "模板编号，唯一的编号，必须有,在进行添加时填入值超过了限定长度(255)，请检查!");
    Validate.isTrue(loggerTemplateEntity.getType() == null || loggerTemplateEntity.getType().length() < 255 , "操作类型，可以自行添加，但是只能是英文,在进行添加时填入值超过了限定长度(255)，请检查!");
    Validate.isTrue(loggerTemplateEntity.getModule() == null || loggerTemplateEntity.getModule().length() < 255 , "业务模块，可以随便编写，但必须是英文,在进行添加时填入值超过了限定长度(255)，请检查!");
    Validate.isTrue(loggerTemplateEntity.getMethodName() == null || loggerTemplateEntity.getMethodName().length() < 255 , "完整的被拦截的方法名,在进行添加时填入值超过了限定长度(255)，请检查!");
    Validate.isTrue(loggerTemplateEntity.getExpression() == null || loggerTemplateEntity.getExpression().length() < 255 , "日志信息表达式,在进行添加时填入值超过了限定长度(255)，请检查!");
    Validate.isTrue(loggerTemplateEntity.getPreQueryComponent() == null || loggerTemplateEntity.getPreQueryComponent().length() < 255 , "进行预查询的bean组件信息,在进行添加时填入值超过了限定长度(255)，请检查!");
    Validate.isTrue(loggerTemplateEntity.getPreQueryMethod() == null || loggerTemplateEntity.getPreQueryMethod().length() < 255 , "进行预查询的bean组件的方法名,在进行添加时填入值超过了限定长度(255)，请检查!");
    Validate.isTrue(loggerTemplateEntity.getPreReturnFilter() == null || loggerTemplateEntity.getPreReturnFilter().length() < 255 , "预查询时，可以使用该属性指定这个业务对象的哪些直接或间接属性需要进行返回,在进行添加时填入值超过了限定长度(255)，请检查!");
    Validate.isTrue(loggerTemplateEntity.getLastQueryComponent() == null || loggerTemplateEntity.getLastQueryComponent().length() < 255 , "进行后置查询的bean组件信息（一般为bean组件名）,在进行添加时填入值超过了限定长度(255)，请检查!");
    Validate.isTrue(loggerTemplateEntity.getLastQueryMethod() == null || loggerTemplateEntity.getLastQueryMethod().length() < 255 , "进行后置查询的bean组件的方法名,在进行添加时填入值超过了限定长度(255)，请检查!");
    Validate.isTrue(loggerTemplateEntity.getLastReturnFilter() == null || loggerTemplateEntity.getLastReturnFilter().length() < 255 , "后置查询时，可以使用该属性指定这个业务对象的哪些直接或间接属性需要进行返回,在进行添加时填入值超过了限定长度(255)，请检查!");
    LoggerTemplateEntity currentLoggerTemplateEntity = this.findByCode(loggerTemplateEntity.getCode());
    Validate.isTrue(currentLoggerTemplateEntity == null, "模板编号，已存在,请检查");
    currentLoggerTemplateEntity = this.findByMethodName(loggerTemplateEntity.getMethodName());
    Validate.isTrue(currentLoggerTemplateEntity == null, "完整的被拦截的方法名已存在,请检查");
  }
  @Transactional
  @Override
  public LoggerTemplateEntity update(LoggerTemplateEntity loggerTemplateEntity) {
    LoggerTemplateEntity current = this.updateForm(loggerTemplateEntity);
    //==================================================== 
    //    这里可以处理第三方系统调用（或特殊处理过程）
    //====================================================
    this.localCache.put(loggerTemplateEntity.getMethodName(),loggerTemplateEntity);
    return current;
  } 
  @Transactional
  @Override
  public LoggerTemplateEntity updateForm(LoggerTemplateEntity loggerTemplateEntity) {
    /* 
     * 针对1.1.3版本的需求，这个对静态模型的修改操作做出调整，新的过程为：
     * 1、如果当前模型对象不是主模型
     * 1.1、那么创建前只会验证基本信息，直接的ManyToOne关联（单选）和ManyToMany关联（多选）
     * 1.2、验证完成后，也只会保存当前对象的基本信息，直接的单选
     * TODO 1.3、ManyToMany的关联（多选），暂时需要开发人员自行处理（求删除、新增绑定的代码已生成）
     * 
     * 2、如果当前模型对象是主业务模型
     *  2.1、创建前会验证当前模型的基本属性，单选和多选属性
     *  2.2、然后还会验证当前模型关联的各个OneToMany明细信息，调用明细对象的服务，明每一条既有明细进行验证
     *  （2.2的步骤还需要注意，如果当前被验证的关联对象是回溯对象，则不需要验证了）
     *  2.3、还会验证当前模型关联的各个OneToOne分组，调用分组对象的服务，对分组中的信息进行验证
     *    2.3.1、包括验证每一个分组项的基本信息、直接的单选、多选信息
     *    2.3.2、以及验证每个分组的OneToMany明细信息
     * */
    
    this.updateValidation(loggerTemplateEntity);
    // ===================基本信息
    String currentId = loggerTemplateEntity.getId();
    Optional<LoggerTemplateEntity> op_currentLoggerTemplateEntity = this.loggerTemplateEntityRepository.findById(currentId);
    LoggerTemplateEntity currentLoggerTemplateEntity = op_currentLoggerTemplateEntity.orElse(null);
    currentLoggerTemplateEntity = Validate.notNull(currentLoggerTemplateEntity ,"未发现指定的原始模型对象信");
    //更新更新时间与账号
    currentLoggerTemplateEntity.setModifyAccount(SecurityUtils.getUserAccount());
    currentLoggerTemplateEntity.setModifyTime(new Date());
    // 开始重新赋值——一般属性
    currentLoggerTemplateEntity.setName(loggerTemplateEntity.getName());
    currentLoggerTemplateEntity.setCode(loggerTemplateEntity.getCode());
    currentLoggerTemplateEntity.setType(loggerTemplateEntity.getType());
    currentLoggerTemplateEntity.setModule(loggerTemplateEntity.getModule());
    currentLoggerTemplateEntity.setMethodName(loggerTemplateEntity.getMethodName());
    currentLoggerTemplateEntity.setMethodParamIndex(loggerTemplateEntity.getMethodParamIndex());
    currentLoggerTemplateEntity.setMethodParamProperty(loggerTemplateEntity.getMethodParamProperty());
    currentLoggerTemplateEntity.setExpression(loggerTemplateEntity.getExpression());
    currentLoggerTemplateEntity.setCompare(loggerTemplateEntity.getCompare());
    currentLoggerTemplateEntity.setPreQueryComponent(loggerTemplateEntity.getPreQueryComponent());
    currentLoggerTemplateEntity.setPreQueryMethod(loggerTemplateEntity.getPreQueryMethod());
    currentLoggerTemplateEntity.setPreReturnFilter(loggerTemplateEntity.getPreReturnFilter());
    currentLoggerTemplateEntity.setLastQueryComponent(loggerTemplateEntity.getLastQueryComponent());
    currentLoggerTemplateEntity.setLastQueryMethod(loggerTemplateEntity.getLastQueryMethod());
    currentLoggerTemplateEntity.setLastReturnFilter(loggerTemplateEntity.getLastReturnFilter());
    currentLoggerTemplateEntity.setState(loggerTemplateEntity.getState());
    
    this.loggerTemplateEntityRepository.saveAndFlush(currentLoggerTemplateEntity);
    return currentLoggerTemplateEntity;
  }
  /**
   * 在更新一个已有的LoggerTemplateEntity模型对象之前，该私有方法检查对象各属性的正确性，其id属性必须有值
   */
  private void updateValidation(LoggerTemplateEntity loggerTemplateEntity) {
    Validate.isTrue(!StringUtils.isBlank(loggerTemplateEntity.getId()), "修改信息时，当期信息的数据编号（主键）必须有值！");
    
    // 基础信息判断，基本属性，需要满足not null
    Validate.notNull(loggerTemplateEntity.getName(), "修改信息时，模板名称不能为空！");
    Validate.notBlank(loggerTemplateEntity.getCode(), "修改信息时，模板编号，唯一的编号，必须有不能为空！");
    Validate.notBlank(loggerTemplateEntity.getType(), "修改信息时，操作类型，可以自行添加，但是只能是英文不能为空！");
    Validate.notBlank(loggerTemplateEntity.getModule(), "修改信息时，业务模块，可以随便编写，但必须是英文不能为空！");
    Validate.notBlank(loggerTemplateEntity.getMethodName(), "修改信息时，完整的被拦截的方法名不能为空！");
    Validate.notNull(loggerTemplateEntity.getMethodParamIndex(), "修改信息时，查询数据详情时，使用该方法的第几个参数作为入参不能为空！");
    Validate.notBlank(loggerTemplateEntity.getExpression(), "修改信息时，日志信息表达式不能为空！");
    Validate.notNull(loggerTemplateEntity.getCompare(), "修改信息时，该日志模板是否需要进行明细比较不能为空！");
    Validate.notNull(loggerTemplateEntity.getState(), "修改信息时，状态不能为空！");
    Validate.isTrue(loggerTemplateEntity.getName() == null || loggerTemplateEntity.getName().length() < 255 , "模板名称,在进行修改时填入值超过了限定长度(255)，请检查!");
    Validate.isTrue(loggerTemplateEntity.getCode() == null || loggerTemplateEntity.getCode().length() < 255 , "模板编号，唯一的编号，必须有,在进行修改时填入值超过了限定长度(255)，请检查!");
    Validate.isTrue(loggerTemplateEntity.getType() == null || loggerTemplateEntity.getType().length() < 255 , "操作类型，可以自行添加，但是只能是英文,在进行修改时填入值超过了限定长度(255)，请检查!");
    Validate.isTrue(loggerTemplateEntity.getModule() == null || loggerTemplateEntity.getModule().length() < 255 , "业务模块，可以随便编写，但必须是英文,在进行修改时填入值超过了限定长度(255)，请检查!");
    Validate.isTrue(loggerTemplateEntity.getMethodName() == null || loggerTemplateEntity.getMethodName().length() < 255 , "完整的被拦截的方法名,在进行修改时填入值超过了限定长度(255)，请检查!");
    Validate.isTrue(loggerTemplateEntity.getExpression() == null || loggerTemplateEntity.getExpression().length() < 255 , "日志信息表达式,在进行修改时填入值超过了限定长度(255)，请检查!");
    Validate.isTrue(loggerTemplateEntity.getPreQueryComponent() == null || loggerTemplateEntity.getPreQueryComponent().length() < 255 , "进行预查询的bean组件信息,在进行修改时填入值超过了限定长度(255)，请检查!");
    Validate.isTrue(loggerTemplateEntity.getPreQueryMethod() == null || loggerTemplateEntity.getPreQueryMethod().length() < 255 , "进行预查询的bean组件的方法名,在进行修改时填入值超过了限定长度(255)，请检查!");
    Validate.isTrue(loggerTemplateEntity.getPreReturnFilter() == null || loggerTemplateEntity.getPreReturnFilter().length() < 255 , "预查询时，可以使用该属性指定这个业务对象的哪些直接或间接属性需要进行返回,在进行修改时填入值超过了限定长度(255)，请检查!");
    Validate.isTrue(loggerTemplateEntity.getLastQueryComponent() == null || loggerTemplateEntity.getLastQueryComponent().length() < 255 , "进行后置查询的bean组件信息（一般为bean组件名）,在进行修改时填入值超过了限定长度(255)，请检查!");
    Validate.isTrue(loggerTemplateEntity.getLastQueryMethod() == null || loggerTemplateEntity.getLastQueryMethod().length() < 255 , "进行后置查询的bean组件的方法名,在进行修改时填入值超过了限定长度(255)，请检查!");
    Validate.isTrue(loggerTemplateEntity.getLastReturnFilter() == null || loggerTemplateEntity.getLastReturnFilter().length() < 255 , "后置查询时，可以使用该属性指定这个业务对象的哪些直接或间接属性需要进行返回,在进行修改时填入值超过了限定长度(255)，请检查!");
    //判断code以及methodName是否改变，若改变则检测是否重复
    LoggerTemplateEntity currentForId = this.findById(loggerTemplateEntity.getId());
    //判断code值是否改变
    if(!StringUtils.equals(currentForId.getCode() , loggerTemplateEntity.getCode())){
      LoggerTemplateEntity currentForCode = this.findByCode(loggerTemplateEntity.getCode());
      Validate.isTrue(currentForCode==null,"修改信息时，填入的code已存在");
    }
    //判断methodName值是否改变
    if(!StringUtils.equals(currentForId.getMethodName() , loggerTemplateEntity.getMethodName())){
      LoggerTemplateEntity currentForMethodName = this.findByMethodName(loggerTemplateEntity.getMethodName());
      Validate.isTrue(currentForMethodName==null,"修改信息时，填入的methodName已存在");
    }
  } 
  @Override
  @Transactional
  public void deleteById(String id) {
    // 只有存在才进行删除
    Validate.notBlank(id , "进行删除时，必须给定主键信息!!");
    LoggerTemplateEntity current = this.findById(id);
    if(current != null) { 
      this.loggerTemplateEntityRepository.delete(current);
    }
  }
  @Override
  public LoggerTemplateEntity findById(String id) {
    // TODO Auto-generated method stub
    return loggerTemplateEntityRepository.findById(id).orElse(null);
  }
  @Override
  public LoggerTemplateEntity findDetailsById(final String id) {
    if(StringUtils.isBlank(id) || localCache.keySet().size()<1) {
      return null;
    }
    Set<Entry<String, LoggerTemplateEntity>> entrys = codeCache.entrySet();
    Entry<String, LoggerTemplateEntity> equalTarget = entrys.stream().filter(item -> StringUtils.equals(item.getValue().getId(), id)).findFirst().orElse(null);
    if(equalTarget == null) {
      return null;
    }
    return equalTarget.getValue();
  }

  @Override
  public LoggerTemplateEntity findByCode(String code) {
    if(StringUtils.isBlank(code)) {
      return null;
    }
    return this.codeCache.get(code);
  } 
  @Override
  public void cache() {
    this.codeCache.clear();
    List<LoggerTemplateEntity> results = this.loggerTemplateEntityRepository.findAll();
    if(CollectionUtils.isEmpty(results)) {
      return;
    }
    
    // 使用code作为存储标记，而且不使用持久化对象
    for (LoggerTemplateEntity loggerTemplate : results) {
      LoggerTemplateEntity target = nebulaToolkitService.copyObjectByWhiteList(loggerTemplate, LoggerTemplateEntity.class, HashSet.class, ArrayList.class);
      this.codeCache.put(loggerTemplate.getCode(), target);
    }
  }
  @Override
  public LoggerTemplateEntity findByMethodName(String methodName) {
    if(StringUtils.isBlank(methodName)) {
      return null;
    }

    Set<Entry<String, LoggerTemplateEntity>> entrys = this.localCache.entrySet();
    Entry<String, LoggerTemplateEntity> equalTarget = entrys.stream().filter(item -> StringUtils.equals(item.getValue().getMethodName(), methodName)).findFirst().orElse(null);
    if(equalTarget == null) {
      return null;
    }
    return equalTarget.getValue();
  }

  @Override
  public Page<LoggerTemplateEntity> findByConditions(LoggerTemplateEntity loggerTemplateEntity, Pageable pageable) {
    Validate.notNull(pageable, "查询分页信息不能为空!");
    Map<String,Object> conditions=new HashMap<>(5);
    String code=loggerTemplateEntity.getCode();
    if(StringUtils.isNotBlank(code)){
      conditions.put("code",code);
    }
    String name=loggerTemplateEntity.getName();
    if(StringUtils.isNotBlank(name)){
      conditions.put("name",name);
    }
    String methodName=loggerTemplateEntity.getMethodName();
    if(StringUtils.isNotBlank(methodName)){
      conditions.put("methodName",methodName);
    }
    Integer state=loggerTemplateEntity.getState();
    if(state!=null){
      conditions.put("state",state);
    }
    return this.loggerTemplateEntityRepository.findByConditions(pageable,conditions);
  }
} 
