package com.bizunited.platform.core.service.migrate.internal;

import com.bizunited.platform.common.enums.MigrateDataTypeEnum;
import com.bizunited.platform.common.enums.NormalStatusEnum;
import com.bizunited.platform.common.service.NebulaToolkitService;
import com.bizunited.platform.common.util.ZipFileUtils;
import com.bizunited.platform.core.common.PlatformContext;
import com.bizunited.platform.core.entity.CodeRuleEntity;
import com.bizunited.platform.core.entity.DataViewAuthEntity;
import com.bizunited.platform.core.entity.DataViewEntity;
import com.bizunited.platform.core.entity.DataViewGroupEntity;
import com.bizunited.platform.core.entity.EnvironmentVariableEntity;
import com.bizunited.platform.core.entity.MigrateExportEntity;
import com.bizunited.platform.core.entity.RemoteServiceAddressEntity;
import com.bizunited.platform.core.entity.RemoteServiceEntity;
import com.bizunited.platform.core.entity.ScriptEntity;
import com.bizunited.platform.core.repository.migrate.MigrateExportRepository;
import com.bizunited.platform.core.service.CodeRuleService;
import com.bizunited.platform.core.service.EnvironmentVariableService;
import com.bizunited.platform.core.service.RemoteServiceAddressService;
import com.bizunited.platform.core.service.ScriptService;
import com.bizunited.platform.core.service.dataview.DataViewAuthService;
import com.bizunited.platform.core.service.dataview.DataViewService;
import com.bizunited.platform.core.service.migrate.MigrateExportService;
import com.bizunited.platform.rbac.server.util.SecurityUtils;
import com.bizunited.platform.user.common.service.user.UserService;
import com.bizunited.platform.user.common.vo.UserVo;
import com.bizunited.platform.venus.common.service.file.VenusFileService;
import com.google.common.collect.Lists;
import com.google.common.collect.Sets;
import org.apache.commons.lang3.ArrayUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.ApplicationContext;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;

import javax.transaction.Transactional;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.ObjectOutputStream;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Date;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.zip.ZipOutputStream;

import static com.bizunited.platform.common.constant.MigrateDataConstants.CODE_RULE_FILENAME;
import static com.bizunited.platform.common.constant.MigrateDataConstants.DATA_VIEW_FILENAME;
import static com.bizunited.platform.common.constant.MigrateDataConstants.ENV_FILENAME;
import static com.bizunited.platform.common.constant.MigrateDataConstants.REMOTE_SERVICE_FILENAME;

/**
 * 基础工具套件中重要数据的迁出服务实现
 * @author yinwenjie
 */
@Service("migrateExportService")
public class MigrateExportServiceImpl implements MigrateExportService {
  private static final Logger LOGGER = LoggerFactory.getLogger(MigrateExportServiceImpl.class);
  @Autowired
  private MigrateExportRepository migrateExportRepository;
  @Autowired
  private ApplicationContext applicationContext;
  @Autowired
  private UserService userService;
  @Autowired
  private DataViewService dataViewService;
  @Autowired
  private VenusFileService venusFileService;
  @Autowired
  private DataViewAuthService dataViewAuthService;
  @Autowired
  private NebulaToolkitService nebulaToolkitService;
  @Autowired
  private CodeRuleService codeRuleService;
  @Autowired
  private EnvironmentVariableService environmentVariableService;
  @Autowired
  private ScriptService scriptService;
  @Autowired
  private PlatformContext platformContext;
  @Autowired
  private RemoteServiceAddressService remoteServiceAddressService;

  @Override
  @Transactional
  public MigrateExportEntity save(MigrateExportEntity export) {
    Validate.notNull(export.getDataType(), "数据类型不能为空");
    UserVo user = SecurityUtils.getCurrentUser();
    export.setCreateTime(new Date());
    export.setCreateUser(user);
    export.setCreator(user.getAccount());
    return migrateExportRepository.save(export);
  }

  @Override
  public List<MigrateExportEntity> findDetailsByDataType(Integer dataType) {
    if(dataType == null) {
      return Lists.newArrayList();
    }
    List<MigrateExportEntity> exports;
    String projectName = platformContext.getAppName();
    if(StringUtils.isNotBlank(projectName)){
      exports = migrateExportRepository.findByDataTypeAndProjectName(dataType, projectName);
    }else {
      exports = migrateExportRepository.findByDataTypeAndBlankProjectName(dataType);
    }
    if(CollectionUtils.isEmpty(exports)) {
      return Lists.newArrayList();
    }
    Map<String, UserVo> userCache = new HashMap<>();
    for (MigrateExportEntity export : exports) {
      String creator = export.getCreator();
      UserVo user = userCache.get(creator);
      if(user == null) {
        user = userService.findByAccount(creator);
        userCache.put(creator, user);
      }
      export.setCreateUser(user);
    }
    return exports;
  }

  /**
   * 保存导出日志
   * @param dataType
   * @param datas
   * @return
   */
  private MigrateExportEntity save(Integer dataType, String datas) {
    Validate.notNull(dataType, "数据类型不能为空");
    MigrateExportEntity export = new MigrateExportEntity();
    export.setDataType(dataType);
    export.setDatas(datas);
    export.setProjectName(platformContext.getAppName());
    MigrateExportService migrateExportService = applicationContext.getBean(MigrateExportService.class);
    return migrateExportService.save(export);
  }

  @Override
  public byte[] exportDataView(String[] ids) {
    /*
     * 处理过程
     * 1、根据传入的id数组查询需要导出的数据视图
     * 2、如果没有导出的数据视图,则返回空byte
     * 3、遍历处理数据视图
     *   3.1、用中度复制工具复制数据视图，包括数据视图的明细项
     *   3.2、处理数据视图的分组信息
     * 4、将数据视图对象信息输出到压缩文件中
     * */
    List<DataViewEntity> dataViews;
    if(ArrayUtils.isEmpty(ids)) {
      DataViewEntity example = new DataViewEntity();
      example.setTstatus(NormalStatusEnum.ENABLE.getStatus());
      dataViews = dataViewService.findAllByConditions(example);
    } else {
      int count = dataViewService.countByIds(ids);
      Validate.isTrue(count == ids.length, "指定导出的数据视图信息与数据库存储的信息不符，请检查");
      dataViews = dataViewService.findDetailsByIds(ids);
    }
    if(CollectionUtils.isEmpty(dataViews)) {
      return new byte[0];
    }
    List<DataViewEntity> exportDataViews = new ArrayList<>(dataViews.size());
    List<DataViewAuthEntity> exportAuths = new ArrayList<>();
    for (DataViewEntity dataView : dataViews) {
      DataViewEntity exportDataView = nebulaToolkitService.copyObjectByWhiteList(dataView, DataViewEntity.class, null, null,
          "dataSource", "fields", "filters", "filters.field", "systemFilters");
      this.handleDataViewGroup(exportDataView, dataView.getDataViewGroup());
      exportDataViews.add(exportDataView);
      Set<DataViewAuthEntity> auths = dataViewAuthService.findByDataView(dataView.getCode());
      if(!CollectionUtils.isEmpty(auths)) {
        Collection<DataViewAuthEntity> cpAuths = nebulaToolkitService.copyCollectionByWhiteList(auths, DataViewAuthEntity.class,
            DataViewAuthEntity.class, HashSet.class, ArrayList.class, "dataView", "horizontalAuths", "horizontalAuths.authRelations",
            "verticalAuths", "verticalAuths.authRelations", "interceptors");
        exportAuths.addAll(cpAuths);
      }
    }
    try (ByteArrayOutputStream baos = new ByteArrayOutputStream();
         ZipOutputStream zos = new ZipOutputStream(baos)) {
      try (ByteArrayOutputStream objBaos = new ByteArrayOutputStream();
           ObjectOutputStream oos = new ObjectOutputStream(objBaos)) {
        oos.writeObject(exportDataViews);
        oos.writeObject(exportAuths);
        ZipFileUtils.writeZipFile(zos, DATA_VIEW_FILENAME, objBaos.toByteArray());
      }
      zos.finish();
      this.save(MigrateDataTypeEnum.DATA_VIEW.getType(), StringUtils.join(ids, ","));
      return baos.toByteArray();
    } catch (IOException e) {
      LOGGER.error(e.getMessage(), e);
      throw new IllegalArgumentException(e.getMessage(), e);
    }
  }

  @Override
  public byte[] exportRemoteService(String[] ids) {
    List<RemoteServiceAddressEntity> remoteServiceAddress;
    if(ArrayUtils.isEmpty(ids)) {
      remoteServiceAddress = remoteServiceAddressService.findByAddressStatus(NormalStatusEnum.ENABLE.getStatus());
    } else {
      int count = remoteServiceAddressService.countByIds(ids);
      Validate.isTrue(count == ids.length, "指定导出的远端调用源信息与数据库存储的信息不符，请检查");
      remoteServiceAddress = remoteServiceAddressService.findDetailsByIds(ids);
    }
    if(CollectionUtils.isEmpty(remoteServiceAddress)) {
      return new byte[0];
    }
    Collection<RemoteServiceAddressEntity> collection = nebulaToolkitService.copyCollectionByWhiteList(remoteServiceAddress,
            RemoteServiceAddressEntity.class, RemoteServiceAddressEntity.class, HashSet.class, ArrayList.class, "remoteServices");
    List<RemoteServiceAddressEntity> exportRemoteServiceAddress = new ArrayList<>(collection);

    List<RemoteServiceEntity> remoteServices = new ArrayList<>();
    for(RemoteServiceAddressEntity item : exportRemoteServiceAddress){
      if(!CollectionUtils.isEmpty(item.getRemoteServices())){
        remoteServices.addAll(item.getRemoteServices());
      }
    }
    try (ByteArrayOutputStream baos = new ByteArrayOutputStream();
         ZipOutputStream zos = new ZipOutputStream(baos)) {
      try (ByteArrayOutputStream objBaos = new ByteArrayOutputStream();
           ObjectOutputStream oos = new ObjectOutputStream(objBaos)) {
        oos.writeObject(exportRemoteServiceAddress);
        ZipFileUtils.writeZipFile(zos, REMOTE_SERVICE_FILENAME, objBaos.toByteArray());
      }
      for (RemoteServiceEntity remoteService : remoteServices) {
        byte[] fileContent = venusFileService.readFileContent(remoteService.getJsonRelativePath(), remoteService.getJsonName());
        Validate.isTrue(fileContent != null && fileContent.length > 0, "未找到远端数据源【%s】的文件内容，请检查!", remoteService.getCode());
        ZipFileUtils.writeZipFile(zos, remoteService.getJsonRelativePath(), remoteService.getJsonName(), fileContent);
      }
      zos.finish();
      this.save(MigrateDataTypeEnum.REMOTE_SERVICE.getType(), StringUtils.join(ids, ","));
      return baos.toByteArray();
    } catch (IOException e) {
      LOGGER.error(e.getMessage(), e);
      throw new IllegalArgumentException(e.getMessage(), e);
    }
  }

  /**
   * 处理数据视图的分组信息，因为分组信息是树结构，所以需要递归处理
   * @param exportDataView
   * @param group
   */
  private void handleDataViewGroup(DataViewEntity exportDataView, DataViewGroupEntity group) {
    if(group == null) {
      return;
    }
    DataViewGroupEntity exportGroup = nebulaToolkitService.copyObjectByWhiteList(group, DataViewGroupEntity.class, null, null);
    DataViewGroupEntity tmpGroup = exportGroup;
    while (group.getParent() != null) {
      group = group.getParent();
      DataViewGroupEntity tmp = nebulaToolkitService.copyObjectByWhiteList(group, DataViewGroupEntity.class, null, null);
      tmpGroup.setParent(tmp);
      tmpGroup = tmp;
    }
    exportDataView.setDataViewGroup(exportGroup);
  }

  /**
   * 对指定的业务编码规则进行迁出
   * @param codeRuleIds
   */
  @Override
  public byte[] exportCodeRule(String[] codeRuleIds) {
    Set<CodeRuleEntity> codeRules;
    if(codeRuleIds == null || codeRuleIds.length == 0) {
      codeRules = codeRuleService.findAll();
    } else {
      int count = codeRuleService.countByIds(codeRuleIds);
      Validate.isTrue(count == codeRuleIds.length, "指定导出的编码规则信息与数据库存储的信息不符，请检查");
      codeRules = codeRuleService.findDetailsByIds(codeRuleIds);
    }
    if(CollectionUtils.isEmpty(codeRules)) {
      return new byte[0];
    }

    Set<ScriptEntity> scripts = Sets.newHashSet();
    for (CodeRuleEntity codeRuleItem : codeRules) {
      ScriptEntity currentScript = scriptService.findById(codeRuleItem.getScript().getId());
      Validate.notNull(currentScript , "当前编码脚本存在问题，没有关联任何脚本信息，不能进行导出");
      scripts.add(currentScript);
    }

    Collection<ScriptEntity> allScripts = this.nebulaToolkitService.copyCollectionByWhiteList(scripts, ScriptEntity.class, ScriptEntity.class, HashSet.class, ArrayList.class);
    Collection<CodeRuleEntity> allCodeRules = this.nebulaToolkitService.copyCollectionByWhiteList(codeRules, CodeRuleEntity.class, CodeRuleEntity.class, HashSet.class, ArrayList.class, "script");

    try (ByteArrayOutputStream fullZipBis = new ByteArrayOutputStream();
         ZipOutputStream zipf = new ZipOutputStream(fullZipBis);) {
      try (ByteArrayOutputStream bos = new ByteArrayOutputStream();
      ObjectOutputStream oos = new ObjectOutputStream(bos)) {
        oos.writeObject(allCodeRules);
        for (ScriptEntity scriptItem : allScripts) {
          // 将动态脚本关联的脚本附件写入压缩文件
          String relativePath = scriptItem.getFileCode();
          String fileName = scriptItem.getFileName();
          byte[] bytes = venusFileService.readFileContent(relativePath, fileName);
          ZipFileUtils.writeZipFile(zipf, relativePath, fileName, bytes);
        }
        ZipFileUtils.writeZipFile(zipf, CODE_RULE_FILENAME, bos.toByteArray());
      }
      zipf.finish();
      this.save(MigrateDataTypeEnum.CODE_RULE.getType(), StringUtils.join(codeRuleIds, ","));
      return fullZipBis.toByteArray();
    } catch(IOException e) {
      LOGGER.error(e.getMessage() , e);
      throw new IllegalArgumentException(e.getMessage());
    }
  }

  /**
   * 用于对指定的全局环境变量进行迁出
   * @param envIds
   */
  @Override
  public byte[] exportEnv(String[] envIds) {
    Set<EnvironmentVariableEntity> envs;
    if(envIds == null || envIds.length == 0) {
      envs = Sets.newHashSet(environmentVariableService.findAll());
    } else {
      long count = environmentVariableService.countByIds(envIds);
      Validate.isTrue(count == envIds.length, "指定导出的全局变量信息与数据库存储的信息不符，请检查");
      envs = environmentVariableService.findDetailsByIds(envIds);
    }
    if(CollectionUtils.isEmpty(envs)) {
      return new byte[0];
    }

    Collection<EnvironmentVariableEntity> allEnvs = this.nebulaToolkitService.copyCollectionByWhiteList(envs, EnvironmentVariableEntity.class, EnvironmentVariableEntity.class, HashSet.class, ArrayList.class);

    try (ByteArrayOutputStream fullZipBis = new ByteArrayOutputStream();
         ZipOutputStream zipf = new ZipOutputStream(fullZipBis);) {
      try (ByteArrayOutputStream bos = new ByteArrayOutputStream();
           ObjectOutputStream oos = new ObjectOutputStream(bos)) {
        oos.writeObject(allEnvs);
        ZipFileUtils.writeZipFile(zipf, ENV_FILENAME, bos.toByteArray());
      }
      zipf.finish();
      this.save(MigrateDataTypeEnum.ENVIRONMENT_VARIABLE.getType(), StringUtils.join(envIds, ","));
      return fullZipBis.toByteArray();
    } catch(IOException e) {
      LOGGER.error(e.getMessage() , e);
      throw new IllegalArgumentException(e.getMessage());
    }
  }
}
