package com.bizunited.platform.core.service.scheduler;

import com.bizunited.platform.common.service.NebulaToolkitService;
import com.bizunited.platform.core.annotations.NebulaServiceMethod;
import com.bizunited.platform.core.common.PlatformContext;
import com.bizunited.platform.core.entity.DynamicTaskSchedulerEntity;
import com.bizunited.platform.core.entity.ScriptEntity;
import com.bizunited.platform.core.repository.DynamicTaskSchedulerRepository;
import com.bizunited.platform.core.service.DynamicTaskSchedulerLogService;
import com.bizunited.platform.core.service.DynamicTaskSchedulerService;
import com.bizunited.platform.core.service.ScriptService;
import com.bizunited.platform.core.service.invoke.InvokeProxyException;
import com.bizunited.platform.core.service.invoke.handle.request.TimerRecordHandle;
import com.bizunited.platform.core.service.scheduler.handle.DynamicTaskInvokeLogHandle;
import com.bizunited.platform.user.common.service.user.UserService;
import com.bizunited.platform.user.common.vo.UserVo;
import com.google.common.collect.Sets;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.NoSuchBeanDefinitionException;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.context.ApplicationContext;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.domain.Pageable;
import org.springframework.scheduling.concurrent.ThreadPoolTaskScheduler;
import org.springframework.scheduling.support.CronTrigger;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;

import javax.transaction.Transactional;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.security.Principal;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Date;
import java.util.HashMap;
import java.util.HashSet;
import java.util.LinkedHashSet;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ScheduledFuture;
import java.util.stream.Collectors;

import static com.bizunited.platform.common.constant.Constants.PROJECT_NAME;

/**
 * 技术中台动态任务管理服务的实现
 * @author yinwenjie
 *
 */
@Service("DynamicTaskSchedulerServiceImpl")
public class DynamicTaskSchedulerServiceImpl implements DynamicTaskSchedulerService {
  @Autowired
  @Qualifier("platform_dynamicTaskScheduler")
  private ThreadPoolTaskScheduler threadPoolTaskScheduler; 
  @Autowired
  private DynamicTaskSchedulerRepository dynamicTaskSchedulerRepository; 
  @Autowired
  private ApplicationContext applicationContext; 
  @Autowired
  private NebulaToolkitService nebulaToolkitService;
  @Autowired
  private ScriptService scriptService;
  @Autowired
  private UserService userService;
  @Autowired
  private DynamicTaskSchedulerLogService dynamicTaskSchedulerLogService;
  @Autowired
  private PlatformContext platformContext;
  /**
   * 当前应用程序名
   */
  @Value("${spring.application.name:default}")
  private String applicationName;

  /**
   * 该映射关系用来记录当前的定时器任务运行情况
   */
  private Map<String, ScheduledFuture<?>> taskMapping = new ConcurrentHashMap<>();
  private static final String ERROR_MESS = "周期性任务执行表单式,填入值超过了限定长度(32)，请检查!";
  private static final String ERROR_MESS_CHARSET = "周期性表单时的中文,填入值超过了限定长度(256)，请检查!";
  
  @Override
  @Transactional
  public void interrupt(String taskCode) {
    Validate.notBlank(taskCode , "停止动态任务时，必须指定任务编号");
    DynamicTaskSchedulerEntity current =  this.dynamicTaskSchedulerRepository.findByTaskCodeAndApplicationName(taskCode, this.applicationName);
    Validate.notNull(current , "未发现任务编号对应的任务信息，请检查!!");
    
    current.setWorkingStatus(1);
    this.dynamicTaskSchedulerRepository.save(current);
  }

  @Override
  @Transactional
  public void start(String taskCode) {
    Validate.notBlank(taskCode , "启动动态任务时，必须指定任务编号");
    DynamicTaskSchedulerEntity current =  this.dynamicTaskSchedulerRepository.findByTaskCodeAndApplicationName(taskCode, this.applicationName);
    Validate.notNull(current , "未发现任务编号对应的任务信息，请检查!!");
    current.setWorkingStatus(0);
    current.setTstatus(1);
    this.dynamicTaskSchedulerRepository.save(current);
  }

  @Transactional
  public void loading() {    
    /*
     * 在2019年11月28日，动态任务执行器的功能发生了变更，不止支持groovy脚本的执行
     * 还支持固定的使用DynamicTaskService注解的Java Method的执行，所以执行步骤更新如下：
     * （注意：由于在系统启动时、脚本执行性质的任务添加时，已经完成了定时任务的配置正确性的检查，所以这里我们任务所有已经存在于数据库中的定时任务信息都是正确的）
     * 
     * 1、初始化过程第一步是查询当前系统中状态为“有效任务”，且运行状态不为“要求停止”的所有定时任务。
     * 2、依次循环这些任务，并做以下处理：
     * 2.1、判定当前任务的执行，是否已经过期，如果过期则试图终止其在threadPoolTaskScheduler中的执行
     * 注意，如果当前任务没有过期，且已经存在于threadPoolTaskScheduler中执行了，则不再需要做过多处理
     * 
     * 2.2、执行到这里，说明当前处理的这个任务还没有加入到threadPoolTaskScheduler中执行，于是需要加入进去执行
     * 如果当前任务是一个一次性任务，则优先使用executePoint进行加入
     * 如果当前任务是一个抽骑行任务，则使用cornExpression进行加入
     * 注意，这个任务到底是一个groovy脚本的执行方式还是一个使用DynamicTaskService注解的Java Method的执行方式
     * 并不在这里进行区别处理，区别处理的详情可以参见DynamicTaskInvokeHandle类
     * 
     * 3、经过以上对那些正在运行的任务的 过期处理；对新的要求运行的任务的启动处理后
     * 需要修改数据库中对应的任务执行状态。
     * 
     * 4、接下来处理那些被用户或者程序设置为“无效”或“要求停止”的运行任务，并试图终止其在threadPoolTaskScheduler中的执行
     * 注意：这里获取到的taskCode可能会和之前已经处理过的taskCode重复，这是因为在以上1、2、3步中，这些TaskCode被设定成了“无效”
     * 但是无所谓拉，因为不会造成幂等性问题
     * 
     * 2019年12月30日，由于多个微服务应用会共享数据库，所以还需要对当前运行任务的应用名进行判定。
     * 不属于当前应用名的任务，是不允许在当前进程中进行加载的
     * */
    
    // 1、=====
    // 工作状态不能包括“要求停止”
    int[] workingStatuses = new int[] {0 , 2};
    Set<DynamicTaskSchedulerEntity> tasks = dynamicTaskSchedulerRepository.findByTstatusAndWorkingStatusAndApplicationName(1, workingStatuses , this.applicationName);
    // 2、======
    List<String> runningTaskCodes = new ArrayList<>(100);
    // 这个集合记录哪些正在运行的任务存在人为的或者客观的错误，需要从运行状态剔除的
    List<String> invalidTaskCodes = new ArrayList<>(100);
    if(!CollectionUtils.isEmpty(tasks)) {
      Date currentTime = new Date();
      for (DynamicTaskSchedulerEntity task : tasks) {
        String taskCode = task.getTaskCode();
        Date validityTime = task.getValidityTime();
        // 如果条件成立，说明有效期已过，不再进行处理，且如果有任务正在运行，也必须取消
        if(validityTime != null && validityTime.getTime() < currentTime.getTime()) {
          ScheduledFuture<?> scheduledFuture = taskMapping.get(taskCode);
          if(scheduledFuture != null) {
            // true的意思是如果当前正在运行，则发送interrupt信号
            scheduledFuture.cancel(true);
            taskMapping.remove(taskCode);
          }
          invalidTaskCodes.add(taskCode);
          continue;
        }
        // 以下视情况进行添加
        // 如果条件成立，则无需进行添加，因为这个bean已经存在了，且当前数据库可能的不同步情况要进行修正
        if(taskMapping.get(taskCode) != null) {
          runningTaskCodes.add(taskCode);
          continue;
        }
        // 开始初始化该任务
        DynamicTaskSchedulerEntity currentTask = this.nebulaToolkitService.copyObjectByWhiteList(task, DynamicTaskSchedulerEntity.class, LinkedHashSet.class, LinkedList.class, new String[] {"script"});
        DynamicTask dynamicTask = new DynamicTask(currentTask , this.applicationContext);
        
        // 3、======这里注意，如果当前task已经在运行内存中了，就不需要再进行加载了
        // 如果条件成立，表示是一个一次性任务
        if(task.getTaskType() == 1) {
          Date executePoint = task.getExecutePoint();
          String cornExpression = task.getExecuteExpression();
          Validate.isTrue(!(executePoint == null && StringUtils.isBlank(cornExpression)) , "指定任务[%s]是一个一次性任务，但是却没有指定任务执行时间，请检查" , task.getTaskCode());
          ScheduledFuture<?> scheduledFuture = null;
          if(executePoint != null) {
            scheduledFuture = this.threadPoolTaskScheduler.schedule(dynamicTask, executePoint);
          } else {
            CronTrigger cronTrigger = new CronTrigger(cornExpression);
            scheduledFuture = this.threadPoolTaskScheduler.schedule(dynamicTask, cronTrigger);
          }
          this.taskMapping.put(taskCode, scheduledFuture);
          runningTaskCodes.add(taskCode);
        } 
        // 如果条件成立，标识是一个周期性任务
        else if(task.getTaskType() == 2) {
          String cornExpression = task.getExecuteExpression();
          Validate.isTrue(StringUtils.isNotBlank(cornExpression) , "指定动态任务[%s]是一个周期性任务，但是却没有指定有效的corn表达式，请检查" , task.getTaskCode());
          CronTrigger cronTrigger = new CronTrigger(cornExpression);
          ScheduledFuture<?> scheduledFuture = this.threadPoolTaskScheduler.schedule(dynamicTask, cronTrigger);
          this.taskMapping.put(taskCode, scheduledFuture);
          runningTaskCodes.add(taskCode);
        } 
        // 其它情况下，直接抛出异常
        else {
          throw new IllegalArgumentException("错误的动态任务类型[" + task.getTaskCode() + "]，请检查");
        }
      }
    }
    
    // 4、========进行数据库中的状态更新
    if(!invalidTaskCodes.isEmpty()) {
      this.dynamicTaskSchedulerRepository.updateInvalidByTaskCode(invalidTaskCodes.toArray(new String[] {}) , this.applicationName);
    }
    if(!runningTaskCodes.isEmpty()) {
      this.dynamicTaskSchedulerRepository.updateRunningByTaskCode(runningTaskCodes.toArray(new String[] {}) , this.applicationName);
    }
    
    // 5、========扫描在这个周期可能已经被其它进程操作，在数据库层中要求被停止的进程，这里将进行停止操作
    Set<DynamicTaskSchedulerEntity> statusTasks = this.dynamicTaskSchedulerRepository.findByTstatusAndApplicationName(0 , this.applicationName);
    if(CollectionUtils.isEmpty(statusTasks)) {
      statusTasks = new HashSet<>();
    }
    Set<DynamicTaskSchedulerEntity> invalidTasks = this.dynamicTaskSchedulerRepository.findByWorkingStatusAndApplicationName(new int[] {1}, this.applicationName);
    if(!CollectionUtils.isEmpty(invalidTasks)) {
      statusTasks.addAll(invalidTasks);
    }
    // 开始停止
    for (DynamicTaskSchedulerEntity statusTask : statusTasks) {
      String invalidTaskCode = statusTask.getTaskCode();
      ScheduledFuture<?> scheduledFuture = taskMapping.remove(invalidTaskCode);
      // 停止内存中的调度器
      if(scheduledFuture != null) {
        scheduledFuture.cancel(true);
      }
      // “停止”数据库中的信息
      DynamicTaskSchedulerEntity cancelTask = this.findByTaskCode(invalidTaskCode);
      if(cancelTask != null) {
        this.dynamicTaskSchedulerRepository.updateInvalidByTaskCode(new String[] {invalidTaskCode}, this.applicationName);
      }
    }
  }
  
  public void unloading() {
    /*
     * 直接通过taskMapping中的任务信息进行停止即可
     * */ 
    if(this.taskMapping.isEmpty()) {
      return;
    }
    Collection<ScheduledFuture<?>> tasks = this.taskMapping.values();
    for (ScheduledFuture<?> task : tasks) {
      task.cancel(true);
    }
  }

  @Override
  @NebulaServiceMethod(name = "DynamicTaskSchedulerService.invoke" 
  , desc = "该方法对指定的taskCode进行一次调用，无论这个动态任务是基于groovy的还是基于java的都在这个方法中被执行"
  , requestHandleTypes = {TimerRecordHandle.class} , responseHandleTypes = {DynamicTaskInvokeLogHandle.class})
  public void invoke(DynamicTaskSchedulerEntity currentTask)  throws InvokeProxyException {
    /*
     * 较大几次调整分别是：
     * a、2019年11月28日，对业务构建平台的定时器功能进行了需求调整
     * b、2020年4月，对定时器和服务源的依赖关系进行了调整
     * 
     * 处理过程调整如下：
     * 1、首先判定任务信息本身的合规性
     * 2、如果当前任务是一个Groovy脚本性质的执行任务，则在加载个参数后，使用scriptService.invoke进行执行
     * 如果当前任务是一个托管到Spring的使用DynamicTaskService注解定义的指定方法，
     * 则采用spring上下文加载后，通过反射调用进行执行（请注意执行时方法本身的入参说明）
     */
    // 1、=======
    Validate.notNull(currentTask , "在进行动态任务执行时，未发现指定的动态调用任务!");
    Validate.notBlank(currentTask.getTaskCode() , "在进行动态任务执行时，未发现指定的动态调用任务的code信息!");
    Map<String, Object> parmas = new HashMap<>();
    parmas.put("currentTask", currentTask);
    
    // 2、======
    // 如果条件成立，说明是执行一个groovy脚本性质的定时任务
    if(currentTask.getInvokeType() == 1) {
      this.invokeGroovyScript(currentTask);
    } 
    // 如果条件成立，说明是执行一个基于注解的java method定时任务
    else if(currentTask.getInvokeType() == 2) {
      this.invokeAnnMethod(currentTask);
    } else {
      // 其它情况抛出异常
      throw new NoSuchBeanDefinitionException(DynamicTaskSchedulerEntity.class);
    }
  }

  /**
   * 该私有方法用于执行基于DynamicTaskService注解的java method方法
   */
  private void invokeAnnMethod(DynamicTaskSchedulerEntity currentTask) throws InvokeProxyException{
    /*
     * 处理过程为：
     * 1、首先拿到currentTask之中的基本信息，并判定有效性
     * 2、然后根据其中的得到的bean的名字、方法的名字，开始进行调用
     * 注意其中的传参要求：如果其中要求了InvokeProxyContext参数，还要拼凑参数进行传入
     * */
    // 1、=======
    String taskCode = currentTask.getTaskCode();
    String invokeBeanName = currentTask.getInvokeBeanName();
    Validate.notBlank(invokeBeanName , "执行动态调用任务时，发现基于DynamicTaskService注解的动态执行任务%s，其invokeBeanName信息非法" , taskCode);
    String invokeMethod = currentTask.getInvokeMethod();
    Validate.notBlank(invokeMethod , "执行动态调用任务时，发现基于DynamicTaskService注解的动态执行任务%s，其invokeMethod信息非法" , taskCode);
    Object currentBean = null;
    try {
      currentBean = applicationContext.getBean(invokeBeanName);
    } catch(NoSuchBeanDefinitionException e) {
      throw new IllegalArgumentException(String.format("执行动态调用任务时，未发现名为%s的spring bean，请检查" , invokeBeanName));
    }
    
    // 2、=======
    Class<?> beanClass = applicationContext.getType(invokeBeanName);
    Method beanClassMethod = null;
    // 开始进行调用
    try {
      beanClassMethod = beanClass.getMethod(invokeMethod, new Class[] {});
      beanClassMethod.invoke(currentBean, new Object[] {});
    } catch(NoSuchMethodException e) {
      throw new IllegalArgumentException(String.format("执行动态调用任务时，未发现名为%s的spring bean中指定的方法%s，请检查" , invokeBeanName , invokeMethod));
    } catch(InvocationTargetException | IllegalAccessException e) {
      throw new IllegalArgumentException("执行动态调用任务时,方法执行错误，请检查" , e);
    }
  }
  
  /**
   * 该私有方法用于执行Groovy脚本性质的定时器任务
   */
  private void invokeGroovyScript(DynamicTaskSchedulerEntity currentTask) throws InvokeProxyException {
    // 脚本信息
    String scriptName = currentTask.getScriptName();
    ScriptEntity scriptEntity = this.scriptService.findByName(scriptName);
    Validate.notNull(scriptEntity , "在进行动态任务执行时，未发现指定的groovy脚本信息，请检查%s!" , currentTask.getId());
    String scriptId = scriptEntity.getId();
    // 内容对象信息
    Map<String, Object> params = new HashMap<>();
    params.put("userService", this.userService);
    params.put("ctx", this.applicationContext);
    params.put("task", currentTask);
    
    // 调用链上下文对象引入，并完成执行
    this.scriptService.invoke(new String[] {scriptId}, params);
  }

  @Override
  @Transactional
  public DynamicTaskSchedulerEntity create(DynamicTaskSchedulerEntity task , String scriptContent , Principal principal) {
    // 1、=========首先进行动态任务基本信息的添加
    Validate.notNull(task , "进行当前操作时，动态任务对象必须传入!!");
    String userAccount = principal.getName();
    UserVo user = userService.findByAccount(userAccount);
    Validate.notNull(user, "未找到当前用户信息");
    Date currentTime = new Date();
    // 设置自填充值
    task.setCreateTime(currentTime);
    task.setCreateAccount(userAccount);
    task.setModifyTime(currentTime);
    task.setModifyAccount(userAccount);
    // 设定为有效任务
    task.setTstatus(1);
    // 设定当前状态为“要求启动”
    task.setWorkingStatus(0);
    // 目前只能通过该接口创建groovy脚本性质的定时任务
    task.setInvokeType(1);
    // 当前的应用程序名
    task.setApplicationName(this.applicationName);
    // 进行对象信息验证
    this.createValidation(task);
    
    // 2、=======然后进行动态任务脚本内容的创建
    ScriptEntity script = new ScriptEntity();
    String scriptName = "dynamicTask_" + task.getTaskCode();
    script.setLanguage("groovy");
    script.setName(scriptName);
    script = this.scriptService.create(script, scriptContent);
    task.setScriptName(scriptName);
    task.setProjectName(platformContext.getAppName());
    // 进行更新
    this.dynamicTaskSchedulerRepository.save(task);
    
    return task;
  }
  
  /**
   * 在创建一个新的DynamicTaskSchedulerEntity模型对象之前，检查对象各属性的正确性，其主键属性必须没有值
   */
  private void createValidation(DynamicTaskSchedulerEntity dynamicTaskSchedulerEntity) { 
    // 判定那些不能为null的输入值：条件为 caninsert = true，且nullable = false
    Validate.isTrue(StringUtils.isBlank(dynamicTaskSchedulerEntity.getId()), "添加信息时，当期信息的数据编号（主键）不能有值！");
    dynamicTaskSchedulerEntity.setId(null);
    Validate.notBlank(dynamicTaskSchedulerEntity.getTaskCode(), "任务唯一编号（只能由英文、数字、下杠构成）不能为空！");
    Validate.notBlank(dynamicTaskSchedulerEntity.getTaskDesc(), "任务中文说明不能为空！");
    Validate.notNull(dynamicTaskSchedulerEntity.getTaskType(), "任务类型不能为空！");
    Validate.notNull(dynamicTaskSchedulerEntity.getCreateTime(), "创建时间不能为空！");
    Validate.notNull(dynamicTaskSchedulerEntity.getModifyTime(), "最后修改时间不能为空！");
    // 验证长度，被验证的这些字段符合特征: 字段类型为String，且不为PK （注意连续空字符串的情况） 
    Validate.isTrue(dynamicTaskSchedulerEntity.getTaskCode() == null || dynamicTaskSchedulerEntity.getTaskCode().length() < 128 , "任务唯一编号（只能由英文、数字、下杠构成）,填入值超过了限定长度(128)，请检查!");
    Validate.isTrue(dynamicTaskSchedulerEntity.getTaskDesc() == null || dynamicTaskSchedulerEntity.getTaskDesc().length() < 256 , "任务中文说明,填入值超过了限定长度(256)，请检查!");
    // 如果当前是一次性任务
    Date currentTime = new Date();
    if(dynamicTaskSchedulerEntity.getTaskType() == 1) {
      Validate.notNull(dynamicTaskSchedulerEntity.getExecutePoint(), "一次性任务的开始时间必须填写");
      Validate.isTrue(dynamicTaskSchedulerEntity.getExecutePoint().getTime() > currentTime.getTime() , "一次性任务的开始时间，必须设定在当前时间点以后");
    }
    // 如果当前是周期性任务
    else if(dynamicTaskSchedulerEntity.getTaskType() == 2) {
      Validate.notBlank(dynamicTaskSchedulerEntity.getExecuteExpression() , "周期性任务必须给定有效的corn表达式!!");
      Validate.notBlank(dynamicTaskSchedulerEntity.getExpressionDesc() , "周期性任务必须传入corn表达式的中文描述!!");
      Validate.isTrue(dynamicTaskSchedulerEntity.getExecuteExpression() == null || dynamicTaskSchedulerEntity.getExecuteExpression().length() < 32 , ERROR_MESS);
      Validate.isTrue(dynamicTaskSchedulerEntity.getExpressionDesc() == null || dynamicTaskSchedulerEntity.getExpressionDesc().length() < 256 , ERROR_MESS_CHARSET);
    }
    // 其它情况报错
    else {
      throw new IllegalArgumentException("动态任务只能是一次性任务或者周期性任务，请检查!!");
    }
    DynamicTaskSchedulerEntity currentDynamicTaskSchedulerEntity = this.dynamicTaskSchedulerRepository.findByTaskCodeAndApplicationName(dynamicTaskSchedulerEntity.getTaskCode() , this.applicationName);
    Validate.isTrue(currentDynamicTaskSchedulerEntity == null, "任务唯一编号（只能由英文、数字、下杠构成）已存在,请检查");
  }

  @Override
  @Transactional
  public DynamicTaskSchedulerEntity update(DynamicTaskSchedulerEntity task, String scriptContent , Principal principal) {
    Validate.notNull(task , "进行当前操作时，动态任务对象必须传入!!");
    String userAccount = principal.getName();
    UserVo uservo = userService.findByAccount(userAccount);
    Validate.notNull(uservo, "未找到当前用户信息");
    Date currentTime = new Date();
    // 开始检查
    this.updateValidation(task);
    
    // 1、===================更新基本信息
    String currentId = task.getId();
    Optional<DynamicTaskSchedulerEntity> op = this.dynamicTaskSchedulerRepository.findById(currentId);
    DynamicTaskSchedulerEntity currentTask = op.orElse(null);
    Validate.notNull(currentTask ,"未发现指定的原始模型对象信");

    // 如果当前任务的状态不是“要求停止”，则不允许进行信息修改，
    Validate.isTrue(currentTask.getTstatus() == 0 , "只有在调度任务被禁用/无效的情况下（tstatus == 0）,才能进行修改!!!");
    Validate.isTrue(currentTask.getWorkingStatus() == 1 , "只有首先停止任务运行（workingStatus == 1），才能进行修改!!");
    
    // 开始重新赋值——一般属性
    currentTask.setTaskDesc(task.getTaskDesc());
    currentTask.setExecutePoint(task.getExecutePoint());
    currentTask.setExecuteExpression(task.getExecuteExpression());
    currentTask.setExpressionDesc(task.getExpressionDesc());
    currentTask.setValidityTime(task.getValidityTime());
    currentTask.setModifyTime(currentTime);
    currentTask.setModifyAccount(userAccount);
    
    // 2、===================更新脚本信息
    String scriptName = currentTask.getScriptName();
    Validate.notBlank(scriptName , "进行更新时，脚本内容必须传入!!");
    ScriptEntity script = this.scriptService.findByName(scriptName);
    Validate.notNull(script , "未找到指定的groovy脚本信息，请检查!!");
    script = this.scriptService.update(script, scriptContent);
    return currentTask;
  }
  /**
   * 在更新一个已有的DynamicTaskSchedulerEntity模型对象之前，该私有方法检查对象各属性的正确性，其id属性必须有值
   */
  private void updateValidation(DynamicTaskSchedulerEntity dynamicTaskSchedulerEntity) { 
    Validate.isTrue(!StringUtils.isBlank(dynamicTaskSchedulerEntity.getId()), "修改信息时，当期信息的数据编号（主键）必须有值！");
    // 基础信息判断，基本属性，需要满足not null
    Validate.notBlank(dynamicTaskSchedulerEntity.getTaskCode(), "任务唯一编号（只能由英文、数字、下杠构成）不能为空！");
    Validate.notBlank(dynamicTaskSchedulerEntity.getTaskDesc(), "任务中文说明不能为空！");
    Validate.notNull(dynamicTaskSchedulerEntity.getTaskType(), "任务类型不能为空！");
    // 如果当前是一次性任务
    Date currentTime = new Date();
    if(dynamicTaskSchedulerEntity.getTaskType() == 1) {
      Validate.notNull(dynamicTaskSchedulerEntity.getExecutePoint(), "一次性任务的开始时间必须填写");
      Validate.isTrue(dynamicTaskSchedulerEntity.getExecutePoint().getTime() > currentTime.getTime() , "一次性任务的开始时间，必须设定在当前时间点以后");
    }
    // 如果当前是周期性任务
    else if(dynamicTaskSchedulerEntity.getTaskType() == 2) {
      Validate.notBlank(dynamicTaskSchedulerEntity.getExecuteExpression() , "周期性任务必须给定有效的corn表达式!!");
      Validate.notBlank(dynamicTaskSchedulerEntity.getExpressionDesc() , "周期性任务必须传入corn表达式的中文描述!!");
      Validate.isTrue(dynamicTaskSchedulerEntity.getExecuteExpression() == null || dynamicTaskSchedulerEntity.getExecuteExpression().length() < 32 , ERROR_MESS);
      Validate.isTrue(dynamicTaskSchedulerEntity.getExpressionDesc() == null || dynamicTaskSchedulerEntity.getExpressionDesc().length() < 256 , ERROR_MESS_CHARSET);
    }
    // 其它情况报错
    else {
      throw new IllegalArgumentException("动态任务只能是一次性任务或者周期性任务，请检查!!");
    }
    
    // 重复性判断，基本属性，需要满足unique = true
    DynamicTaskSchedulerEntity currentForTaskCode = this.dynamicTaskSchedulerRepository.findByTaskCodeAndApplicationName(dynamicTaskSchedulerEntity.getTaskCode() , this.applicationName);
    Validate.isTrue(currentForTaskCode == null || StringUtils.equals(currentForTaskCode.getId() , dynamicTaskSchedulerEntity.getId()) , "任务唯一编号（只能由英文、数字、下杠构成）已存在,请检查"); 
    // 验证长度，被验证的这些字段符合特征: 字段类型为String，且不为PK，且canupdate = true
    Validate.isTrue(dynamicTaskSchedulerEntity.getTaskCode() == null || dynamicTaskSchedulerEntity.getTaskCode().length() < 128 , "任务唯一编号（只能由英文、数字、下杠构成）,填入值超过了限定长度(128)，请检查!");
    Validate.isTrue(dynamicTaskSchedulerEntity.getTaskDesc() == null || dynamicTaskSchedulerEntity.getTaskDesc().length() < 256 , "任务中文说明,填入值超过了限定长度(256)，请检查!");
    Validate.isTrue(dynamicTaskSchedulerEntity.getExecuteExpression() == null || dynamicTaskSchedulerEntity.getExecuteExpression().length() < 32 , ERROR_MESS);
    Validate.isTrue(dynamicTaskSchedulerEntity.getExpressionDesc() == null || dynamicTaskSchedulerEntity.getExpressionDesc().length() < 256 , ERROR_MESS_CHARSET);
  } 
  @Override
  @Transactional
  public void updateInvalidByTaskCode(String[] taskCodes) {
    Validate.notNull(taskCodes , "在进行强制停止时，至少传入一个动态定时任务的编号code!!");
    Validate.isTrue(taskCodes.length > 0 , "在进行强制停止时，至少传入一个动态定时任务的编号code!!");
    this.dynamicTaskSchedulerRepository.updateInvalidByTaskCode(taskCodes ,this.applicationName);
  }

  @Override
  @Transactional
  public void updateExsitReflexTasks(List<DynamicTaskSchedulerEntity> currentScanTasks) {
    /*
     * 在判定了入参有效性后，处理过程为：
     * 1、扫描了所有基于DynamicTaskService注解的java method形式的方法并验证后
     * 就可以开始对比和当前数据库中存在的任务数据进行对比，并分离出哪些是新增的定时器任务，哪些是需要删除（真删除）的定时器任务，哪些是需要更新的定时器任务
     * 2、开始对本次扫描发现的基于DynamicTaskService注解的java method形式的方法进行任务处理
     * 包括进行添加、删除或者修改操作
     * */
    Validate.isTrue(!threadPoolTaskScheduler.getScheduledExecutor().isShutdown() , "当前定时任务执行器已经开始运行，不能通过updateExsitReflexTasks方法进行任务更新!");
    
    // 1、======
    Date currentTime = new Date();
    Set<DynamicTaskSchedulerEntity> exsitDynamicTasks = this.dynamicTaskSchedulerRepository.findByInvokeTypeAndApplicationName(2, this.applicationName);
    if(exsitDynamicTasks == null) {
      exsitDynamicTasks = Sets.newHashSet();
    }
    Set<DynamicTaskSchedulerEntity> deleteCollections = Sets.newHashSet();
    Set<DynamicTaskSchedulerEntity> updateCollections = Sets.newHashSet();
    Set<DynamicTaskSchedulerEntity> createCollections = Sets.newHashSet();
    this.nebulaToolkitService.collectionDiscrepancy(currentScanTasks, exsitDynamicTasks, DynamicTaskSchedulerEntity::getTaskCode, deleteCollections, updateCollections, createCollections);
    
    // 2、======
    // 首先处理新增的集合信息
    if(!createCollections.isEmpty()) {
      for (DynamicTaskSchedulerEntity createItem : createCollections) {
        // 默认就是admin
        createItem.setCreateAccount("admin");
        createItem.setCreateTime(currentTime);
        createItem.setModifyAccount("admin");
        createItem.setModifyTime(currentTime);
        
        // 任务状态：1有效
        createItem.setTstatus(1);
        // 任务运行状态：0要求运行
        createItem.setWorkingStatus(0);
        this.dynamicTaskSchedulerRepository.save(createItem);
      }
    }
    // 然后进行删除操作（真删除）
    if(!deleteCollections.isEmpty()) {
      for (DynamicTaskSchedulerEntity deleteItem : deleteCollections) {
        String dynamicTaskSchedulerId = deleteItem.getId();
        this.dynamicTaskSchedulerLogService.deleteByDynamicTaskSchedulerId(dynamicTaskSchedulerId);
        this.dynamicTaskSchedulerRepository.flush();
        this.dynamicTaskSchedulerRepository.delete(deleteItem);
      }
    }
    // 最后进行修改操作（修改时，那些无效状态的，人工停止状态的任务，不能改变）
    if(!updateCollections.isEmpty()) {
      // 基于当前入参的currentScanTasks，转成一个Map集合，以便在update操作时使用
      Map<String, DynamicTaskSchedulerEntity> targetTasksMapping = currentScanTasks.stream().collect(Collectors.toMap(DynamicTaskSchedulerEntity::getTaskCode, item->item ));
      for (DynamicTaskSchedulerEntity updateItem : updateCollections) {        
        String taskCode = updateItem.getTaskCode();
        DynamicTaskSchedulerEntity targetItem = targetTasksMapping.get(taskCode);
        updateItem.setExecuteExpression(targetItem.getExecuteExpression());
        updateItem.setExecutePoint(targetItem.getExecutePoint());
        updateItem.setExpressionDesc(targetItem.getExpressionDesc());
        updateItem.setInvokeBeanName(targetItem.getInvokeBeanName());
        updateItem.setInvokeMethod(targetItem.getInvokeMethod());
        updateItem.setProxyContextParam(targetItem.getProxyContextParam());
        updateItem.setTaskDesc(targetItem.getTaskDesc());
        updateItem.setModifyAccount("admin");
        updateItem.setModifyTime(currentTime);
        updateItem.setValidityTime(targetItem.getValidityTime());
        this.dynamicTaskSchedulerRepository.save(updateItem);
      }
    }
  }

  @Override
  public Set<DynamicTaskSchedulerEntity> findAll() {
    String projectName = platformContext.getAppName();
    if(StringUtils.isNotBlank(projectName)){
      return this.dynamicTaskSchedulerRepository.findByProjectName(projectName);
    }
    return this.dynamicTaskSchedulerRepository.findByBlankProjectName();
  }
  @Override
  public DynamicTaskSchedulerEntity findByTaskCode(String taskCode) {
    if(StringUtils.isBlank(taskCode)) {
      return null;
    }
    DynamicTaskSchedulerEntity task = this.dynamicTaskSchedulerRepository.findByTaskCodeAndApplicationName(taskCode , this.applicationName);
    return task;
  }

  @Override
  public Page<DynamicTaskSchedulerEntity> findByConditions(Pageable pageable, Integer taskType, String taskCode, Integer tstatus, Integer workingStatus, String taskDesc , Integer invokeType) {
    Map<String,Object> conditions = new HashMap<>();
    if(taskType != null){
      conditions.put("taskType",taskType);
    }
    if(StringUtils.isNotEmpty(taskCode)){
      conditions.put("taskCode",taskCode);
    }
    if(tstatus != null){
      conditions.put("tstatus",tstatus);
    }
    if(workingStatus != null){
      conditions.put("workingStatus",workingStatus);
    }
    if(StringUtils.isNotEmpty(taskDesc)){
      conditions.put("taskDesc",taskDesc);
    }
    if(invokeType != null) {
      conditions.put("invokeType",invokeType);
    }
    conditions.put(PROJECT_NAME,platformContext.getAppName());
    if(pageable ==null){
      pageable = PageRequest.of(0,50);
    }
    
    return this.dynamicTaskSchedulerRepository.queryPage(pageable, conditions);
  }

  /**
   * 删除定时任务接口
   *
   * @param taskcode
   */
  @Override
  @Transactional
  public void deleteByTaskcode(String taskcode) {
    DynamicTaskSchedulerEntity task = this.dynamicTaskSchedulerRepository.findByTaskCode(taskcode);
    Validate.notNull(task,"没有指定的定时任务");
    Validate.isTrue(task.getInvokeType() == 1 , "删除类型错误，只能删除groovy类型的任务");
    Validate.isTrue(task.getTstatus() == 0 , "任务状态错误，只有已停止的任务才能删除");
    //先删除该定时任务的日志信息
    this.dynamicTaskSchedulerLogService.deleteByDynamicTaskSchedulerId(task.getId());
    this.dynamicTaskSchedulerRepository.delete(task);
  }
}